#include <gtk/gtk.h>
#include <glade/glade.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <unistd.h>
#include <sys/wait.h>

#define GLADE_APP "/usr/share/gtrayicon/gtrayicon.glade"
#define DEFAULT_ACTIVATE_ICON "/usr/share/gtrayicon/activate.svg"
#define DEFAULT_DEACTIVATE_ICON "/usr/share/gtrayicon/deactivate.svg"
#define ON TRUE
#define OFF FALSE

GladeXML *xml;
GtkStatusIcon *icon;

/* command line options */
gchar *tooltip = NULL;
gchar *activate_cmd = NULL;
gchar *deactivate_cmd = NULL;
gchar *activate_icon = NULL;
gchar *deactivate_icon = NULL;
gchar *menu_file = NULL;
gboolean state = OFF;

static GOptionEntry entries[] = {
	{ "tooltip", 0, 0, G_OPTION_ARG_STRING, &tooltip, "string to appear as tooltip", "string" },
	{ "activate", 0, 0, G_OPTION_ARG_STRING, &activate_cmd, "activate command", "cmd" },
	{ "deactivate", 0, 0, G_OPTION_ARG_STRING, &deactivate_cmd, "deactivate command", "cmd" },
	{ "activate-icon", 0, 0, G_OPTION_ARG_STRING, &activate_icon, "icon to show for 'activate' action", "icon-path" },
	{ "deactivate-icon", 0, 0, G_OPTION_ARG_STRING, &deactivate_icon, "icon to show for 'deactivate'", "icon-path" },
	{ "menu-file", 0, 0, G_OPTION_ARG_STRING, &menu_file, "custom menu to display for right button", "menu-path" },
	{ "activated", 0, 0, G_OPTION_ARG_NONE, &state, "start in 'activated' state (default: 'deactivated' state)", NULL },
	{ NULL }
};

/* functions to run commands */
void run_cmd(const gchar *cmd) {	/* double fork to avoid zombies */
	if (fork() == 0) { /* child */
		if (fork() == 0) { /* sub-child */
			execl("/bin/sh", "sh", "-c", cmd, NULL);	/* to allow shell commands */
			exit(0);
		} else
			exit(0); /* kill the sub-parent */
	} else {
		wait(NULL); /* wait for child to exit */
	}
}

void run_cmd_signal_wrapper(GtkWidget *w, const gchar *cmd) {
	run_cmd(cmd);
}

/* interface callbacks */
void main_quit() {
	gtk_main_quit();
	exit(0);
}
	
void menu_popup(GtkStatusIcon *status_icon, gint button, gint activate_time, gpointer data) {
	GtkWidget *menu;

	menu = glade_xml_get_widget(xml, "popup_menu");
	gtk_widget_show_all(menu);
	gtk_menu_popup(GTK_MENU(menu), NULL, NULL, NULL, status_icon, button, activate_time);
}

void on_quit_item_activate(GtkWidget *menu, gpointer data) {
	main_quit();
}

void on_about_item_activate(GtkWidget *widget, gpointer data) {
	GtkWidget *about;
	
	about = glade_xml_get_widget(xml, "aboutdialog1");
	gtk_widget_show(about);
}

void change_state(gint mode) {
	if (mode == ON) {
		run_cmd(activate_cmd);
		gtk_status_icon_set_from_file(icon, deactivate_icon);
	} else {
		run_cmd(deactivate_cmd);
		gtk_status_icon_set_from_file(icon, activate_icon);
	}
}

void icon_activate(GtkStatusIcon *status_icon, gpointer data) {
	state = !state; /* next state is !current-state */
	change_state(state);
}

/* menu parser (uses glib key file parser) */
gint populate_popup_menu_from_file(const gchar *file, GError **error) {
	GKeyFile *keyfile;
	gchar **groups, **keys, **values;
	gsize groups_size, keys_size, values_size;
	gint group_pos, key_pos, value_pos, menu_pos, main_menu_pos;
	GtkWidget *image, *item, *parent, *popup_menu;
	gboolean main_section = FALSE;
	
	keyfile = g_key_file_new();
	if (g_key_file_load_from_file(keyfile, menu_file, G_KEY_FILE_NONE, error) == FALSE) {
		return -1;
	}

	g_key_file_set_list_separator(keyfile, ',');
	groups = g_key_file_get_groups(keyfile, &groups_size);

	if (groups_size == 0) {
		return -1;
	}
	
	popup_menu = glade_xml_get_widget(xml, "popup_menu");
	gtk_menu_shell_insert((GtkMenuShell*)popup_menu, gtk_separator_menu_item_new(), 0);
	main_menu_pos = 0;
	
	for(group_pos = 0; group_pos < groups_size; group_pos++) {
	
		if (g_strcmp0(groups[group_pos], "main") == 0) { /* main group */
			main_section = TRUE;
			parent = popup_menu;
		} else { /* normal submenu */
			main_section = FALSE;
			item = gtk_menu_item_new_with_label(groups[group_pos]);
			parent = gtk_menu_new();
			gtk_menu_item_set_submenu((GtkMenuItem*)item, parent);
			gtk_menu_shell_insert((GtkMenuShell*)popup_menu, item, main_menu_pos);
			main_menu_pos++;
			menu_pos = 0;
		}

		keys = g_key_file_get_keys(keyfile, groups[group_pos], &keys_size, NULL);

		for(key_pos = 0; key_pos < keys_size; key_pos++) {
			values = g_key_file_get_string_list(keyfile, groups[group_pos], keys[key_pos], &values_size, NULL);
			
			if (values_size == 1) { /* no icon */
				item = gtk_menu_item_new_with_label(keys[key_pos]);
			} else if (values_size == 2) { /* command and icon */
				item = gtk_image_menu_item_new_with_label(keys[key_pos]);
				image = gtk_image_new_from_file(values[1]);
				gtk_image_menu_item_set_image((GtkImageMenuItem*)item, image);
			}
			
			g_signal_connect(G_OBJECT(item), "activate", G_CALLBACK(run_cmd_signal_wrapper), g_strdup(values[0]));
			
			if (main_section)
				gtk_menu_shell_insert((GtkMenuShell*)parent, item, main_menu_pos++);
			else
				gtk_menu_shell_insert((GtkMenuShell*)parent, item, menu_pos++);
		}
	}
	
	g_key_file_free(keyfile);
	return 0;
}

int main(int argc, char *argv[]) {
	GError *error = NULL;
	GOptionContext *context;
	
	/* parse command line options */
	context = g_option_context_new ("- Generic tray icon");
	g_option_context_add_main_entries (context, entries, "");
	g_option_context_add_group (context, gtk_get_option_group (TRUE));
	
	if (!g_option_context_parse (context, &argc, &argv, &error)) {
		fprintf(stderr, "gtrayicon: option parsing failed > %s\n", error->message);
		exit(1);
	}

	if (!activate_cmd || !deactivate_cmd) {
		perror("gtrayicon: you must provide activate/deactivate commands. Use '--help' parameter for more information.\n");
		exit(1);
	}
	
    /* load the interface */
	gtk_init(&argc, &argv);
    xml = glade_xml_new(GLADE_APP, NULL, NULL);

    if (!xml) {
    	perror("gtrayicon: error loading glade file.");
    	exit(1);
    }
	
	/* connect the signals in the interface */
    glade_xml_signal_autoconnect(xml);
    
    /* initialize status icon */
	icon = gtk_status_icon_new();
    gtk_status_icon_set_visible(icon, TRUE);
    
	if (tooltip)
	    gtk_status_icon_set_tooltip(icon, tooltip);

	if (!activate_icon)
		activate_icon = DEFAULT_ACTIVATE_ICON;
		
	if (!deactivate_icon)
		deactivate_icon = DEFAULT_DEACTIVATE_ICON;
	
	/* check for user supplied menu and parse it */	
	if (menu_file) {
		if (populate_popup_menu_from_file(menu_file, &error) != 0)
			fprintf(stderr, "gtrayicon: warning loading menu file > %s\n", error->message);
	}
	
	/* check initial state */
	if (state == ON)
		change_state(ON); /* execute activate cmd */
	else
		gtk_status_icon_set_from_file(icon, activate_icon); /* don't run anything, start in deactivated state */

	/* trap some signals */
	signal(SIGINT, main_quit);
	signal(SIGTERM, main_quit);
	signal(SIGQUIT, main_quit);	

    g_signal_connect(G_OBJECT(icon), "popup-menu", G_CALLBACK(menu_popup), NULL);
    g_signal_connect(G_OBJECT(icon), "activate", G_CALLBACK(icon_activate), NULL);

    /* start the event loop */
    gtk_main();

    return 0;
}

