//
//  Lynkeos
//  $Id: MyImageListItem.h,v 1.8 2005/01/27 23:07:35 j-etienne Exp $
//
//  Created by Jean-Etienne LAMIAUD on Tue Sep 30 2003.
//  Copyright (c) 2003-2005. Jean-Etienne LAMIAUD
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
// 

/*!
 * @header
 * @abstract Image list item classes
 * @discussion The root class for image list items is MyImageListItem. It is 
 *    further subclassed in concrete classes for still images (MyImage), 
 *    Quicktime movies (MyMovie)and images from a movie (MyMovieImage).
 */
#ifndef __MYIMAGELISTITEM_H
#define __MYIMAGELISTITEM_H

#import <Foundation/Foundation.h>

#ifdef GNUSTEP
#import "LynkeosCommon.h"
#import <AppKit/NSImage.h>
#import "LynkeosFfmpegMovie.h"
#endif

/*!
 * @class MyImageListItem
 * @abstract Common class for all image list items
 * @discussion It provides a unique interface to the outline view, and the 
 *    common data and methods for the image list items.
 * @ingroup Models
 */
@interface MyImageListItem : NSObject <NSCoding>
{
@protected
    NSURL*           _itemURL;
    NSString*        _itemName;           // To be displayed to user
    int              _selection_state;    // Can be tri state
    bool             _hasSearchSquare;    // Has image its own search square ?
    MyIntegerPoint   _searchSquareOrigin; // Size is constant for the whole list
    bool             _aligned;            // Was realigned (or not)
    NSPoint          _alignOffset;
    double           _quality;
}

// Initialization
- (id) initWithURL :(NSURL*)url ;

// Accessors
- (NSURL*) getURL ;
- (int) getSelectionState;
- (NSString*)getName;
- (bool) hasSearchSquare;
- (MyIntegerPoint) searchSquareOrigin;
- (bool) isAligned;
- (NSPoint) alignOffset;
- (double) getQuality;

- (NSNumber*) getIndex;			// To be implemented in descendants
- (NSImage*) getImage;			// Idem
- (NSSize) imageSize;

- (void) setSelected :(bool)value;	// Boolean option
- (void) invalidateSearchSquare ;
- (void) setSearchSquareOrigin :(MyIntegerPoint)origin;
- (void)invalidateAlignment ;
- (void) setAlignOffset :(NSPoint)offset;
- (void) setQuality :(double)quality;

// Creator
+ (id) imageListItemWithURL :(NSURL*)url;
@end

/*!
 * @class MyImage
 * @abstract Still image model class
 * @discussion It only overrides some methods from MyImageListItem
 *    and provide the non implemented ones.
 * @ingroup Models
 */
@interface MyImage : MyImageListItem
@end

/*!
 * @class MyMovieImage
 * @abstract Movie image model class
 * @discussion The image is characterized by the movie it comes from
 *    and the movie time at which it is contained.
 *
 *    The index is there to be displayed in outline view
 * @ingroup Models
 */
@interface MyMovieImage : MyImageListItem
{
@private
    id _parent;
    TimeValue _time;
    long _index;
}
// Initialization
- (id) initWithTime :(TimeValue)time parent:(id)up index:(long)index ;
// Accessors
- (TimeValue) getTime ;
- (id) getParent ;

- (void) setParent :(id)parent ;

// Creator
+ (id) movieImageWithTime :(TimeValue)time parent:(id)up index:(long)index ;
@end

/*!
 * @class MyMovie
 * @abstract Movie model class
 * @discussion It keeps track of all its MyMovieImage, provides action
 *    on them and allows to retrieve the NSImage
 * @ingroup Models
 */
@interface MyMovie : MyImageListItem <NSLocking>
{
@private
    NSMovie          *_movie;
    NSMutableArray   * _images;
    NSLock           *_qtLock;
}

- (NSMovie*) getMovie ;

- (int) numberOfImages ;
- (int) indexOfImage :(MyMovieImage*)item ;
- (MyMovieImage*) getMovieImageAtIndex :(int)index ;
- (NSImage*) getImageAtIndex :(int)index ;

- (void) addMovieImage :(MyMovieImage*)item ;
- (void) deleteMovieImage :(MyMovieImage*)item ;

- (void) imageSelectionChanged ;
@end

#endif
