/* Copyright (c) 2001-2020, David A. Clunie DBA Pixelmed Publishing. All rights reserved. */

package com.pixelmed.convert;

import com.pixelmed.apps.SetCharacteristicsFromSummary;
import com.pixelmed.apps.TiledPyramid;

import com.pixelmed.dicom.Attribute;
import com.pixelmed.dicom.AttributeList;
import com.pixelmed.dicom.AttributeTag;
import com.pixelmed.dicom.AttributeTagAttribute;
import com.pixelmed.dicom.BinaryOutputStream;
import com.pixelmed.dicom.ClinicalTrialsAttributes;
import com.pixelmed.dicom.CodeStringAttribute;
import com.pixelmed.dicom.CodedSequenceItem;
import com.pixelmed.dicom.CodingSchemeIdentification;
import com.pixelmed.dicom.CompressedFrameDecoder;
import com.pixelmed.dicom.CompressedFrameEncoder;
import com.pixelmed.dicom.DateAttribute;
import com.pixelmed.dicom.DateTimeAttribute;
import com.pixelmed.dicom.DecimalStringAttribute;
import com.pixelmed.dicom.DicomDictionary;
import com.pixelmed.dicom.DicomException;
import com.pixelmed.dicom.FileMetaInformation;
import com.pixelmed.dicom.FloatDoubleAttribute;
import com.pixelmed.dicom.FloatSingleAttribute;
import com.pixelmed.dicom.FunctionalGroupUtilities;
import com.pixelmed.dicom.IntegerStringAttribute;
import com.pixelmed.dicom.LongStringAttribute;
import com.pixelmed.dicom.LongTextAttribute;
import com.pixelmed.dicom.OtherByteAttribute;
import com.pixelmed.dicom.OtherByteAttributeMultipleCompressedFrames;
import com.pixelmed.dicom.OtherByteAttributeMultipleFilesOnDisk;
import com.pixelmed.dicom.OtherWordAttribute;
import com.pixelmed.dicom.OtherWordAttributeMultipleFilesOnDisk;
import com.pixelmed.dicom.PersonNameAttribute;
import com.pixelmed.dicom.SequenceAttribute;
import com.pixelmed.dicom.SequenceItem;
import com.pixelmed.dicom.ShortStringAttribute;
import com.pixelmed.dicom.ShortTextAttribute;
import com.pixelmed.dicom.SOPClass;
import com.pixelmed.dicom.TagFromName;
import com.pixelmed.dicom.TiledFramesIndex;
import com.pixelmed.dicom.TimeAttribute;
import com.pixelmed.dicom.TransferSyntax;
import com.pixelmed.dicom.UIDGenerator;
import com.pixelmed.dicom.UniqueIdentifierAttribute;
import com.pixelmed.dicom.UnlimitedTextAttribute;
import com.pixelmed.dicom.UnsignedLongAttribute;
import com.pixelmed.dicom.UnsignedShortAttribute;
import com.pixelmed.dicom.VersionAndConstants;

import com.pixelmed.display.SourceImage;

import com.pixelmed.utils.FileUtilities;

import java.awt.color.ColorSpace;
import java.awt.image.BufferedImage;
import java.awt.image.DataBuffer;
import java.awt.image.Raster;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.io.IOException;
import java.io.StringReader;

import java.nio.ByteOrder;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathFactory;

import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

import com.pixelmed.slf4j.Logger;
import com.pixelmed.slf4j.LoggerFactory;

/**
 * <p>A class for converting TIFF files into DICOM images of a specified or appropriate SOP Class.</p>
 *
 * <p>Defaults to producing single frame output unless a multi-frame SOP Class is explicitly
 * requested (e.g., for WSI, request Whole Slide Microscopy Image Storage, which is
 * "1.2.840.10008.5.1.4.1.1.77.1.6".</p>
 *
 * <p>Supports conversion of tiled pyramidal whole slide images such as in Aperio/Leica SVS format.</p>
 *
 * <p>Supports creation of dual-personality DICOM-TIFF files using either classic TIFF or BigTIFF,
 * optionally with inclusion of a down-sampled pyramid inside the same file in a private DICOM attribute,
 * in order to support TIFF WSI viewers that won't work without a pyramid.</p>
 *
 * <p>Uses any ICC profile present in the TIFF file otherwise assumes sRGB.</p>
 *
 * <p>Uses a JSON summary description file as the source of identification and descriptive metadata
 * as described in {@link com.pixelmed.apps.SetCharacteristicsFromSummary SetCharacteristicsFromSummary}.</p>
 *
 * <p>E.g.:</p>
 * <pre>
 * {
 * 	"top" : {
 * 		"PatientName" : "PixelMed^AperioCMU-1",
 * 		"PatientID" : "PX7832548325932",
 * 		"StudyID" : "S07-100",
 * 		"SeriesNumber" : "1",
 * 		"AccessionNumber" : "S07-100",
 * 		"ContainerIdentifier" : "S07-100 A 5 1",
 * 		"IssuerOfTheContainerIdentifierSequence" : [],
 * 		"ContainerTypeCodeSequence" : { "cv" : "433466003", "csd" : "SCT", "cm" : "Microscope slide" },
 * 		"SpecimenDescriptionSequence" : [
 * 	      {
 * 		    "SpecimenIdentifier" : "S07-100 A 5 1",
 * 		    "IssuerOfTheSpecimenIdentifierSequence" : [],
 * 		    "SpecimenUID" : "1.2.840.99790.986.33.1677.1.1.19.5",
 * 		    "SpecimenShortDescription" : "Part A: LEFT UPPER LOBE, Block 5: Mass (2 pc), Slide 1: H&amp;E",
 * 		    "SpecimenDetailedDescription" : "A: Received fresh for intraoperative consultation, labeled with the patient's name, number and 'left upper lobe,' is a pink-tan, wedge-shaped segment of soft tissue, 6.9 x 4.2 x 1.0 cm. The pleural surface is pink-tan and glistening with a stapled line measuring 12.0 cm. in length. The pleural surface shows a 0.5 cm. area of puckering. The pleural surface is inked black. The cut surface reveals a 1.2 x 1.1 cm, white-gray, irregular mass abutting the pleural surface and deep to the puckered area. The remainder of the cut surface is red-brown and congested. No other lesions are identified. Representative sections are submitted. Block 5: 'Mass' (2 pieces)",
 * 		    "SpecimenPreparationSequence" : [
 * 		      {
 * 			    "SpecimenPreparationStepContentItemSequence" : [
 * 			      {
 * 		    		"ValueType" : "TEXT",
 * 					"ConceptNameCodeSequence" : { "cv" : "121041", "csd" : "DCM", "cm" : "Specimen Identifier" },
 * 		    		"TextValue" : "S07-100 A 5 1"
 * 			      },
 * 			      {
 * 		    		"ValueType" : "CODE",
 * 					"ConceptNameCodeSequence" : { "cv" : "111701", "csd" : "DCM", "cm" : "Processing type" },
 * 					"ConceptCodeSequence" :     { "cv" : "127790008", "csd" : "SCT", "cm" : "Staining" }
 * 			      },
 * 			      {
 * 		    		"ValueType" : "CODE",
 * 					"ConceptNameCodeSequence" : { "cv" : "424361007", "csd" : "SCT", "cm" : "Using substance" },
 * 					"ConceptCodeSequence" :     { "cv" : "12710003", "csd" : "SCT", "cm" : "hematoxylin stain" }
 * 			      },
 * 			      {
 * 		    		"ValueType" : "CODE",
 * 					"ConceptNameCodeSequence" : { "cv" : "424361007", "csd" : "SCT", "cm" : "Using substance" },
 * 					"ConceptCodeSequence" :     { "cv" : "36879007", "csd" : "SCT", "cm" : "water soluble eosin stain" }
 * 			      }
 * 			    ]
 * 		      }
 * 		    ],
 * 		    "PrimaryAnatomicStructureSequence" : { "cv" : "44714003", "csd" : "SCT", "cm" : "Left Upper Lobe of Lung" }
 * 	      }
 * 		],
 * 		"OpticalPathSequence" : [
 * 	      {
 * 		    "OpticalPathIdentifier" : "1",
 * 		    "IlluminationColorCodeSequence" : { "cv" : "414298005", "csd" : "SCT", "cm" : "Full Spectrum" },
 * 		    "IlluminationTypeCodeSequence" :  { "cv" : "111744",  "csd" : "DCM", "cm" : "Brightfield illumination" }
 * 	      }
 * 		]
 * 	}
 * }
 * </pre>
 *
 * @see	com.pixelmed.apps.SetCharacteristicsFromSummary
 * @see	com.pixelmed.apps.TiledPyramid
 * @see	com.pixelmed.dicom.SOPClass
 *
 * @author	dclunie
 */

public class TIFFToDicom {
	private static final String identString = "@(#) $Header: /userland/cvs/pixelmed/imgbook/com/pixelmed/convert/TIFFToDicom.java,v 1.46 2021/06/17 17:12:43 dclunie Exp $";

	private static final Logger slf4jlogger = LoggerFactory.getLogger(TIFFToDicom.class);

	private List<File> filesToDeleteAfterWritingDicomFile = null;

	private UIDGenerator u = new UIDGenerator();
	
	private static byte[] stripSOIEOIMarkers(byte[] bytes) {
		byte[] newBytes = null;
		int l = bytes.length;
		if (l >= 4
		 && (bytes[0]&0xff) == 0xff
		 && (bytes[1]&0xff) == 0xd8
		 && (bytes[l-2]&0xff) == 0xff
		 && (bytes[l-1]&0xff) == 0xd9) {
			if (l > 4) {
				int newL = l-4;
				newBytes = new byte[newL];
				System.arraycopy(bytes,2,newBytes,0,newL);
			}
			// else leave it null since now empty
		}
		else {
			slf4jlogger.error("stripSOIEOIMarkers(): Unable to remove SOI and EOI markers");
			newBytes = bytes;
		}
		return newBytes;
	}
	
	private static byte[] insertJPEGTablesIntoAbbreviatedBitStream(byte[] bytes,byte[] jpegTables) {
		byte[] newBytes = null;
		int l = bytes.length;
		if (l > 2
		 && (bytes[0]&0xff) == 0xff
		 && (bytes[1]&0xff) == 0xd8) {
			int tableL = jpegTables.length;
			int newL = l + tableL;
			newBytes = new byte[newL];
			System.arraycopy(bytes,     0,newBytes,0,       2);
			System.arraycopy(jpegTables,0,newBytes,2,       tableL);
			System.arraycopy(bytes,     2,newBytes,2+tableL,l-2);
		}
		else {
			slf4jlogger.error("insertJPEGTablesIntoAbbreviatedBitStream(): Unable to insert JPEG Tables");
			newBytes = bytes;
		}
		return newBytes;
	}
	
	// http://www.sno.phy.queensu.ca/~phil/exiftool/TagNames/JPEG.html#Adobe
	// per JPEG-EPS.pdf 18 Adobe Application-Specific JPEG Marker
	// http://fileformats.archiveteam.org/wiki/JPEG#Color_format
	// http://docs.oracle.com/javase/8/docs/api/javax/imageio/metadata/doc-files/jpeg_metadata.html#color
	// http://exiftool.org/forum/index.php?topic=8695.0
	
	private static byte[] AdobeAPP14_RGB = {
		(byte)0xFF, (byte)0xEE,
		(byte)0x00, (byte)14,	/* big endian length includes length itself but not the marker */
		(byte)'A', (byte)'d', (byte)'o', (byte)'b', (byte)'e',
		(byte)0x00,(byte)0x65, /* DCTEncodeVersion 0x65 */
		(byte)0x00,(byte)0x00, /* APP14Flags0 0 */
		(byte)0x00,(byte)0x00, /* APP14Flags1 0 */
		(byte)0x00 /* ColorTransform 0 = Unknown (RGB or CMYK) */
	};
	
	private static byte[] insertAdobeAPP14WithRGBTransformIntoBitStream(byte[] bytes) {
		byte[] newBytes = null;
		int l = bytes.length;
		if (l > 2
		 && (bytes[0]&0xff) == 0xff
		 && (bytes[1]&0xff) == 0xd8) {
			int app14L = AdobeAPP14_RGB.length;
			int newL = l + app14L;
			newBytes = new byte[newL];
			System.arraycopy(bytes,         0,newBytes,0,       2);
			System.arraycopy(AdobeAPP14_RGB,0,newBytes,2,       app14L);
			System.arraycopy(bytes,         2,newBytes,2+app14L,l-2);
		}
		else {
			slf4jlogger.error("insertAdobeAPP14WithRGBTransformIntoBitStream(): Unable to insert APP14");
			newBytes = bytes;
		}
		return newBytes;
	}

	private void addTotalPixelMatrixOriginSequence(AttributeList list,double xOffsetInSlideCoordinateSystem,double yOffsetInSlideCoordinateSystem) throws DicomException {
		SequenceAttribute aTotalPixelMatrixOriginSequence = new SequenceAttribute(DicomDictionary.StandardDictionary.getTagFromName("TotalPixelMatrixOriginSequence"));
		list.put(aTotalPixelMatrixOriginSequence);
		{
			AttributeList itemList = new AttributeList();
			aTotalPixelMatrixOriginSequence.addItem(itemList);
			{ Attribute a = new DecimalStringAttribute(DicomDictionary.StandardDictionary.getTagFromName("XOffsetInSlideCoordinateSystem")); a.addValue(xOffsetInSlideCoordinateSystem); itemList.put(a); }
			{ Attribute a = new DecimalStringAttribute(DicomDictionary.StandardDictionary.getTagFromName("YOffsetInSlideCoordinateSystem")); a.addValue(yOffsetInSlideCoordinateSystem); itemList.put(a); }
		}
	}

	/**
	 * <p>Create a multi-frame DICOM Pixel Data attribute from the TIFF pixel data, recompressing it if requested.</p>
	 *
	 * <p>Recompresses the frames if requested, returning an updated photometric value if changed by recompression.</p>
	 *
	 * <p>Otherwise uses the supplied compressed bitstream, fixing it if necessary to signal RGB if really RGB not YCbCr and
	 * inserting factored out JPEG tables to turn abbreviated into interchange format JPEG bitstreams.</p>
	 *
	 * @param	inputFile
	 * @param	list
	 * @param	numberOfTiles
	 * @param	tileOffsets
	 * @param	tileByteCounts
	 * @param	tileWidth
	 * @param	tileLength
	 * @param	bitsPerSample
	 * @param	compression				the compression value in the TIFF source
	 * @param	photometric				the photometric value in the TIFF source
	 * @param	jpegTables				the JPEG tables in the TIFF source to be inserted in to the abbreviated format JPEG stream to make interchange format or before decompression
	 * @param	iccProfile				the ICC Profile value in the TIFF source, if any
	 * @param	recompressAsFormat		scheme to recompress uncompressed or previously compressed data if different than what was read, either "jpeg" or "jpeg2000"
	 * @param	recompressLossy			use lossy rather than lossless recompression if supported by scheme (not yet implemented)
	 * @return							the updated TIFF photometric value, which may be changed by recompression
	 * @throws	IOException				if there is an error reading or writing
	 * @throws	DicomException			if the image cannot be compressed
	 * @throws	TIFFException
	 */
	private long generateDICOMPixelDataMultiFrameImageFromTIFFFile(TIFFFile inputFile,AttributeList list,
				int numberOfTiles,long[] tileOffsets,long[] tileByteCounts,long tileWidth,long tileLength,
				long bitsPerSample,long compression,long photometric,byte[] jpegTables,byte[] iccProfile,
				String recompressAsFormat,boolean recompressLossy) throws IOException, DicomException, TIFFException {

		long outputPhotometric = photometric;
		
		if (list == null) {
			list = new AttributeList();
		}
		
		if (numberOfTiles > 2147483647l) {	// (2^31)-1 IS positive value limit
			throw new TIFFException("Number of tiles exceeds maximum IS value for NumberOfFrames = "+numberOfTiles);
		}
		if (tileWidth > 65535l || tileLength > 65535l) {	// maximum US value
			throw new TIFFException("tileWidth "+tileWidth+" and/or tileLength "+tileLength+" exceeds maximum US value for Columns and/or Rows");
		}
		
		if (compression == 0 || compression == 1) {		// absent or specified as uncompressed
			if (recompressAsFormat == null || recompressAsFormat.length() == 0) {
				// repeat the same file for every tile so that we can reuse existing MultipleFilesOnDisk classes
				File file = new File(inputFile.getFileName());
				File[] files = new File[numberOfTiles];
				for (int i=0; i<numberOfTiles; ++i) {
					files[i] = file;
				}
				if (bitsPerSample == 8) {
					slf4jlogger.debug("generateDICOMPixelDataMultiFrameImageFromTIFFFile(): copying uncompressed 8 bit input to output");
					Attribute aPixelData = new OtherByteAttributeMultipleFilesOnDisk(TagFromName.PixelData,files,tileOffsets,tileByteCounts);
					long vl = aPixelData.getPaddedVL();
					if ((vl & 0xfffffffel) != vl) {
						throw new TIFFException("Value length of Pixel Data "+vl+" exceeds maximum Value Length supported by DICOM");
					}
					list.put(aPixelData);
				}
				else if (bitsPerSample == 16) {
					slf4jlogger.debug("generateDICOMPixelDataMultiFrameImageFromTIFFFile(): copying uncompressed 16 bit input to output");
					Attribute aPixelData = new OtherWordAttributeMultipleFilesOnDisk(TagFromName.PixelData,files,tileOffsets,tileByteCounts,inputFile.getByteOrder() == ByteOrder.BIG_ENDIAN);
					long vl = aPixelData.getPaddedVL();
					if ((vl & 0xfffffffel) != vl) {
						throw new TIFFException("Value length of Pixel Data "+vl+" exceeds maximum Value Length supported by DICOM");
					}
					list.put(aPixelData);
				}
				else {
					throw new TIFFException("Unsupported bitsPerSample = "+bitsPerSample);
				}
				// photometric unchanged
			}
			else {
				if (bitsPerSample == 8) {
					slf4jlogger.debug("generateDICOMPixelDataMultiFrameImageFromTIFFFile(): compressing uncompressed 16 bit input");
					File[] files = new File[numberOfTiles];
					for (int tileNumber=0; tileNumber<numberOfTiles; ++tileNumber) {
						long pixelOffset = tileOffsets[tileNumber];
						long pixelByteCount = tileByteCounts[tileNumber];
						byte[] values = new byte[(int)pixelByteCount];
						inputFile.seek(pixelOffset);
						inputFile.read(values);
						BufferedImage img = SourceImage.createPixelInterleavedByteThreeComponentColorImage(
							(int)tileWidth,(int)tileLength,values,0/*offset*/,
							ColorSpace.getInstance(ColorSpace.CS_sRGB),	// should check for presence of TIFF ICC profile ? :(
							false/*isChrominanceHorizontallyDownsampledBy2*/);

						// recompressLossy not yet implemented ... default for JPEG is best quality, J2K is lossless :(
						// will always transform color space by default
						File tmpFile = CompressedFrameEncoder.getCompressedFrameAsFile(new AttributeList(),img,recompressAsFormat,File.createTempFile("TIFFToDicom","."+recompressAsFormat));
						files[tileNumber] = tmpFile;
						tmpFile.deleteOnExit();
						if (slf4jlogger.isTraceEnabled()) slf4jlogger.trace("Tile {} created compressed temporary file {}",tileNumber,tmpFile.toString());
						// photometric changed, since CompressedFrameEncoder always transforms color space
						outputPhotometric = 6;	// TIFF definition of YCbCr is generic, so use it to signal YBR_FULL_4r22 for JPEG and YBR_RCT or YBR_ICT for J2K
					}
					Attribute aPixelData = new OtherByteAttributeMultipleCompressedFrames(TagFromName.PixelData,files);
					list.put(aPixelData);
					if (filesToDeleteAfterWritingDicomFile == null) {
						filesToDeleteAfterWritingDicomFile = new ArrayList<>(Arrays.asList(files));	// make a copy because Arrays.asList() is documented not to support any adding elements "https://stackoverflow.com/questions/5755477/java-list-add-unsupportedoperationexception"
					}
					else {
						Collections.addAll(filesToDeleteAfterWritingDicomFile,files);
					}
				}
				else {
					throw new TIFFException("Unsupported bitsPerSample = "+bitsPerSample+" for compression");
				}
				//throw new TIFFException("Compression as "+(recompressLossy ? "lossy" : "lossless")+" "+recompressAsFormat+" not supported");
			}
		}
		else if (compression == 7 && recompressAsFormat.equals("jpeg")				// "new" JPEG per TTN2 as used by Aperio in SVS
			  || (compression == 33003 || compression == 33005) && recompressAsFormat.equals("jpeg2000")) {	// Aperio J2K YCbCr or RGB
			slf4jlogger.debug("generateDICOMPixelDataMultiFrameImageFromTIFFFile(): copying compressed bit stream from input to output without recompressing it");
			// because we need to edit the stream to insert the jpegTables, need to write lots of temporary files to feed to OtherByteAttributeMultipleCompressedFrames file-based constructor
			File[] files = new File[numberOfTiles];
			for (int tileNumber=0; tileNumber<numberOfTiles; ++tileNumber) {
				long pixelOffset = tileOffsets[tileNumber];
				long pixelByteCount = tileByteCounts[tileNumber];
				if (pixelByteCount > Integer.MAX_VALUE) {
					throw new TIFFException("For frame "+tileNumber+", compressed pixelByteCount to be read "+pixelByteCount+" exceeds maximum Java array size "+Integer.MAX_VALUE+" and fragmentation not yet supported");
				}
				byte[] values = new byte[(int)pixelByteCount];
				inputFile.seek(pixelOffset);
				inputFile.read(values);
				
				if (jpegTables != null) {		// should not be present for 33005
					values = insertJPEGTablesIntoAbbreviatedBitStream(values,jpegTables);
				}
				if (compression == 7/*JPEG*/ && photometric == 2/*RGB*/) {
					slf4jlogger.trace("JPEG RGB so adding APP14");
					values = insertAdobeAPP14WithRGBTransformIntoBitStream(values);
				}
				
				if (values.length > 0xfffffffel) {
					throw new TIFFException("For frame "+tileNumber+", compressed pixelByteCount to be written "+values.length+" exceeds maximum single fragment size 0xfffffffe and fragmentation not yet supported");
				}
				File tmpFile = File.createTempFile("TIFFToDicom",".jpeg");
				files[tileNumber] = tmpFile;
				tmpFile.deleteOnExit();
				BufferedOutputStream o = new BufferedOutputStream(new FileOutputStream(tmpFile));
				o.write(values);
				o.flush();
				o.close();
				if (slf4jlogger.isTraceEnabled()) slf4jlogger.trace("Tile {} wrote {} bytes to {}",tileNumber,values.length,tmpFile.toString());
				if (compression == 33003
				 /*|| compression == 33005*/) {	// do NOT change since no MCT (value is 0 in SGcod) (001263)
					outputPhotometric = 6;	// TIFF definition of YCbCr is generic, so use it to signal YBR_RCT or YBR_ICT for J2K
				}
				//else photometric unchanged
			}
			Attribute aPixelData = new OtherByteAttributeMultipleCompressedFrames(TagFromName.PixelData,files);
			list.put(aPixelData);
			if (filesToDeleteAfterWritingDicomFile == null) {
				filesToDeleteAfterWritingDicomFile = new ArrayList<>(Arrays.asList(files));	// make a copy because Arrays.asList() is documented not to support any adding elements "https://stackoverflow.com/questions/5755477/java-list-add-unsupportedoperationexception"
			}
			else {
				Collections.addAll(filesToDeleteAfterWritingDicomFile,files);
			}
		}
		else if ((compression == 7 || compression == 33003 || compression == 33005)		// "new" JPEG per TTN2 as used by Aperio in SVS, Aperio J2K YCbCr or RGB
			  && (recompressAsFormat.equals("jpeg") || recompressAsFormat.equals("jpeg2000"))) {
			// decompress and recompress each frame
			{
				if (bitsPerSample == 8) {
					slf4jlogger.debug("generateDICOMPixelDataMultiFrameImageFromTIFFFile(): decompressing each 8 bit input frame and recompressing it");
					String transferSyntax = compression == 7 ? TransferSyntax.JPEGBaseline : TransferSyntax.JPEG2000;	// take care to keep this in sync with enclosing test of supported schemes
					CompressedFrameDecoder decoder = new CompressedFrameDecoder(
						transferSyntax,
						1/*bytesPerSample*/,
						(int)tileWidth,(int)tileLength,
						3/*samples*/,		// hmmm ..../ :(
						ColorSpace.getInstance(ColorSpace.CS_sRGB),
						photometric == 6);	// should check for presence of TIFF ICC profile ? :(

					File[] files = new File[numberOfTiles];
					for (int tileNumber=0; tileNumber<numberOfTiles; ++tileNumber) {
						long pixelOffset = tileOffsets[tileNumber];
						long pixelByteCount = tileByteCounts[tileNumber];
						if (pixelByteCount > Integer.MAX_VALUE) {
							throw new TIFFException("For frame "+tileNumber+", compressed pixelByteCount to be read "+pixelByteCount+" exceeds maximum Java array size "+Integer.MAX_VALUE+" and fragmentation not yet supported");
						}
						byte[] values = new byte[(int)pixelByteCount];
						inputFile.seek(pixelOffset);
						inputFile.read(values);

						if (jpegTables != null) {		// should not be present for 33003 or 33005
							values = insertJPEGTablesIntoAbbreviatedBitStream(values,jpegTables);
						}
						if (compression == 7/*JPEG*/ && photometric == 2/*RGB*/) {
							slf4jlogger.trace("JPEG RGB so adding APP14");
							values = insertAdobeAPP14WithRGBTransformIntoBitStream(values);
						}
						
						BufferedImage img = decoder.getDecompressedFrameAsBufferedImage(values);
						
						// recompressLossy not yet implemented ... default for JPEG is best quality, J2K is lossless :(
						// will always transform color space by default
						File tmpFile = CompressedFrameEncoder.getCompressedFrameAsFile(new AttributeList(),img,recompressAsFormat,File.createTempFile("TIFFToDicom","."+recompressAsFormat));
						files[tileNumber] = tmpFile;
						tmpFile.deleteOnExit();
						if (slf4jlogger.isTraceEnabled()) slf4jlogger.trace("Tile {} created compressed temporary file {}",tileNumber,tmpFile.toString());
						// photometric changed, since CompressedFrameEncoder always transforms color space
						outputPhotometric = 6;	// TIFF definition of YCbCr is generic, so use it to signal YBR_FULL_422 for JPEG and YBR_RCT or YBR_ICT for J2K
					}
					Attribute aPixelData = new OtherByteAttributeMultipleCompressedFrames(TagFromName.PixelData,files);
					list.put(aPixelData);
					if (filesToDeleteAfterWritingDicomFile == null) {
						filesToDeleteAfterWritingDicomFile = new ArrayList<>(Arrays.asList(files));	// make a copy because Arrays.asList() is documented not to support any adding elements "https://stackoverflow.com/questions/5755477/java-list-add-unsupportedoperationexception"
					}
					else {
						Collections.addAll(filesToDeleteAfterWritingDicomFile,files);
					}
				}
				else {
					throw new TIFFException("Unsupported bitsPerSample = "+bitsPerSample+" for compression");
				}
				//throw new TIFFException("Recompression as "+(recompressLossy ? "lossy" : "lossless")+" "+recompressAsFormat+" not supported");
			}
		}
		else {
			throw new TIFFException("Unsupported compression = "+compression+" or unsupported transformation to "+recompressAsFormat);
		}

		return outputPhotometric;
	}
	
	private long generateDICOMPixelDataSingleFrameImageFromTIFFFileMergingStrips(TIFFFile inputFile,AttributeList list,
				long imageWidth,long imageLength,
				long[] pixelOffset,long[] pixelByteCount,long pixelWidth,long rowsPerStrip,
				long bitsPerSample,long compression,long photometric,long samplesPerPixel,
				byte[] jpegTables,byte[] iccProfile,String recompressAsFormat,boolean recompressLossy) throws IOException, DicomException, TIFFException {
		
		long outputPhotometric = photometric;
		
		if (list == null) {
			list = new AttributeList();
		}

		if (compression == 0 || compression == 1) {		// absent or specified as uncompressed
			if (recompressAsFormat == null || recompressAsFormat.length() == 0) {
				if (bitsPerSample == 8) {
					long totalLength = imageWidth * imageLength * samplesPerPixel;
					if (totalLength %2 != 0) ++totalLength;
					//long totalLength = 0;
					//for (int i=0; i<pixelByteCount.length; ++i) {
					//	totalLength += pixelByteCount[i];
					//}
					//if (totalLength%2 == 1) ++totalLength;
					slf4jlogger.debug("generateDICOMPixelDataSingleFrameImageFromTIFFFileMergingStrips(): totalLength = {}",totalLength);
					if (totalLength > Integer.MAX_VALUE) {
						throw new TIFFException("Uncompressed image too large to allocate = "+totalLength);
					}
					byte[] values = new byte[(int)totalLength];
					int offsetIntoValues = 0;
					for (int i=0; i<pixelOffset.length; ++i) {
						long fileOffset = pixelOffset[i];
						slf4jlogger.debug("generateDICOMPixelDataSingleFrameImageFromTIFFFileMergingStrips(): pixelOffset[{}] = {}",i,fileOffset);
						inputFile.seek(fileOffset);
						int bytesToRead = (int)pixelByteCount[i];
						slf4jlogger.debug("generateDICOMPixelDataSingleFrameImageFromTIFFFileMergingStrips(): pixelByteCount[{}] = {}",i,bytesToRead);
						inputFile.read(values,offsetIntoValues,bytesToRead);
						offsetIntoValues += bytesToRead;
					}
					Attribute aPixelData = new OtherByteAttribute(TagFromName.PixelData);
					aPixelData.setValues(values);
					list.put(aPixelData);
				}
				else {
					throw new TIFFException("Unsupported bitsPerSample = "+bitsPerSample);
				}
			}
			else {
				throw new TIFFException("Compression as "+(recompressLossy ? "lossy" : "lossless")+" "+recompressAsFormat+" not supported");
			}
		}
		else if (compression == 7) {				// "new" JPEG per TTN2 as used by Aperio in SVS
			if (recompressAsFormat == null || recompressAsFormat.length() == 0) {
				// decompress each strip
				if (bitsPerSample == 8) {
					slf4jlogger.debug("generateDICOMPixelDataSingleFrameImageFromTIFFFileMergingStrips(): decompressing each 8 bit input strip");
					CompressedFrameDecoder decoder = new CompressedFrameDecoder(
						TransferSyntax.JPEGBaseline,
						1/*bytesPerSample*/,
						(int)pixelWidth,(int)rowsPerStrip,
						3/*samples*/,		// hmmm ..../ :(
						ColorSpace.getInstance(ColorSpace.CS_sRGB),
						photometric == 6);	// should check for presence of TIFF ICC profile ? :(
					
					long totalLength = imageWidth * imageLength * samplesPerPixel;
					if (totalLength %2 != 0) ++totalLength;
					slf4jlogger.debug("generateDICOMPixelDataSingleFrameImageFromTIFFFileMergingStrips(): totalLength = {}",totalLength);
					if (totalLength > Integer.MAX_VALUE) {
						throw new TIFFException("Uncompressed image too large to allocate = "+totalLength);
					}
					byte[] values = new byte[(int)totalLength];
					int offsetIntoValues = 0;
					for (int i=0; i<pixelOffset.length; ++i) {
						byte[] compressedValues = new byte[(int)pixelByteCount[i]];
						inputFile.seek(pixelOffset[i]);
						inputFile.read(compressedValues);
						if (jpegTables != null) {
							compressedValues = insertJPEGTablesIntoAbbreviatedBitStream(compressedValues,jpegTables);
						}
						if (compression == 7/*JPEG*/ && photometric == 2/*RGB*/) {
//System.err.println("JPEG RGB so adding APP14");
							compressedValues = insertAdobeAPP14WithRGBTransformIntoBitStream(compressedValues);
						}
						BufferedImage img = decoder.getDecompressedFrameAsBufferedImage(compressedValues);
						int decompressedWidth = img.getWidth();
						slf4jlogger.trace("generateDICOMPixelDataSingleFrameImageFromTIFFFileMergingStrips(): strip {} decompressedWidth = {}",i,decompressedWidth);
						int decompressedHeight = img.getHeight();
						slf4jlogger.trace("generateDICOMPixelDataSingleFrameImageFromTIFFFileMergingStrips(): strip {} decompressedHeight = {}",i,decompressedHeight);
						Raster raster = img.getData();
						if (raster.getTransferType() == DataBuffer.TYPE_BYTE) {
							byte[] decompressedStrip = (byte[])(raster.getDataElements(0,0,decompressedWidth,decompressedHeight,null));
							System.arraycopy(decompressedStrip,0,values,offsetIntoValues,decompressedStrip.length);
							offsetIntoValues += decompressedStrip.length;
						}
					}
					Attribute aPixelData = new OtherByteAttribute(TagFromName.PixelData);
					aPixelData.setValues(values);
					list.put(aPixelData);
				}
				else {
					throw new TIFFException("Unsupported bitsPerSample = "+bitsPerSample);
				}
			}
			else {
				throw new TIFFException("Compression as "+(recompressLossy ? "lossy" : "lossless")+" "+recompressAsFormat+" not supported");
			}
		}
		else {
			throw new TIFFException("Unsupported compression = "+compression);
		}

		return outputPhotometric;
	}

	private long generateDICOMPixelDataSingleFrameImageFromTIFFFile(TIFFFile inputFile,AttributeList list,
				long pixelOffset,long pixelByteCount,long pixelWidth,long pixelLength,
				long bitsPerSample,long compression,long photometric,byte[] jpegTables,byte[] iccProfile,String recompressAsFormat,boolean recompressLossy) throws IOException, DicomException, TIFFException {
		
		long outputPhotometric = photometric;
		
		if (list == null) {
			list = new AttributeList();
		}
		
		inputFile.seek(pixelOffset);
		if (compression == 0 || compression == 1) {		// absent or specified as uncompressed
			if (recompressAsFormat == null || recompressAsFormat.length() == 0) {
				if (bitsPerSample == 8) {
					byte[] values = new byte[(int)pixelByteCount];
					inputFile.read(values);
					Attribute aPixelData = new OtherByteAttribute(TagFromName.PixelData);
					aPixelData.setValues(values);
					list.put(aPixelData);
				}
				else if (bitsPerSample == 16) {
					short[] values = new short[(int)(pixelByteCount/2)];
					inputFile.read(values);
					Attribute aPixelData = new OtherWordAttribute(TagFromName.PixelData);
					aPixelData.setValues(values);
					list.put(aPixelData);
				}
				else {
					throw new TIFFException("Unsupported bitsPerSample = "+bitsPerSample);
				}
			}
			else {
				throw new TIFFException("Compression as "+(recompressLossy ? "lossy" : "lossless")+" "+recompressAsFormat+" not supported");
			}
		}
		else if (compression == 7				// "new" JPEG per TTN2 as used by Aperio in SVS
			  || compression == 33003			// Aperio J2K YCbCr
			  || compression == 33005) {		// Aperio J2K RGB
			byte[] values = new byte[(int)pixelByteCount];
			inputFile.read(values);
			if (jpegTables != null) {			// should not be present for 33003 or 33005
				values = insertJPEGTablesIntoAbbreviatedBitStream(values,jpegTables);
			}
			if (compression == 7/*JPEG*/ && photometric == 2/*RGB*/) {
//System.err.println("JPEG RGB so adding APP14");
				values = insertAdobeAPP14WithRGBTransformIntoBitStream(values);
			}
			byte[][] frames = new byte[1][];
			frames[0] = values;
			Attribute aPixelData = new OtherByteAttributeMultipleCompressedFrames(TagFromName.PixelData,frames);
			list.put(aPixelData);
			if (compression == 33003
			 /*|| compression == 33005*/) {	// do NOT change since no MCT (value is 0 in SGcod) (001263)
				outputPhotometric = 6;	// TIFF definition of YCbCr is generic, so use it to signal YBR_RCT or YBR_ICT for J2K
			}
			//else photometric unchanged
		}
		else {
			throw new TIFFException("Unsupported compression = "+compression);
		}

		return outputPhotometric;
	}
	
	private static AttributeList generateDICOMPixelDataModuleAttributes(AttributeList list,
			int numberOfFrames,long pixelWidth,long pixelLength,
			long bitsPerSample,long compression,long photometric,long samplesPerPixel,long planarConfig,long sampleFormat,String recompressAsFormat,boolean recompressLossy,String sopClass) throws IOException, DicomException, TIFFException {
		
		if (list == null) {
			list = new AttributeList();
		}
		
		String photometricInterpretation = "";
		switch ((int)photometric) {
			case 0:	photometricInterpretation = "MONOCHROME1"; break;
			case 1:	photometricInterpretation = "MONOCHROME2"; break;
			case 2:	photometricInterpretation = "RGB"; break;
			case 3:	photometricInterpretation = "PALETTE COLOR"; break;
			case 4:	photometricInterpretation = "TRANSPARENCY"; break;		// not standard DICOM
			case 5:	photometricInterpretation = "CMYK"; break;				// retired in DICOM
			case 6:	photometricInterpretation = (recompressAsFormat != null && recompressAsFormat.equals("jpeg2000")) ? (recompressLossy ? "YBR_ICT" : "YBR_RCT") : "YBR_FULL_422"; break;
			case 8:	photometricInterpretation = "CIELAB"; break;			// not standard DICOM
		}
		{ Attribute a = new CodeStringAttribute(TagFromName.PhotometricInterpretation); a.addValue(photometricInterpretation); list.put(a); }

		{ Attribute a = new UnsignedShortAttribute(TagFromName.BitsAllocated); a.addValue((int)bitsPerSample); list.put(a); }
		{ Attribute a = new UnsignedShortAttribute(TagFromName.BitsStored); a.addValue((int)bitsPerSample); list.put(a); }
		{ Attribute a = new UnsignedShortAttribute(TagFromName.HighBit); a.addValue((int)bitsPerSample-1); list.put(a); }
		{ Attribute a = new UnsignedShortAttribute(TagFromName.Rows); a.addValue((int)pixelLength); list.put(a); }
		{ Attribute a = new UnsignedShortAttribute(TagFromName.Columns); a.addValue((int)pixelWidth); list.put(a); }
			

		boolean signed = false;
		if (sampleFormat == 2) {
			signed = true;
			// do not check for other values, like 3 for IEEE float
		}
		{ Attribute a = new UnsignedShortAttribute(TagFromName.PixelRepresentation); a.addValue(signed ? 1 : 0); list.put(a); }

		list.remove(TagFromName.NumberOfFrames);
		if (SOPClass.isMultiframeImageStorage(sopClass)) {
			Attribute a = new IntegerStringAttribute(TagFromName.NumberOfFrames); a.addValue(numberOfFrames); list.put(a);
		}
			
		{ Attribute a = new UnsignedShortAttribute(TagFromName.SamplesPerPixel); a.addValue((int)samplesPerPixel); list.put(a); }
						
		list.remove(TagFromName.PlanarConfiguration);
		if (samplesPerPixel > 1) {
				Attribute a = new UnsignedShortAttribute(TagFromName.PlanarConfiguration); a.addValue((int)planarConfig-1); list.put(a);	// TIFF is 1 or 2 but sometimes absent (0), DICOM is 0 or 1
		}

		return list;
	}

	// copied and derived from CommonConvertedAttributeGeneration.addParametricMapFrameTypeSharedFunctionalGroup() - should refactor :(
	private static AttributeList addWholeSlideMicroscopyImageFrameTypeSharedFunctionalGroup(AttributeList list,String imageFlavor,String imageDerivation) throws DicomException {
		// override default from CommonConvertedAttributeGeneration; same as FrameType; no way of determining this and most are VOLUME not LABEL or LOCALIZER :(
		Attribute aFrameType = new CodeStringAttribute(TagFromName.FrameType);
		aFrameType.addValue("DERIVED");
		aFrameType.addValue("PRIMARY");
		aFrameType.addValue(imageFlavor);
		aFrameType.addValue(imageDerivation);
		list = FunctionalGroupUtilities.generateFrameTypeSharedFunctionalGroup(list,DicomDictionary.StandardDictionary.getTagFromName("WholeSlideMicroscopyImageFrameTypeSequence"),aFrameType);
		return list;
	}
	
	private byte[] addICCProfileToOpticalPathSequence(AttributeList list,byte[] iccProfile) throws DicomException {
		AttributeList opticalPathSequenceItemList = SequenceAttribute.getAttributeListFromWithinSequenceWithSingleItem(list,DicomDictionary.StandardDictionary.getTagFromName("OpticalPathSequence"));
		if (opticalPathSequenceItemList != null) {
			if (iccProfile == null || iccProfile.length == 0) {
				InputStream iccProfileStream = getClass().getResourceAsStream("/com/pixelmed/dicom/sRGBColorSpaceProfileInputDevice.icc");
				try {
					iccProfile = FileUtilities.readAllBytes(iccProfileStream);
					int iccProfileLength = iccProfile.length;
					if (iccProfileLength %2 != 0) {
						byte[] newICCProfile = new byte[iccProfileLength+1];
						System.arraycopy(iccProfile,0,newICCProfile,0,iccProfileLength);
						iccProfile = newICCProfile;
						iccProfileLength = iccProfile.length;
					}
				}
				catch (IOException e) {
					throw new DicomException("Failed to read ICC profile resource: "+e);
				}
				{ Attribute a = new CodeStringAttribute(DicomDictionary.StandardDictionary.getTagFromName("ColorSpace")); a.addValue("SRGB"); opticalPathSequenceItemList.put(a); }
			}
			else {
				slf4jlogger.debug("Using ICC Profile from TIFF IFD");
				// do not add ColorSpace since we do not know what it is or if it is any recognized standard value
			}
			if (iccProfile != null && iccProfile.length > 0) {
				{ Attribute a = new OtherByteAttribute(TagFromName.ICCProfile); a.setValues(iccProfile); opticalPathSequenceItemList.put(a); }
				slf4jlogger.debug("addICCProfileToOpticalPathSequence(): Created ICC Profile attribute of length {}",iccProfile.length);
			}
		}
		return iccProfile;
	}

	// (001270)
	private void addObjectiveLensPowerToOpticalPathSequence(AttributeList list,double objectiveLensPower) throws DicomException {
		if (objectiveLensPower != 0) {
			AttributeList opticalPathSequenceItemList = SequenceAttribute.getAttributeListFromWithinSequenceWithSingleItem(list,DicomDictionary.StandardDictionary.getTagFromName("OpticalPathSequence"));
			if (opticalPathSequenceItemList != null) {
				double existingObjectiveLensPower = Attribute.getSingleDoubleValueOrDefault(list,DicomDictionary.StandardDictionary.getTagFromName("ObjectiveLensPower"),0);
				if (existingObjectiveLensPower == 0) {
					{ Attribute a = new DecimalStringAttribute(DicomDictionary.StandardDictionary.getTagFromName("ObjectiveLensPower")); a.addValue(objectiveLensPower); opticalPathSequenceItemList.put(a); }
					slf4jlogger.debug("addObjectiveLensPowerToOpticalPathSequence(): added ObjectiveLensPower {}",objectiveLensPower);
				}
				else {
					slf4jlogger.debug("addObjectiveLensPowerToOpticalPathSequence(): not overriding non-zero ObjectiveLensPower {} with replacement {}",existingObjectiveLensPower,objectiveLensPower);
				}
			}
			else {
				slf4jlogger.debug("addObjectiveLensPowerToOpticalPathSequence(): no opticalPathSequenceItemList to add to - not trying to create it now (too late)");
			}
		}
		else {
			slf4jlogger.debug("addObjectiveLensPowerToOpticalPathSequence(): no ObjectiveLensPower value to use, so nothing to do");
		}
	}
	
	// (001285)
	private void addOpticalPathIdentifierAndDescriptionToOpticalPathSequence(AttributeList list,String opticalPathIdentifier,String opticalPathDescription) throws DicomException {
		if (opticalPathIdentifier != null && opticalPathIdentifier.length() > 0
		 || opticalPathDescription != null && opticalPathDescription.length() > 0) {
			AttributeList opticalPathSequenceItemList = SequenceAttribute.getAttributeListFromWithinSequenceWithSingleItem(list,DicomDictionary.StandardDictionary.getTagFromName("OpticalPathSequence"));
			if (opticalPathSequenceItemList != null) {
				if (opticalPathIdentifier != null && opticalPathIdentifier.length() > 0) {
					{ Attribute a = new ShortStringAttribute(DicomDictionary.StandardDictionary.getTagFromName("OpticalPathIdentifier")); a.addValue(opticalPathIdentifier); opticalPathSequenceItemList.put(a); }
					slf4jlogger.debug("addOpticalPathIdentifierAndDescriptionToOpticalPathSequence(): added or replacing opticalPathIdentifier {}",opticalPathIdentifier);
				}
				if (opticalPathDescription != null && opticalPathDescription.length() > 0) {
					{ Attribute a = new ShortTextAttribute(DicomDictionary.StandardDictionary.getTagFromName("OpticalPathDescription")); a.addValue(opticalPathDescription); opticalPathSequenceItemList.put(a); }
					slf4jlogger.debug("addOpticalPathIdentifierAndDescriptionToOpticalPathSequence(): added or replacing opticalPathDescription {}",opticalPathDescription);
				}
			}
			else {
				slf4jlogger.debug("addOpticalPathIdentifierAndDescriptionToOpticalPathSequence(): no opticalPathSequenceItemList to add to - not trying to create it now (too late)");
			}
		}
		else {
			slf4jlogger.debug("addOpticalPathIdentifierAndDescriptionToOpticalPathSequence(): no OpticalPathIdentifier or OpticalPathDescription value to use, so nothing to do");
		}
	}
	

	private final long totalArrayValues(long[] values) {
		long total = 0;
		if (values != null) {
			for (long value : values) {
				total+=value;
			}
		}
		return total;
	}

	private AttributeList insertLossyImageCompressionHistory(AttributeList list,
			long compression,long outputCompression,
			long originalCompressedByteCount,long imageWidth,long imageLength,long bitsPerSample,long samplesPerPixel
			) throws DicomException {
		
		if (list == null) {
			list = new AttributeList();
		}
		
		// inspired by AttributeList.insertLossyImageCompressionHistoryIfDecompressed() but can't reuse directly - should refactor :(

		{
			String lossyImageCompression="00";
			Set<String> lossyImageCompressionMethod = new HashSet<String>();
			boolean wasOriginallyCompressed = false;
			boolean wasRecompressed = false;

			if (compression == 7) {			// "new" JPEG per TTN2 as used by Aperio in SVS
				wasOriginallyCompressed = true;
				lossyImageCompression="01";
				lossyImageCompressionMethod.add("ISO_10918_1");
			}
			else if (compression == 33003	// Aperio J2K YCbCr
				  || compression == 33005	// Aperio J2K RGB
			) {
				wasOriginallyCompressed = true;
				lossyImageCompression="01";
				lossyImageCompressionMethod.add("ISO_15444_1");
			}
			
			if (outputCompression == 7) {		// "new" JPEG per TTN2 as used by Aperio in SVS
				lossyImageCompression="01";
				lossyImageCompressionMethod.add("ISO_10918_1");
				if (wasOriginallyCompressed && compression != outputCompression) {
					wasRecompressed = true;
				}
			}
			else if (outputCompression == 33003	// Aperio J2K YCbCr
				  || outputCompression == 33005	// Aperio J2K RGB
			) {
				lossyImageCompression="01";
				lossyImageCompressionMethod.add("ISO_15444_1");
				if (wasOriginallyCompressed && compression != outputCompression) {
					wasRecompressed = true;
				}
			}
			
			{ Attribute a = new CodeStringAttribute(TagFromName.LossyImageCompression); a.addValue(lossyImageCompression); list.put(a); }
			
			if (!lossyImageCompressionMethod.isEmpty()) {
				Attribute a = new CodeStringAttribute(TagFromName.LossyImageCompressionMethod);
				for (String v : lossyImageCompressionMethod) {
					a.addValue(v);
				}
				list.put(a);
			}
			
			if (wasOriginallyCompressed) {
				// compute CR with precision of three decimal places
				long bytesPerSample = (bitsPerSample-1)/8+1;
				long decompressedByteCount = imageWidth*imageLength*samplesPerPixel*bytesPerSample;
				double compressionRatio = (long)(decompressedByteCount*1000/originalCompressedByteCount);
				compressionRatio = compressionRatio / 1000;
				slf4jlogger.debug("insertLossyImageCompressionHistory(): decompressedByteCount = {}",decompressedByteCount);
				slf4jlogger.debug("insertLossyImageCompressionHistory(): wasOriginallyCompressed with originalCompressedByteCount = {}",originalCompressedByteCount);
				slf4jlogger.debug("insertLossyImageCompressionHistory(): wasOriginallyCompressed with compressionRatio = {}",compressionRatio);
				Attribute a = new DecimalStringAttribute(TagFromName.LossyImageCompressionRatio);
				a.addValue(compressionRatio);
				list.put(a);
			}
			
			if (wasRecompressed) {
				// should add new compression ratio value based on recompressedByteCount :(
			}
		}

		return list;
	}
	
	private AttributeList generateDICOMWholeSlideMicroscopyImageAttributes(AttributeList list,
			long imageWidth,long imageLength,String frameOfReferenceUID,double mmPerPixel,double objectiveLensPower,
			String opticalPathIdentifier,String opticalPathDescription,
			double xOffsetInSlideCoordinateSystem,double yOffsetInSlideCoordinateSystem,
			String containerIdentifier,String specimenIdentifier,String specimenUID,
			String imageFlavor,String imageDerivation) throws DicomException {
		
		if (list == null) {
			list = new AttributeList();
		}
		
		// Frame of Reference Module
		{ Attribute a = new UniqueIdentifierAttribute(TagFromName.FrameOfReferenceUID); a.addValue(frameOfReferenceUID); list.put(a); }	// (001244) (001256)
		{ Attribute a = new LongStringAttribute(TagFromName.PositionReferenceIndicator); a.addValue("SLIDE_CORNER"); list.put(a); }	// (001272)

		// Whole Slide Microscopy Series Module
		
		// Multi-frame Functional Groups Module
		
		addWholeSlideMicroscopyImageFrameTypeSharedFunctionalGroup(list,imageFlavor,imageDerivation);

		{
			SequenceAttribute aSharedFunctionalGroupsSequence = (SequenceAttribute)list.get(TagFromName.SharedFunctionalGroupsSequence);
			AttributeList sharedFunctionalGroupsSequenceList = SequenceAttribute.getAttributeListFromWithinSequenceWithSingleItem(aSharedFunctionalGroupsSequence);

			SequenceAttribute aPixelMeasuresSequence = new SequenceAttribute(TagFromName.PixelMeasuresSequence);
			sharedFunctionalGroupsSequenceList.put(aPixelMeasuresSequence);
			AttributeList itemList = new AttributeList();
			aPixelMeasuresSequence.addItem(itemList);

			// note that order in DICOM in PixelSpacing is "adjacent row spacing", then "adjacent column spacing" ...
			{ Attribute a = new DecimalStringAttribute(TagFromName.PixelSpacing); a.addValue(mmPerPixel); a.addValue(mmPerPixel); itemList.put(a); }
			{ Attribute a = new DecimalStringAttribute(TagFromName.SliceThickness); a.addValue(0); itemList.put(a); }	// No way of determining this but required :(
			//{ Attribute a = new DecimalStringAttribute(TagFromName.SpacingBetweenSlices); a.addValue(sliceSpacing); itemList.put(a); }
		}

		
		// Multi-frame Dimension Module - add it even though we are using TILED_FULL so not adding Per-Frame Functional Group :(
		{
			// derived from IndexedLabelMapToSegmentation.IndexedLabelMapToSegmentation() - should refactor :(
			String dimensionOrganizationUID = u.getAnotherNewUID();
			{
				SequenceAttribute saDimensionOrganizationSequence = new SequenceAttribute(TagFromName.DimensionOrganizationSequence);
				list.put(saDimensionOrganizationSequence);
				{
					AttributeList itemList = new AttributeList();
					saDimensionOrganizationSequence.addItem(itemList);
					{ Attribute a = new UniqueIdentifierAttribute(TagFromName.DimensionOrganizationUID); a.addValue(dimensionOrganizationUID); itemList.put(a); }
				}
			}
			{ Attribute a = new CodeStringAttribute(TagFromName.DimensionOrganizationType); a.addValue("TILED_FULL"); list.put(a); }
			{
				SequenceAttribute saDimensionIndexSequence = new SequenceAttribute(TagFromName.DimensionIndexSequence);
				list.put(saDimensionIndexSequence);
				{
					AttributeList itemList = new AttributeList();
					saDimensionIndexSequence.addItem(itemList);
					{ AttributeTagAttribute a = new AttributeTagAttribute(TagFromName.DimensionIndexPointer); a.addValue(TagFromName.RowPositionInTotalImagePixelMatrix); itemList.put(a); }
					{ AttributeTagAttribute a = new AttributeTagAttribute(TagFromName.FunctionalGroupPointer); a.addValue(TagFromName.PlanePositionSlideSequence); itemList.put(a); }
					{ Attribute a = new UniqueIdentifierAttribute(TagFromName.DimensionOrganizationUID); a.addValue(dimensionOrganizationUID); itemList.put(a); }
					{ Attribute a = new LongStringAttribute(TagFromName.DimensionDescriptionLabel); a.addValue("Row Position"); itemList.put(a); }
				}
				{
					AttributeList itemList = new AttributeList();
					saDimensionIndexSequence.addItem(itemList);
					{ AttributeTagAttribute a = new AttributeTagAttribute(TagFromName.DimensionIndexPointer); a.addValue(TagFromName.ColumnPositionInTotalImagePixelMatrix); itemList.put(a); }
					{ AttributeTagAttribute a = new AttributeTagAttribute(TagFromName.FunctionalGroupPointer); a.addValue(TagFromName.PlanePositionSlideSequence); itemList.put(a); }
					{ Attribute a = new UniqueIdentifierAttribute(TagFromName.DimensionOrganizationUID); a.addValue(dimensionOrganizationUID); itemList.put(a); }
					{ Attribute a = new LongStringAttribute(TagFromName.DimensionDescriptionLabel); a.addValue("Column Position"); itemList.put(a); }
				}
			}
		}


		// Specimen Module

		{ Attribute a = new LongStringAttribute(DicomDictionary.StandardDictionary.getTagFromName("ContainerIdentifier")); a.addValue(containerIdentifier); list.put(a); }					// Dummy value - should be able to override this :(
		{ Attribute a = new SequenceAttribute(DicomDictionary.StandardDictionary.getTagFromName("IssuerOfTheContainerIdentifierSequence")); list.put(a); }
		CodedSequenceItem.putSingleCodedSequenceItem(list,DicomDictionary.StandardDictionary.getTagFromName("ContainerTypeCodeSequence"),"433466003","SCT","Microscope slide");	// No way of determining this :(
		{
			SequenceAttribute aSpecimenDescriptionSequence = new SequenceAttribute(DicomDictionary.StandardDictionary.getTagFromName("SpecimenDescriptionSequence"));
			list.put(aSpecimenDescriptionSequence);
			{
				AttributeList itemList = new AttributeList();
				aSpecimenDescriptionSequence.addItem(itemList);
				{ Attribute a = new LongStringAttribute(DicomDictionary.StandardDictionary.getTagFromName("SpecimenIdentifier")); a.addValue(specimenIdentifier); itemList.put(a); }	// Dummy value - should be able to override this :(
				{ Attribute a = new SequenceAttribute(DicomDictionary.StandardDictionary.getTagFromName("IssuerOfTheSpecimenIdentifierSequence")); itemList.put(a); }
				{ Attribute a = new UniqueIdentifierAttribute(DicomDictionary.StandardDictionary.getTagFromName("SpecimenUID")); a.addValue(specimenUID); itemList.put(a); }
				{ Attribute a = new SequenceAttribute(DicomDictionary.StandardDictionary.getTagFromName("SpecimenPreparationSequence")); itemList.put(a); }						// Would be nice to be able to populate this :(
			}
		}

		// Whole Slide Microscopy Image Module
		
		{ Attribute a = new CodeStringAttribute(TagFromName.ImageType); a.addValue("DERIVED"); a.addValue("PRIMARY"); a.addValue(imageFlavor); a.addValue(imageDerivation); list.put(a); }	// override default from CommonConvertedAttributeGeneration; same as FrameType

		{ Attribute a = new FloatSingleAttribute(DicomDictionary.StandardDictionary.getTagFromName("ImagedVolumeWidth"));  a.addValue(imageWidth*mmPerPixel); list.put(a); }
		{ Attribute a = new FloatSingleAttribute(DicomDictionary.StandardDictionary.getTagFromName("ImagedVolumeHeight")); a.addValue(imageLength*mmPerPixel); list.put(a); }
		{ Attribute a = new FloatSingleAttribute(DicomDictionary.StandardDictionary.getTagFromName("ImagedVolumeDepth"));  a.addValue(0); list.put(a); }							// No way of determining this :(

		{ Attribute a = new UnsignedLongAttribute(DicomDictionary.StandardDictionary.getTagFromName("TotalPixelMatrixColumns")); a.addValue(imageWidth);  list.put(a); }
		{ Attribute a = new UnsignedLongAttribute(DicomDictionary.StandardDictionary.getTagFromName("TotalPixelMatrixRows")); a.addValue(imageLength); list.put(a); }
		{ Attribute a = new UnsignedLongAttribute(DicomDictionary.StandardDictionary.getTagFromName("TotalPixelMatrixFocalPlanes")); a.addValue(1); list.put(a); }
		
		addTotalPixelMatrixOriginSequence(list,xOffsetInSlideCoordinateSystem,yOffsetInSlideCoordinateSystem);
		
		// assume slide on its side with label on left, which seems to be what Aperio, Hamamatsu, AIDPATH are
		{ Attribute a = new DecimalStringAttribute(TagFromName.ImageOrientationSlide); a.addValue(0.0); a.addValue(-1.0); a.addValue(0.0); a.addValue(-1.0); a.addValue(0.0); a.addValue(0.0); list.put(a); }
		{ Attribute a = new DateTimeAttribute(TagFromName.AcquisitionDateTime); list.put(a); }							// No way of determining this :(
		// AcquisitionDuration is optional after CP 1821
		
		// Lossy Image Compression - handled later by insertLossyImageCompressionHistory
		// Lossy Image Compression Ratio
		// Lossy Image Compression Method
		
		{ Attribute a = new CodeStringAttribute(TagFromName.VolumetricProperties); a.addValue("VOLUME"); list.put(a); }
		{ Attribute a = new CodeStringAttribute(DicomDictionary.StandardDictionary.getTagFromName("SpecimenLabelInImage")); a.addValue("NO"); list.put(a); }		// No way of determining this and most not :(
		{ Attribute a = new CodeStringAttribute(DicomDictionary.StandardDictionary.getTagFromName("FocusMethod")); a.addValue("AUTO"); list.put(a); }			// No way of determining this and most are :(
		{ Attribute a = new CodeStringAttribute(DicomDictionary.StandardDictionary.getTagFromName("ExtendedDepthOfField")); a.addValue("NO"); list.put(a); }		// No way of determining this and most not :(
		// NumberOfFocalPlanes - not need if ExtendedDepthOfField NO
		// DistanceBetweenFocalPlanes - not need if ExtendedDepthOfField NO
		// AcquisitionDeviceProcessingDescription - Type 3
		// ConvolutionKernel - Type 3
		{ Attribute a = new UnsignedShortAttribute(DicomDictionary.StandardDictionary.getTagFromName("RecommendedAbsentPixelCIELabValue")); a.addValue(0xFFFF); a.addValue(0); a.addValue(0); list.put(a); }		// white (0xFFFF is 100 per PS3.3 C.10.7.1.1)

		// Optical Path Module

		{ Attribute a = new UnsignedLongAttribute(DicomDictionary.StandardDictionary.getTagFromName("NumberOfOpticalPaths")); a.addValue(1); list.put(a); }
		{
			SequenceAttribute aOpticalPathSequence = new SequenceAttribute(DicomDictionary.StandardDictionary.getTagFromName("OpticalPathSequence"));
			list.put(aOpticalPathSequence);
			{
				AttributeList opticalPathSequenceItemList = new AttributeList();
				aOpticalPathSequence.addItem(opticalPathSequenceItemList);
				{ Attribute a = new ShortStringAttribute(DicomDictionary.StandardDictionary.getTagFromName("OpticalPathIdentifier")); a.addValue("1"); opticalPathSequenceItemList.put(a); }
				CodedSequenceItem.putSingleCodedSequenceItem(opticalPathSequenceItemList,DicomDictionary.StandardDictionary.getTagFromName("IlluminationColorCodeSequence"),"414298005","SCT","Full Spectrum");
				CodedSequenceItem.putSingleCodedSequenceItem(opticalPathSequenceItemList,DicomDictionary.StandardDictionary.getTagFromName("IlluminationTypeCodeSequence"),"111744","DCM","Brightfield illumination");
				// ICCProfile and ColorSpace are added later
				// ObjectiveLensPower could be supplied later in which case this default will be overridden but may be reapplied later
				if (objectiveLensPower != 0) {	// (001270)
					{ Attribute a = new DecimalStringAttribute(DicomDictionary.StandardDictionary.getTagFromName("ObjectiveLensPower")); a.addValue(objectiveLensPower); opticalPathSequenceItemList.put(a); }
				}
			}
		}

		// Multi-Resolution Navigation Module
		// Slide Label Module

		return list;
	}
	
	// reuse same private group and creator as for com.pixelmed.dicom.PrivatePixelData
	private static final String pixelmedPrivateCreatorForPyramidData = "PixelMed Publishing";
	private static final int pixelmedPrivatePyramidDataGroup = 0x7FDF;	// Must be BEFORE (7FE0,0010) because we assume elsewhere that DataSetTrailingPadding will immediately follow (7FE0,0010)
	private static final AttributeTag pixelmedPrivatePyramidDataBlockReservation = new AttributeTag(pixelmedPrivatePyramidDataGroup,0x0010);
	private static final AttributeTag pixelmedPrivatePyramidData = new AttributeTag(pixelmedPrivatePyramidDataGroup,0x1001);
	
	private void queueTemporaryPixelDataFilesForDeletion(Attribute aPixelData) {
		if (aPixelData != null) {
			File[] frameFiles = null;
			if (aPixelData instanceof OtherByteAttributeMultipleCompressedFrames) {
				frameFiles = ((OtherByteAttributeMultipleCompressedFrames)aPixelData).getFiles();
			}
			else if (aPixelData instanceof OtherByteAttributeMultipleFilesOnDisk) {
				frameFiles = ((OtherByteAttributeMultipleFilesOnDisk)aPixelData).getFiles();
			}
			else if (aPixelData instanceof OtherWordAttributeMultipleFilesOnDisk) {
				frameFiles = ((OtherWordAttributeMultipleFilesOnDisk)aPixelData).getFiles();
			}
			if (frameFiles != null) {
				if (filesToDeleteAfterWritingDicomFile == null) {
					filesToDeleteAfterWritingDicomFile = new ArrayList<>(Arrays.asList(frameFiles));	// make a copy because Arrays.asList() is documented not to support any adding elements "https://stackoverflow.com/questions/5755477/java-list-add-unsupportedoperationexception"
				}
				else {
					Collections.addAll(filesToDeleteAfterWritingDicomFile,frameFiles);
				}
			}
		}
	}

	private int generateDICOMPyramidPixelDataModule(AttributeList list,String outputformat,String transferSyntax) throws DicomException, IOException {
		int numberOfPyramidLevels = 1;
		{ Attribute a = new LongStringAttribute(pixelmedPrivatePyramidDataBlockReservation); a.addValue(pixelmedPrivateCreatorForPyramidData); list.put(a); }
		SequenceAttribute pyramidData = new SequenceAttribute(pixelmedPrivatePyramidData);
		list.put(pyramidData);
		
		boolean isFirstList = true;
		AttributeList oldList = list;
		while (true) {
			TiledFramesIndex index = new TiledFramesIndex(oldList,true/*physical*/,false/*buildInverseIndex*/,true/*ignorePlanePosition*/);
			int numberOfColumnsOfTiles = index.getNumberOfColumnsOfTiles();
			int numberOfRowsOfTiles = index.getNumberOfRowsOfTiles();
			if (numberOfColumnsOfTiles <= 1 && numberOfRowsOfTiles <= 1) break;
			++numberOfPyramidLevels;
			slf4jlogger.debug("generateDICOMPyramidPixelDataModule(): downsampling from numberOfColumnsOfTiles = {}, numberOfRowsOfTiles = {}",numberOfColumnsOfTiles,numberOfRowsOfTiles);
			AttributeList newList = new AttributeList();
			if (!isFirstList) {
				Attribute a = new UniqueIdentifierAttribute(TagFromName.TransferSyntaxUID); a.addValue(transferSyntax); oldList.put(a);	// need this or won't decompress
			}
			TiledPyramid.createDownsampledDICOMAttributes(oldList,newList,index,outputformat,true/*populateunchangedimagepixeldescriptionmacroattributes*/,false/*populatefunctionalgroups*/);
			if (!isFirstList) {
				oldList.remove(TagFromName.TransferSyntaxUID);	// need to remove it again since not allowed anywhere except meta header, which will be added later
			}
			pyramidData.addItem(newList);
			queueTemporaryPixelDataFilesForDeletion(newList.get(TagFromName.PixelData));	// PixelData in newList will use files that need to be deleted after writing
			oldList = newList;
			isFirstList = false;
		}
		return numberOfPyramidLevels;
	}
	
	private String mergeImageDescription(String[] description) {
		StringBuffer buf = new StringBuffer();
		if (description != null && description.length > 0) {
			slf4jlogger.debug("mergeImageDescription(): description.length = {}",description.length);
			for (String d : description) {
				if (buf.length() > 0) {
					buf.append("\n");
				}
				buf.append(d);
			}
		}
		return buf.toString();
	}
	
	private void parseTIFFImageDescription(String[] description,AttributeList descriptionList,AttributeList commonDescriptionList) throws DicomException {
		AttributeList list = new AttributeList();
		String manufacturer = "";
		String manufacturerModelName = "";
		String softwareVersions = "";
		String deviceSerialNumber = "";
		String date = "";
		String time = "";
		if (description != null && description.length > 0) {
			slf4jlogger.debug("parseTIFFImageDescription(): description.length = {}",description.length);
			for (String d : description) {
				slf4jlogger.debug("parseTIFFImageDescription(): String = {}",d);
				
				// need to check XML first, since string "Aperio" may appear in XML
				if (d.startsWith("<?xml")) {	// (001285)
					slf4jlogger.debug("parseTIFFImageDescription(): Parsing OME-TIFF XML metadata");
					try {
						Document document = DocumentBuilderFactory.newInstance().newDocumentBuilder().parse(new InputSource(new StringReader(d)));
						XPathFactory xpf = XPathFactory.newInstance();
						// OME/Instrument/Microscope@  Model="Aperio AT2" Manufacturer="Leica Biosystems"
						manufacturer = xpf.newXPath().evaluate("/OME/Instrument/Microscope/@Manufacturer",document);
						manufacturerModelName = xpf.newXPath().evaluate("/OME/Instrument/Microscope/@Model",document);
						slf4jlogger.debug("parseTIFFImageDescription(): found manufacturer {}",manufacturer);
						slf4jlogger.debug("parseTIFFImageDescription(): found manufacturerModelName {}",manufacturerModelName);
					}
					catch (Exception e) {
						slf4jlogger.error("Failed to parse OME-TIFF XML metadata in ImageDescription ",e);
					}
				}
				else if (d.contains("Aperio")) {
					manufacturer = "Leica Biosystems";
					manufacturerModelName = "Aperio";	// in absence of more specific information about whether AT2 (DX), CS2, GT450 (DX)
					slf4jlogger.debug("parseTIFFImageDescription(): found manufacturer {}",manufacturer);
					slf4jlogger.debug("parseTIFFImageDescription(): found manufacturerModelName {}",manufacturerModelName);

					// Aperio Image Library v10.0.51
					// 46920x33014 [0,100 46000x32914] (256x256) JPEG/RGB Q=30|AppMag = 20|StripeWidth = 2040|ScanScope ID = CPAPERIOCS|Filename = CMU-1|Date = 12/29/09|Time = 09:59:15|User = b414003d-95c6-48b0-9369-8010ed517ba7|Parmset = USM Filter|MPP = 0.4990|Left = 25.691574|Top = 23.449873|LineCameraSkew = -0.000424|LineAreaXOffset = 0.019265|LineAreaYOffset = -0.000313|Focus Offset = 0.000000|ImageID = 1004486|OriginalWidth = 46920|Originalheight = 33014|Filtered = 5|ICC Profile = ScanScope v1

					// Aperio Image Library v10.0.51
					// 46000x32914 -> 1024x732 - |AppMag = 20|StripeWidth = 2040|ScanScope ID = CPAPERIOCS|Filename = CMU-1|Date = 12/29/09|Time = 09:59:15|User = b414003d-95c6-48b0-9369-8010ed517ba7|Parmset = USM Filter|MPP = 0.4990|Left = 25.691574|Top = 23.449873|LineCameraSkew = -0.000424|LineAreaXOffset = 0.019265|LineAreaYOffset = -0.000313|Focus Offset = 0.000000|ImageID = 1004486|OriginalWidth = 46920|Originalheight = 33014|Filtered = 5|ICC Profile = ScanScope v1

					// Aperio Image Library v10.0.51
					// 46920x33014 [0,100 46000x32914] (256x256) -> 11500x8228 JPEG/RGB Q=65
				
					// Aperio Image Library v11.2.1
					// 46000x32914 [0,0 46000x32893] (240x240) J2K/KDU Q=30;CMU-1;Aperio Image Library v10.0.51
					// 46920x33014 [0,100 46000x32914] (256x256) JPEG/RGB Q=30|AppMag = 20|StripeWidth = 2040|ScanScope ID = CPAPERIOCS|Filename = CMU-1|Date = 12/29/09|Time = 09:59:15|User = b414003d-95c6-48b0-9369-8010ed517ba7|Parmset = USM Filter|MPP = 0.4990|Left = 25.691574|Top = 23.449873|LineCameraSkew = -0.000424|LineAreaXOffset = 0.019265|LineAreaYOffset = -0.000313|Focus Offset = 0.000000|ImageID = 1004486|OriginalWidth = 46920|Originalheight = 33014|Filtered = 5|OriginalWidth = 46000|OriginalHeight = 32914

					// Aperio Image Library v11.2.1
					// 46000x32893 -> 1024x732 - ;CMU-1;Aperio Image Library v10.0.51
					// 46920x33014 [0,100 46000x32914] (256x256) JPEG/RGB Q=30|AppMag = 20|StripeWidth = 2040|ScanScope ID = CPAPERIOCS|Filename = CMU-1|Date = 12/29/09|Time = 09:59:15|User = b414003d-95c6-48b0-9369-8010ed517ba7|Parmset = USM Filter|MPP = 0.4990|Left = 25.691574|Top = 23.449873|LineCameraSkew = -0.000424|LineAreaXOffset = 0.019265|LineAreaYOffset = -0.000313|Focus Offset = 0.000000|ImageID = 1004486|OriginalWidth = 46920|Originalheight = 33014|Filtered = 5|OriginalWidth = 46000|OriginalHeight = 32914

					// Aperio Image Library v11.2.1
					// macro 1280x431
					
					// Aperio Image Library v11.0.37
					// 29600x42592 (256x256) J2K/KDU Q=70;BioImagene iScan|Scanner ID = BI10N0294|AppMag = 20|MPP = 0.46500
					
					// Aperio Leica Biosystems GT450 v1.0.1
					// 152855x79623 [0,0,152855x79623] (256x256) JPEG/YCC Q=91|AppMag = 40|Date = 07/22/2020|Exposure Scale = 0.000001|Exposure Time = 8|Filtered = 3|Focus Offset = 0.089996|Gamma = 2.2|Left = 6.8904371261597|MPP = 0.263592|Rack = 1|ScanScope ID = SS45002|Slide = 1|StripeWidth = 4096|Time = 08:34:09|Time Zone = GMT+0100|Top = 23.217206954956

					try {
						BufferedReader r = new BufferedReader(new StringReader(d));
						String line = null;
						while ((line=r.readLine()) != null) {
							{
								// |Date = 12/29/09|
								Pattern p = Pattern.compile(".*[|]Date[ ]*=[ ]*([0-9][0-9])/([0-9][0-9])/([0-9][0-9])[|].*");
								Matcher m = p.matcher(line);
								if (m.matches()) {
									slf4jlogger.debug("parseTIFFImageDescription(): have date match");
									int groupCount = m.groupCount();
									if (groupCount == 3) {
										slf4jlogger.debug("parseTIFFImageDescription(): have date correct groupCount");
										String month = m.group(1);
										String day = m.group(2);
										String twodigityear = m.group(3);
										date = "20" + twodigityear + month + day;
										slf4jlogger.debug("parseTIFFImageDescription(): found date {}",date);
									}
								}
							}
							{
								// |Time = 09:59:15|
								Pattern p = Pattern.compile(".*[|]Time[ ]*=[ ]*([0-9][0-9]):([0-9][0-9]):([0-9][0-9])[|].*");
								Matcher m = p.matcher(line);
								if (m.matches()) {
									slf4jlogger.debug("parseTIFFImageDescription(): have time match");
									int groupCount = m.groupCount();
									if (groupCount == 3) {
										slf4jlogger.debug("parseTIFFImageDescription(): have time correct groupCount");
										String hh = m.group(1);
										String mm = m.group(2);
										String ss = m.group(3);
										time = hh + mm + ss;
										slf4jlogger.debug("parseTIFFImageDescription(): found time {}",time);
									}
								}
							}
							{
								// Aperio Image Library v10.0.51
								// Aperio Image Library vFS90 01
								Pattern p = Pattern.compile(".*Aperio Image Library (v[A-Z0-9][A-Z0-9. ]*[0-9]).*");
								Matcher m = p.matcher(line);
								if (m.matches()) {
									slf4jlogger.debug("parseTIFFImageDescription(): have Aperio Image Library match");
									int groupCount = m.groupCount();
									if (groupCount == 1) {
										slf4jlogger.debug("parseTIFFImageDescription(): have Aperio Image Library correct groupCount");
										softwareVersions = m.group(1);
										slf4jlogger.debug("parseTIFFImageDescription(): found softwareVersions (Aperio Image Library) {}",softwareVersions);
									}
								}
							}
							{
								// Aperio Leica Biosystems GT450 v1.0.1
								Pattern p = Pattern.compile(".*Aperio Leica Biosystems GT450 (v[0-9][0-9.]*[0-9]).*");
								Matcher m = p.matcher(line);
								if (m.matches()) {
									slf4jlogger.debug("parseTIFFImageDescription(): have Aperio Leica Biosystems GT450 match");
									manufacturerModelName = "GT450";
									int groupCount = m.groupCount();
									if (groupCount == 1) {
										slf4jlogger.debug("parseTIFFImageDescription(): have Aperio Leica Biosystems GT450 correct groupCount");
										softwareVersions = m.group(1);
										slf4jlogger.debug("parseTIFFImageDescription(): found softwareVersions (Aperio Leica Biosystems GT450) {}",softwareVersions);
									}
								}
							}
							{
								// |ScanScope ID = CPAPERIOCS|
								// |ScanScope ID = SS1302|
								Pattern p = Pattern.compile(".*[|]ScanScope ID[ ]*=[ ]*([^|]*)[|].*");
								Matcher m = p.matcher(line);
								if (m.matches()) {
									slf4jlogger.debug("parseTIFFImageDescription(): have ScanScope ID match");
									int groupCount = m.groupCount();
									if (groupCount == 1) {
										slf4jlogger.debug("parseTIFFImageDescription(): have ScanScope ID correct groupCount");
										deviceSerialNumber = m.group(1);
										slf4jlogger.debug("parseTIFFImageDescription(): found deviceSerialNumber (ScanScope ID) {}",deviceSerialNumber);
									}
								}
							}
							{
								// |Scanner ID = BI10N0294|
								Pattern p = Pattern.compile(".*[|]Scanner ID[ ]*=[ ]*([^|]*)[|].*");
								Matcher m = p.matcher(line);
								if (m.matches()) {
									slf4jlogger.debug("parseTIFFImageDescription(): have Scanner ID match");
									int groupCount = m.groupCount();
									if (groupCount == 1) {
										slf4jlogger.debug("parseTIFFImageDescription(): have Scanner ID correct groupCount");
										deviceSerialNumber = m.group(1);
										slf4jlogger.debug("parseTIFFImageDescription(): found deviceSerialNumber (Scanner ID) {}",deviceSerialNumber);
									}
								}
							}
							{
								// ;BioImagene iScan|
								Pattern p = Pattern.compile(".*;BioImagene iScan[|].*");
								Matcher m = p.matcher(line);
								if (m.matches()) {
									slf4jlogger.debug("parseTIFFImageDescription(): have BioImagene iScan match");
									manufacturer="BioImagene";
									manufacturerModelName="iScan";
								}
							}
						}
					}
					catch (IOException e) {
						slf4jlogger.error("Failed to parse ImageDescription ",e);
					}
				}
				else if (d.contains("X scan size")) {
					// encountered in 3D Histech uncompressed TIFF samples
					manufacturer = "3D Histech";
					slf4jlogger.debug("parseTIFFImageDescription(): guessing manufacturer {}",manufacturer);
					
					// ImageDescription: X scan size = 4.27mm
					// Y scan size = 28.90mm
					// X offset = 74.00mm
					// Y offset = 23.90mm
					// X resolution = 17067
					// Y resolution = 115600
					// Triple Simultaneous Acquisition
					// Resolution (um) = 0.25
					// Tissue Start Pixel = 40400
					// Tissue End Pixel = 108800
					// Source = Bright Field
					
					//try {
					//	BufferedReader r = new BufferedReader(new StringReader(d));
					//	String line = null;
					//	while ((line=r.readLine()) != null) {
					//	}
					//}
					//catch (IOException e) {
					//	slf4jlogger.error("Failed to parse ImageDescription ",e);
					//}
				}
				else {
					slf4jlogger.debug("parseTIFFImageDescription(): nothing recognized in ImageDescription");
				}
			}
			// common ...
			if (manufacturer.length() > 0) {
				{ Attribute a = new LongStringAttribute(TagFromName.Manufacturer); a.addValue(manufacturer); commonDescriptionList.put(a); }
			}
			if (manufacturerModelName.length() > 0) {
				{ Attribute a = new LongStringAttribute(TagFromName.ManufacturerModelName); a.addValue(manufacturerModelName); commonDescriptionList.put(a); }
			}
			if (softwareVersions.length() > 0) {
				{ Attribute a = new LongStringAttribute(TagFromName.SoftwareVersions); a.addValue(softwareVersions); commonDescriptionList.put(a); }
			}
			if (deviceSerialNumber.length() > 0) {
				{ Attribute a = new LongStringAttribute(TagFromName.DeviceSerialNumber); a.addValue(deviceSerialNumber); commonDescriptionList.put(a); }
			}
			if (date.length() > 0) {
				{ Attribute a = new DateTimeAttribute(TagFromName.AcquisitionDateTime); a.addValue(date+time); commonDescriptionList.put(a); }
				{ Attribute a = new DateAttribute(TagFromName.AcquisitionDate); a.addValue(date); commonDescriptionList.put(a); }
				{ Attribute a = new DateAttribute(TagFromName.ContentDate); a.addValue(date); commonDescriptionList.put(a); }
				{ Attribute a = new DateAttribute(TagFromName.SeriesDate); a.addValue(date); commonDescriptionList.put(a); }
				{ Attribute a = new DateAttribute(TagFromName.StudyDate); a.addValue(date); commonDescriptionList.put(a); }
			}
			if (time.length() > 0) {
				{ Attribute a = new TimeAttribute(TagFromName.AcquisitionTime); a.addValue(time); commonDescriptionList.put(a); }
				{ Attribute a = new TimeAttribute(TagFromName.ContentTime); a.addValue(time); commonDescriptionList.put(a); }
				{ Attribute a = new TimeAttribute(TagFromName.SeriesTime); a.addValue(time); commonDescriptionList.put(a); }
				{ Attribute a = new TimeAttribute(TagFromName.StudyTime); a.addValue(time); commonDescriptionList.put(a); }
			}
		}
		else {
				slf4jlogger.debug("parseTIFFImageDescription(): no ImageDescription");
		}
	}
	
	private String[][] getImageFlavorAndDerivationByIFD(ArrayList<TIFFImageFileDirectory> ifdlist) {
		int numberOfIFDs = ifdlist.size();
		String[][] imageFlavorAndDerivationByIFD = new String[numberOfIFDs][];
		boolean isOMETIFFXML = false;
		boolean isAperioSVS = false;
		int dirNum = 0;
		for (TIFFImageFileDirectory ifd : ifdlist) {
			slf4jlogger.debug("getImageFlavorAndDerivationByIFD(): Directory={}",dirNum);
			imageFlavorAndDerivationByIFD[dirNum] = new String[2];
			imageFlavorAndDerivationByIFD[dirNum][0] = "VOLUME";	// default flavor if not otherwise recognized
			imageFlavorAndDerivationByIFD[dirNum][1] = "NONE";		// default derivation if not otherwise recognized
			if (dirNum == 0) {
				String[] description = ifd.getStringValues(TIFFTags.IMAGEDESCRIPTION);
				if (description != null && description.length > 0) {
					slf4jlogger.debug("getImageFlavorAndDerivationByIFD(): description.length = {}",description.length);
					for (String d : description) {
						slf4jlogger.debug("getImageFlavorAndDerivationByIFD(): String = {}",d);
						// need to check XML first, since string "Aperio" may appear in XML
						if (d.startsWith("<?xml")) {	// (001285)
							isOMETIFFXML = true;
						}
						else if (d.contains("Aperio")) {
							isAperioSVS = true;			// assume SVS but could be other Aperio format, theoretically ? :(
						}
					}
				}
				slf4jlogger.debug("getImageFlavorAndDerivationByIFD(): isOMETIFFXML={}",isOMETIFFXML);
				slf4jlogger.debug("getImageFlavorAndDerivationByIFD(): isAperioSVS={}",isAperioSVS);
			}
			if (isOMETIFFXML) {
				// do nothing special ... assume defaults are OK
			}
			else if (isAperioSVS) {
				// SVS layout per "MAN-0069_RevB_Digital_Slides_and_Third-party_data_interchange.pdf":
				//
				// "The first image in an SVS file is always the baseline image (full resolution). This image is always tiled, usually with a tile size of 240 x 240 pixels.
				// The second image is always a thumbnail, typically with dimensions of about 1024 x 768 pixels. Unlike the other slide images, the thumbnail image is always stripped.
				// Following the thumbnail there may be one or more intermediate “pyramid” images. These are always compressed with the same type of compression as the baseline image, and have a tiled organization with the same tile size.
				// Optionally at the end of an SVS file there may be a slide label image, which is a low resolution picture taken of the slide’s label,
				// and/or a macro camera image, which is a low resolution picture taken of the entire slide.
				// The label and macro images are always stripped. If present the label image is compressed with LZW compression, and the macro image with JPEG compression."
				//
				// "The intermediate resolution images in an SVS file are typically 1/4th resolution of the previous image"

				boolean isMacro = false;
				boolean isLabel = false;
				{
					String[] description = ifd.getStringValues(TIFFTags.IMAGEDESCRIPTION);
					if (description != null && description.length > 0) {
						slf4jlogger.debug("getImageFlavorAndDerivationByIFD(): description.length = {}",description.length);
						for (String d : description) {
							slf4jlogger.debug("getImageFlavorAndDerivationByIFD(): String = {}",d);
							if (d.contains("macro")) {
								isMacro = true;
							}
							else if (d.contains("label")) {
								isLabel = true;
							}
						}
					}
				}
				
				if (isMacro) {
					slf4jlogger.debug("getImageFlavorAndDerivationByIFD(): is macro based on IMAGEDESCRIPTION so using OVERVIEW flavor");
					imageFlavorAndDerivationByIFD[dirNum][0] = "OVERVIEW";
					imageFlavorAndDerivationByIFD[dirNum][1] = "NONE";
				}
				else if (isLabel) {
					slf4jlogger.debug("getImageFlavorAndDerivationByIFD(): is label based on IMAGEDESCRIPTION so using LABEL flavor");
					imageFlavorAndDerivationByIFD[dirNum][0] = "LABEL";
					imageFlavorAndDerivationByIFD[dirNum][1] = "NONE";
				}
				else {
					long[] tileOffsets = ifd.getNumericValues(TIFFTags.TILEOFFSETS);
					boolean isTiled = tileOffsets != null;
					
					if (isTiled) {
						slf4jlogger.debug("getImageFlavorAndDerivationByIFD(): is tiled so using VOLUME flavor");
						imageFlavorAndDerivationByIFD[dirNum][0] = "VOLUME";
						if (dirNum == 0) {
							imageFlavorAndDerivationByIFD[dirNum][1] = "NONE";
						}
						else {
							imageFlavorAndDerivationByIFD[dirNum][1] = "RESAMPLED";		// (001258)
						}
					}
					else {
						if (dirNum == 1) {
							slf4jlogger.debug("getImageFlavorAndDerivationByIFD(): is not tiled and 2nd IFD entry so using THUMBNAIL flavor");
							imageFlavorAndDerivationByIFD[dirNum][0] = "THUMBNAIL";		// not a DICOM standard defined term (yet), but want to distinguish from VOLUME and OVERVIEW, and not use LOCALIZER (as Lecia does)
							imageFlavorAndDerivationByIFD[dirNum][1] = "RESAMPLED";		// (001258)
						}
						else if (dirNum == (numberOfIFDs-1) || dirNum == (numberOfIFDs-2)) {
							long compression = ifd.getSingleNumericValue(TIFFTags.COMPRESSION,0,0);
							slf4jlogger.debug("getImageFlavorAndDerivationByIFD(): compression={}",compression);
					
							if (compression == 5) {			// LZW
								slf4jlogger.debug("getImageFlavorAndDerivationByIFD(): is not tiled and last or 2nd last IFD entry and LZW so using THUMBNAIL flavor");
								imageFlavorAndDerivationByIFD[dirNum][0] = "LABEL";
								imageFlavorAndDerivationByIFD[dirNum][1] = "NONE";
							}
							else if (compression == 7) {	// new JPEG
								slf4jlogger.debug("getImageFlavorAndDerivationByIFD(): is not tiled and last or 2nd last IFD entry and JPEG so using OVERVIEW flavor");
								imageFlavorAndDerivationByIFD[dirNum][0] = "OVERVIEW";
								imageFlavorAndDerivationByIFD[dirNum][1] = "NONE";
							}
						}
					}
				}
			}
			slf4jlogger.debug("getImageFlavorAndDerivationByIFD(): imageFlavorAndDerivationByIFD[{}] flavor={}",dirNum,imageFlavorAndDerivationByIFD[dirNum][0]);
			slf4jlogger.debug("getImageFlavorAndDerivationByIFD(): imageFlavorAndDerivationByIFD[{}] derivation={}",dirNum,imageFlavorAndDerivationByIFD[dirNum][1]);
			++dirNum;
		}
		return imageFlavorAndDerivationByIFD;
	}
	
	private String chooseTransferSyntaxForCompressionSchemeIfNotSpecifiedExplicitly(String transferSyntax,long compression) throws TIFFException {
		slf4jlogger.debug("chooseTransferSyntaxForCompressionSchemeIfNotSpecifiedExplicitly(): transferSyntax specified as = {}",transferSyntax);
		slf4jlogger.debug("chooseTransferSyntaxForCompressionSchemeIfNotSpecifiedExplicitly(): compression = {}",compression);
		if (transferSyntax == null || transferSyntax.length() == 0) {
			if (compression == 0 || compression == 1) {		// absent or specified as uncompressed
				transferSyntax = TransferSyntax.ExplicitVRLittleEndian;
			}
			else if (compression == 7) {		// "new" JPEG per TTN2 as used by Aperio in SVS
				// really should check what is in there ... could be lossless, or 12 bit per TTN2 :(
				transferSyntax = TransferSyntax.JPEGBaseline;
			}
			else if (compression == 33003 || compression == 33005) {	// Aperio J2K YCbCr or RGB
				transferSyntax = TransferSyntax.JPEG2000;
			}
			else {
				throw new TIFFException("Unsupported compression = "+compression);
			}
		}
		slf4jlogger.debug("chooseTransferSyntaxForCompressionSchemeIfNotSpecifiedExplicitly(): transferSyntax now = {}",transferSyntax);
		return transferSyntax;
	}

	private String chooseRecompressAsFormatFromTransferSyntax(String transferSyntax) throws TIFFException {
		//slf4jlogger.debug("chooseTransferSyntaxForCompressionSchemeIfNotSpecifiedExplicitly(): transferSyntax = {}",transferSyntax);
		String recompressAsFormat = null;
		if (transferSyntax != null && transferSyntax.length() > 0) {
			recompressAsFormat = CompressedFrameEncoder.chooseOutputFormatForTransferSyntax(transferSyntax);
		}
		slf4jlogger.debug("chooseRecompressAsFormatFromTransferSyntax(): recompressAsFormat = {}",recompressAsFormat);
		return recompressAsFormat;
	}

	private long chooseOutputCompressionForRecompressAsFormatGivenInputCompression(String recompressAsFormat,long compression) throws TIFFException {
		slf4jlogger.debug("chooseOutputCompressionForRecompressAsFormatGivenInputCompression(): recompressAsFormat = {}",recompressAsFormat);
		slf4jlogger.debug("chooseOutputCompressionForRecompressAsFormatGivenInputCompression(): compression = {}",compression);
		long outputCompression = compression;	// default is same as input
		if (recompressAsFormat == null || recompressAsFormat.length() == 0) {
			outputCompression = 1;
		}
		else {
			if (recompressAsFormat.equals("jpeg")) {
				outputCompression = 7;
			}
			else if (recompressAsFormat.equals("jpeg2000")) {
				if (compression != 33003 && compression != 33005) {
					outputCompression = 33003;		// if recompressing, and need to choose something
				}
			}
		}
		slf4jlogger.debug("chooseOutputCompressionForRecompressAsFormatGivenInputCompression(): outputCompression = {}",outputCompression);
		return outputCompression;
	}

	private void createOrAppendToManufacturerModelNameAndInsertOrReplace(AttributeList list) throws DicomException {
		String manufacturerModelName = Attribute.getSingleStringValueOrEmptyString(list,TagFromName.ManufacturerModelName);
		if (manufacturerModelName.length() > 0) {
			manufacturerModelName = manufacturerModelName + " converted by ";
		}
		manufacturerModelName = manufacturerModelName + this.getClass().getName();
		{ Attribute a = new LongStringAttribute(TagFromName.ManufacturerModelName); a.addValue(manufacturerModelName); list.put(a); }
	}

	private void createOrAppendToSoftwareVersionsAndInsertOrReplace(AttributeList list) throws DicomException {
		Attribute a = list.get(TagFromName.SoftwareVersions);
		if (a == null) {
			a = new LongStringAttribute(TagFromName.SoftwareVersions);
			list.put(a);
		}
		a.addValue(VersionAndConstants.getBuildDate());
	}

	private void addContributingEquipmentSequence(AttributeList list) throws DicomException {
		ClinicalTrialsAttributes.addContributingEquipmentSequence(list,true,new CodedSequenceItem("109103","DCM","Modifying Equipment"),
																  "PixelMed",													// Manufacturer
																  "PixelMed",													// Institution Name
																  "Software Development",										// Institutional Department Name
																  "Bangor, PA",													// Institution Address
																  null,															// Station Name
																  this.getClass().getName(),									// Manufacturer's Model Name
																  null,															// Device Serial Number
																  "Vers. "+VersionAndConstants.getBuildDate(),					// Software Version(s)
																  "TIFF to DICOM conversion");
	}

	private void convertTIFFTilesToDicomMultiFrame(String jsonfile,TIFFFile inputFile,String outputFileName,int instanceNumber,
				long imageWidth,long imageLength,
				long[] tileOffsets,long[] tileByteCounts,long tileWidth,long tileLength,
				long bitsPerSample,long compression,byte[] jpegTables,byte[] iccProfile,long photometric,long samplesPerPixel,long planarConfig,long sampleFormat,
				String frameOfReferenceUID,double mmPerPixel,double objectiveLensPower,
				String opticalPathIdentifier,String opticalPathDescription,
				double xOffsetInSlideCoordinateSystem,double yOffsetInSlideCoordinateSystem,
				String modality,String sopClass,String transferSyntax,
				String containerIdentifier,String specimenIdentifier,String specimenUID,
				String imageFlavor,String imageDerivation,String imageDescription,AttributeList descriptionList,
				boolean addTIFF,boolean useBigTIFF,boolean addPyramid) throws IOException, DicomException, TIFFException {

		slf4jlogger.debug("convertTIFFTilesToDicomMultiFrame(): instanceNumber = {}",instanceNumber);
		slf4jlogger.debug("convertTIFFTilesToDicomMultiFrame(): transferSyntax supplied = {}",transferSyntax);
		slf4jlogger.debug("convertTIFFTilesToDicomMultiFrame(): photometric in TIFF file = {}",photometric);

		transferSyntax = chooseTransferSyntaxForCompressionSchemeIfNotSpecifiedExplicitly(transferSyntax,compression);
		String recompressAsFormat = chooseRecompressAsFormatFromTransferSyntax(transferSyntax);
		boolean recompressLossy = new TransferSyntax(transferSyntax).isLossy();
		slf4jlogger.debug("convertTIFFTilesToDicomMultiFrame(): recompressLossy = {}",recompressLossy);

		AttributeList list = new AttributeList();
		
		int numberOfTiles = tileOffsets.length;
		long outputPhotometric = generateDICOMPixelDataMultiFrameImageFromTIFFFile(inputFile,list,numberOfTiles,tileOffsets,tileByteCounts,tileWidth,tileLength,bitsPerSample,compression,photometric,jpegTables,iccProfile,recompressAsFormat,recompressLossy);
		long outputCompression = chooseOutputCompressionForRecompressAsFormatGivenInputCompression(recompressAsFormat,compression);

		slf4jlogger.debug("convertTIFFTilesToDicomMultiFrame(): photometric {}changed from {} to {}",(photometric == outputPhotometric ? "un" : ""),photometric,outputPhotometric);
		slf4jlogger.debug("convertTIFFTilesToDicomMultiFrame(): compression {}changed from {} to {}",(compression == outputCompression ? "un" : ""),compression,outputCompression);

		generateDICOMPixelDataModuleAttributes(list,numberOfTiles,tileWidth,tileLength,bitsPerSample,outputCompression,outputPhotometric,samplesPerPixel,planarConfig,sampleFormat,recompressAsFormat,recompressLossy,sopClass);
		
		CommonConvertedAttributeGeneration.generateCommonAttributes(list,""/*patientName*/,""/*patientID*/,""/*studyID*/,""/*seriesNumber*/,Integer.toString(instanceNumber),modality,sopClass,false/*generateUnassignedConverted*/);
		list.remove(TagFromName.SoftwareVersions);	// will set later - do not want default from CommonConvertedAttributeGeneration.generateCommonAttributes
		
		if (SOPClass.VLWholeSlideMicroscopyImageStorage.equals(sopClass)) {
			generateDICOMWholeSlideMicroscopyImageAttributes(list,imageWidth,imageLength,frameOfReferenceUID,mmPerPixel,objectiveLensPower,opticalPathIdentifier,opticalPathDescription,xOffsetInSlideCoordinateSystem,yOffsetInSlideCoordinateSystem,containerIdentifier,specimenIdentifier,specimenUID,imageFlavor,imageDerivation);
		}

		insertLossyImageCompressionHistory(list,compression,outputCompression,totalArrayValues(tileByteCounts),imageWidth,imageLength,bitsPerSample,samplesPerPixel);

		if (imageDescription != null && imageDescription.length() > 0) {
			if (imageDescription.length() <= 10240) {	// (001281)
				{ Attribute a = new LongTextAttribute(TagFromName.ImageComments); a.addValue(imageDescription); list.put(a); }
			}
			else {
				{ Attribute a = new UnlimitedTextAttribute(TagFromName.TextValue); a.addValue(imageDescription); list.put(a); }
			}
		}

		if (descriptionList != null) {
			// override such things as Manufacturer, UIDs, dates, times, TotalPixelMatrixOriginSequence, if they were obtained from the ImageDescription TIFF tag or need to be common
			list.putAll(descriptionList);
		}

		createOrAppendToManufacturerModelNameAndInsertOrReplace(list);
		createOrAppendToSoftwareVersionsAndInsertOrReplace(list);

		addContributingEquipmentSequence(list);

		new SetCharacteristicsFromSummary(jsonfile,list);

		// only now add ICC profile, so as not be overriden by any OpticalPathSequence in SetCharacteristicsFromSummary
		if (SOPClass.VLWholeSlideMicroscopyImageStorage.equals(sopClass)) {
			if (samplesPerPixel > 1) {	// (001229)
				iccProfile = addICCProfileToOpticalPathSequence(list,iccProfile);	// adds known or default, since required
			}
		}
		else if (iccProfile != null && iccProfile.length > 0) {		// add known
			{ Attribute a = new OtherByteAttribute(TagFromName.ICCProfile); a.setValues(iccProfile); list.put(a); }
			slf4jlogger.debug("Created ICC Profile attribute of length {}",iccProfile.length);
		}
		// else do not add default ICC Profile

		// only now add ObjectiveLensPower, so as not to be overriden by any OpticalPathSequence if present without ObjectiveLensPower in any addition from SetCharacteristicsFromSummary (001270)
		if (SOPClass.VLWholeSlideMicroscopyImageStorage.equals(sopClass)) {
			addObjectiveLensPowerToOpticalPathSequence(list,objectiveLensPower);
			addOpticalPathIdentifierAndDescriptionToOpticalPathSequence(list,opticalPathIdentifier,opticalPathDescription);	// (001285)
		}

		CodingSchemeIdentification.replaceCodingSchemeIdentificationSequenceWithCodingSchemesUsedInAttributeList(list);
		
		list.insertSuitableSpecificCharacterSetForAllStringValues();	// (001158)

		FileMetaInformation.addFileMetaInformation(list,transferSyntax,"OURAETITLE");
		
		int numberOfPyramidLevels = 1;	// at a minimum the base layer in top level data set PixelData

		//boolean addPyramid = SOPClass.VLWholeSlideMicroscopyImageStorage.equals(sopClass);
		//boolean addPyramid = false;

		if (addPyramid) {
			try {
				numberOfPyramidLevels = generateDICOMPyramidPixelDataModule(list,recompressAsFormat,transferSyntax);	// will use existing PixelData attribute contents ... need to do this after TransferSyntax is set in FileMetaInformation
			}
			catch (DicomException e) {
				e.printStackTrace(System.err);
			}
		}

		byte[] preamble = null;
		
		if (addTIFF) {
			long lowerPhotometric =	// what to use when/if making lower pyramidal levels
				transferSyntax.equals(TransferSyntax.JPEGBaseline) || transferSyntax.equals(TransferSyntax.JPEG2000)
				? 6				// YCbCr, since that is what the codec will do regardless, i.e., be consistent with what TiledPyramid.createDownsampledDICOMAttributes() does
				: photometric;	// leave it the same unless we are recompressing it; is independent of whatever outputPhotometric happens to be
			slf4jlogger.debug("convertTIFFTilesToDicomMultiFrame(): lowerPhotometric = {}",lowerPhotometric);

			try {
				long[][] frameDataByteOffsets = new long[numberOfPyramidLevels][];
				long[][] frameDataLengths = new long[numberOfPyramidLevels][];
				long[] imageWidths = new long[numberOfPyramidLevels];
				long[] imageLengths = new long[numberOfPyramidLevels];
				long  byteOffsetFromFileStartOfNextAttributeAfterPixelData = AddTIFFOrOffsetTables.getByteOffsetsAndLengthsOfFrameDataFromStartOfFile(list,transferSyntax,frameDataByteOffsets,frameDataLengths,imageWidths,imageLengths);
				preamble = AddTIFFOrOffsetTables.makeTIFFInPreambleAndAddDataSetTrailingPadding(byteOffsetFromFileStartOfNextAttributeAfterPixelData,numberOfPyramidLevels,frameDataByteOffsets,frameDataLengths,imageWidths,imageLengths,list,
					tileWidth,tileLength,bitsPerSample,outputCompression,outputPhotometric,lowerPhotometric,samplesPerPixel,planarConfig,sampleFormat,iccProfile,mmPerPixel,useBigTIFF);
			}
			catch (DicomException e) {
				e.printStackTrace(System.err);
			}
		}
		
		list.write(outputFileName,transferSyntax,true,true,preamble);
		
		if (filesToDeleteAfterWritingDicomFile != null) {
			for (File tmpFile : filesToDeleteAfterWritingDicomFile) {
				tmpFile.delete();
			}
			filesToDeleteAfterWritingDicomFile = null;
		}
	}
	
	private void convertTIFFTilesToDicomSingleFrameMergingStrips(String jsonfile,TIFFFile inputFile,String outputFileName,int instanceNumber,
				long imageWidth,long imageLength,
				long[] pixelOffset,long[] pixelByteCount,long pixelWidth,long rowsPerStrip,
				long bitsPerSample,long compression,byte[] jpegTables,byte[] iccProfile,long photometric,long samplesPerPixel,long planarConfig,long sampleFormat,
				String frameOfReferenceUID,double mmPerPixel,double objectiveLensPower,
				String opticalPathIdentifier,String opticalPathDescription,
				double xOffsetInSlideCoordinateSystem,double yOffsetInSlideCoordinateSystem,
				String modality,String sopClass,String transferSyntax,
				String containerIdentifier,String specimenIdentifier,String specimenUID,
				String imageFlavor,String imageDerivation,String imageDescription,AttributeList descriptionList,
				boolean addTIFF,boolean useBigTIFF) throws IOException, DicomException, TIFFException {

		slf4jlogger.debug("convertTIFFTilesToDicomSingleFrameMergingStrips(): instanceNumber = {}",instanceNumber);

		transferSyntax = chooseTransferSyntaxForCompressionSchemeIfNotSpecifiedExplicitly(transferSyntax,compression);
		String recompressAsFormat = chooseRecompressAsFormatFromTransferSyntax(transferSyntax);
		boolean recompressLossy = new TransferSyntax(transferSyntax).isLossy();
		slf4jlogger.debug("convertTIFFTilesToDicomMultiFrame(): recompressLossy = {}",recompressLossy);

		AttributeList list = new AttributeList();
		
		long outputPhotometric = generateDICOMPixelDataSingleFrameImageFromTIFFFileMergingStrips(inputFile,list,imageWidth,imageLength,pixelOffset,pixelByteCount,pixelWidth,rowsPerStrip,bitsPerSample,compression,photometric,samplesPerPixel,jpegTables,iccProfile,recompressAsFormat,recompressLossy);
		long outputCompression = chooseOutputCompressionForRecompressAsFormatGivenInputCompression(recompressAsFormat,compression);

		slf4jlogger.debug("convertTIFFTilesToDicomSingleFrameMergingStrips(): photometric {}changed from {} to {}",(photometric == outputPhotometric ? "un" : ""),photometric,outputPhotometric);
		slf4jlogger.debug("convertTIFFTilesToDicomSingleFrameMergingStrips(): compression {}changed from {} to {}",(compression == outputCompression ? "un" : ""),compression,outputCompression);

		generateDICOMPixelDataModuleAttributes(list,1/*numberOfFrames*/,imageWidth,imageLength,bitsPerSample,outputCompression,outputPhotometric,samplesPerPixel,planarConfig,sampleFormat,recompressAsFormat,recompressLossy,sopClass);

		if (iccProfile != null && iccProfile.length > 0) {
			{ Attribute a = new OtherByteAttribute(TagFromName.ICCProfile); a.setValues(iccProfile); list.put(a); }
			slf4jlogger.debug("convertTIFFTilesToDicomSingleFrameMergingStrips(): Created ICC Profile attribute of length {}",iccProfile.length);
		}

		CommonConvertedAttributeGeneration.generateCommonAttributes(list,""/*patientName*/,""/*patientID*/,""/*studyID*/,""/*seriesNumber*/,Integer.toString(instanceNumber),modality,sopClass,false/*generateUnassignedConverted*/);
		list.remove(TagFromName.SoftwareVersions);	// will set later - do not want default from CommonConvertedAttributeGeneration.generateCommonAttributes

		if (SOPClass.VLWholeSlideMicroscopyImageStorage.equals(sopClass)) {
			generateDICOMWholeSlideMicroscopyImageAttributes(list,imageWidth,imageLength,frameOfReferenceUID,mmPerPixel,objectiveLensPower,opticalPathIdentifier,opticalPathDescription,xOffsetInSlideCoordinateSystem,yOffsetInSlideCoordinateSystem,containerIdentifier,specimenIdentifier,specimenUID,imageFlavor,imageDerivation);
		}

		insertLossyImageCompressionHistory(list,compression,outputCompression,totalArrayValues(pixelByteCount),imageWidth,imageLength,bitsPerSample,samplesPerPixel);

		if (imageDescription != null && imageDescription.length() > 0) {
			if (imageDescription.length() <= 10240) {	// (001281)
				{ Attribute a = new LongTextAttribute(TagFromName.ImageComments); a.addValue(imageDescription); list.put(a); }
			}
			else {
				{ Attribute a = new UnlimitedTextAttribute(TagFromName.TextValue); a.addValue(imageDescription); list.put(a); }
			}
		}

		if (descriptionList != null) {
			// override such things as Manufacturer, UIDs, dates, times, TotalPixelMatrixOriginSequence, if they were obtained from the ImageDescription TIFF tag or need to be common
			list.putAll(descriptionList);
		}

		createOrAppendToManufacturerModelNameAndInsertOrReplace(list);
		createOrAppendToSoftwareVersionsAndInsertOrReplace(list);
		
		addContributingEquipmentSequence(list);

		new SetCharacteristicsFromSummary(jsonfile,list);

		// only now add ICC profile, so as not be overriden by any OpticalPathSequence in SetCharacteristicsFromSummary
		if (SOPClass.VLWholeSlideMicroscopyImageStorage.equals(sopClass)) {
			if (samplesPerPixel > 1) {	// (001229)
				iccProfile = addICCProfileToOpticalPathSequence(list,iccProfile);	// adds known or default, since required
			}
		}
		else if (iccProfile != null && iccProfile.length > 0) {		// add known
			{ Attribute a = new OtherByteAttribute(TagFromName.ICCProfile); a.setValues(iccProfile); list.put(a); }
			slf4jlogger.debug("Created ICC Profile attribute of length {}",iccProfile.length);
		}
		// else do not add default ICC Profile

		// only now add ObjectiveLensPower, so as not to be overriden by any OpticalPathSequence if present without ObjectiveLensPower in any addition from SetCharacteristicsFromSummary (001270)
		if (SOPClass.VLWholeSlideMicroscopyImageStorage.equals(sopClass)) {
			addObjectiveLensPowerToOpticalPathSequence(list,objectiveLensPower);
			addOpticalPathIdentifierAndDescriptionToOpticalPathSequence(list,opticalPathIdentifier,opticalPathDescription);	// (001285)
		}

		CodingSchemeIdentification.replaceCodingSchemeIdentificationSequenceWithCodingSchemesUsedInAttributeList(list);

		list.insertSuitableSpecificCharacterSetForAllStringValues();

		FileMetaInformation.addFileMetaInformation(list,transferSyntax,"OURAETITLE");

		byte[] preamble = null;
		
		//if (addTIFF) slf4jlogger.warn("convertTIFFTilesToDicomSingleFrameMergingStrips(): Adding TIFF not yet implemented for single frame conversion");
		if (addTIFF) {
			try {
				// no pyramids
				long[][] frameDataByteOffsets = new long[1][];
				long[][] frameDataLengths = new long[1][];
				long[] imageWidths = new long[1];
				long[] imageLengths = new long[1];
				long  byteOffsetFromFileStartOfNextAttributeAfterPixelData = AddTIFFOrOffsetTables.getByteOffsetsAndLengthsOfFrameDataFromStartOfFile(list,transferSyntax,frameDataByteOffsets,frameDataLengths,imageWidths,imageLengths);
				preamble = AddTIFFOrOffsetTables.makeTIFFInPreambleAndAddDataSetTrailingPadding(byteOffsetFromFileStartOfNextAttributeAfterPixelData,1/*numberOfPyramidLevels*/,frameDataByteOffsets,frameDataLengths,imageWidths,imageLengths,list,
					imageWidth,imageLength,bitsPerSample,outputCompression,outputPhotometric,outputPhotometric/*lowerPhotometric*/,samplesPerPixel,planarConfig,sampleFormat,iccProfile,mmPerPixel,useBigTIFF);
			}
			catch (DicomException e) {
				e.printStackTrace(System.err);
			}
		}
		
		list.write(outputFileName,transferSyntax,true,true,preamble);

		if (filesToDeleteAfterWritingDicomFile != null) {
			for (File tmpFile : filesToDeleteAfterWritingDicomFile) {
				tmpFile.delete();
			}
			filesToDeleteAfterWritingDicomFile = null;
		}
	}
	
	private void convertTIFFTilesToDicomSingleFrame(String jsonfile,TIFFFile inputFile,String outputFileName,int instanceNumber,
				long imageWidth,long imageLength,
				long pixelOffset,long pixelByteCount,long pixelWidth,long pixelLength,
				long bitsPerSample,long compression,byte[] jpegTables,byte[] iccProfile,long photometric,long samplesPerPixel,long planarConfig,long sampleFormat,
				String modality,String sopClass,String transferSyntax,
				String imageFlavor,String imageDerivation,String imageDescription,AttributeList descriptionList,
				boolean addTIFF,boolean useBigTIFF) throws IOException, DicomException, TIFFException {

		slf4jlogger.debug("convertTIFFTilesToDicomSingleFrame(): instanceNumber = {}",instanceNumber);

		if (addTIFF) slf4jlogger.warn("convertTIFFTilesToDicomSingleFrame(): Adding TIFF not yet implemented for single frame conversion");

		transferSyntax = chooseTransferSyntaxForCompressionSchemeIfNotSpecifiedExplicitly(transferSyntax,compression);
		String recompressAsFormat = chooseRecompressAsFormatFromTransferSyntax(transferSyntax);
		boolean recompressLossy = new TransferSyntax(transferSyntax).isLossy();
		slf4jlogger.debug("convertTIFFTilesToDicomMultiFrame(): recompressLossy = {}",recompressLossy);

		AttributeList list = new AttributeList();
		
		long outputPhotometric = generateDICOMPixelDataSingleFrameImageFromTIFFFile(inputFile,list,pixelOffset,pixelByteCount,pixelWidth,pixelLength,bitsPerSample,compression,photometric,jpegTables,iccProfile,recompressAsFormat,recompressLossy);
		long outputCompression = chooseOutputCompressionForRecompressAsFormatGivenInputCompression(recompressAsFormat,compression);

		slf4jlogger.debug("convertTIFFTilesToDicomSingleFrame(): photometric {}changed from {} to {}",(photometric == outputPhotometric ? "un" : ""),photometric,outputPhotometric);
		slf4jlogger.debug("convertTIFFTilesToDicomSingleFrame(): compression {}changed from {} to {}",(compression == outputCompression ? "un" : ""),compression,outputCompression);

		generateDICOMPixelDataModuleAttributes(list,1/*numberOfFrames*/,pixelWidth,pixelLength,bitsPerSample,outputCompression,outputPhotometric,samplesPerPixel,planarConfig,sampleFormat,recompressAsFormat,recompressLossy,sopClass);

		if (iccProfile != null && iccProfile.length > 0) {
			{ Attribute a = new OtherByteAttribute(TagFromName.ICCProfile); a.setValues(iccProfile); list.put(a); }
			slf4jlogger.debug("Created ICC Profile attribute of length {}",iccProfile.length);
		}

		CommonConvertedAttributeGeneration.generateCommonAttributes(list,""/*patientName*/,""/*patientID*/,""/*studyID*/,""/*seriesNumber*/,Integer.toString(instanceNumber),modality,sopClass,false/*generateUnassignedConverted*/);
		list.remove(TagFromName.SoftwareVersions);	// will set later - do not want default from CommonConvertedAttributeGeneration.generateCommonAttributes

		insertLossyImageCompressionHistory(list,compression,outputCompression,pixelByteCount,imageWidth,imageLength,bitsPerSample,samplesPerPixel);

		if (imageDescription != null && imageDescription.length() > 0) {
			if (imageDescription.length() <= 10240) {	// (001281)
				{ Attribute a = new LongTextAttribute(TagFromName.ImageComments); a.addValue(imageDescription); list.put(a); }
			}
			else {
				{ Attribute a = new UnlimitedTextAttribute(TagFromName.TextValue); a.addValue(imageDescription); list.put(a); }
			}
		}

		if (descriptionList != null) {
			// override such things as Manufacturer, UIDs, dates, times, TotalPixelMatrixOriginSequence, if they were obtained from the ImageDescription TIFF tag or need to be common
			list.putAll(descriptionList);
		}

		createOrAppendToManufacturerModelNameAndInsertOrReplace(list);
		createOrAppendToSoftwareVersionsAndInsertOrReplace(list);

		addContributingEquipmentSequence(list);
		
		new SetCharacteristicsFromSummary(jsonfile,list);

		CodingSchemeIdentification.replaceCodingSchemeIdentificationSequenceWithCodingSchemesUsedInAttributeList(list);

		list.insertSuitableSpecificCharacterSetForAllStringValues();

		FileMetaInformation.addFileMetaInformation(list,transferSyntax,"OURAETITLE");
		list.write(outputFileName,transferSyntax,true,true);
		
		if (filesToDeleteAfterWritingDicomFile != null) {
			for (File tmpFile : filesToDeleteAfterWritingDicomFile) {
				tmpFile.delete();
			}
			filesToDeleteAfterWritingDicomFile = null;
		}
	}
	
	private class WSIFrameOfReference {
		String uidPyramid;
		String uidOverview;
		int dirNumOfOverview;
		double mmPerPixelBaseLayerDefault;
		double mmPerPixelBaseLayer;
		double mmPerPixelOverviewImage;
		double[] mmPerPixel;			// indexed by IFD (dirNum)
		double objectiveLensPower;
		double xOffsetInSlideCoordinateSystemPyramid;
		double yOffsetInSlideCoordinateSystemPyramid;
		double xOffsetInSlideCoordinateSystemOverview;
		double yOffsetInSlideCoordinateSystemOverview;
		String[] opticalPathIdentifier;		// indexed by IFD (dirNum)
		String[] opticalPathDescription;	// indexed by IFD (dirNum)

		double getmmPerPixelForIFD(int dirNum) {
			return mmPerPixel[dirNum];
		}
		
		double getObjectiveLensPower() {
			return objectiveLensPower;
		}
		
		double getXOffsetInSlideCoordinateSystemForIFD(int dirNum) {
			return (dirNumOfOverview != -1 && dirNumOfOverview == dirNum) ? xOffsetInSlideCoordinateSystemOverview : xOffsetInSlideCoordinateSystemPyramid;
		}
		
		double getYOffsetInSlideCoordinateSystemForIFD(int dirNum) {
			return (dirNumOfOverview != -1 && dirNumOfOverview == dirNum) ? yOffsetInSlideCoordinateSystemOverview : yOffsetInSlideCoordinateSystemPyramid;
		}
		
		String getFrameOfReferenceUIDForIFD(int dirNum) {
			return (dirNumOfOverview != -1 && dirNumOfOverview == dirNum) ? uidOverview : uidPyramid;
		}
		
		String getOpticalPathIdentifierForIFD(int dirNum) {
			return opticalPathIdentifier[dirNum];
		}
		
		String getOpticalPathDescriptionForIFD(int dirNum) {
			return opticalPathDescription[dirNum];
		}

		WSIFrameOfReference(ArrayList<TIFFImageFileDirectory> ifdlist,String[][] imageFlavorAndDerivationByIFD) throws DicomException {
			uidPyramid = u.getAnotherNewUID();
			
			dirNumOfOverview = -1;	// flag that we have not encountered an overview image yet (not 0, since 0 is a valid dirNum)
			
			mmPerPixelBaseLayerDefault = 0.5/1000;	// typically 20× (0.5 μm/pixel) and 40× (0.25 μm/pixel) - assume 20x for 1st IFD if not overriden later :(
			slf4jlogger.debug("WSIFrameOfReference(): mmPerPixelBaseLayerDefault (assuming 20x)={}",mmPerPixelBaseLayerDefault);
			
			mmPerPixelBaseLayer = 0;		// keep track of this for computing pixel spacing for lower pyramid layers - will set the value (or pick default) once we start parsing the IFDs (001265)

			mmPerPixel = new double[ifdlist.size()];

			objectiveLensPower = 0;

			opticalPathIdentifier = new String[ifdlist.size()];		// (001285)
			opticalPathDescription = new String[ifdlist.size()];

			boolean haveOverviewRelativeTop = false;
			boolean haveOverviewRelativeLeft = false;
			boolean haveOverviewPixelSpacing = false;
			
			long widthOfBaseLayerInPixels = 0;
			long widthOfOverviewInPixels = 0;
			long lengthOfOverviewInPixels = 0;	// i.e., height

			double distanceLongAxisSlideFromMacroLeftEdge = 0d;
			double distanceShortAxisSlideFromMacroBottomEdge = 0d;

			int dirNum = 0;
			for (TIFFImageFileDirectory ifd : ifdlist) {
				slf4jlogger.debug("WSIFrameOfReference(): Directory={}",dirNum);
				
				String[] description = ifd.getStringValues(TIFFTags.IMAGEDESCRIPTION);	// not ASCII, in case is XML and uses special characters like "µ"
				{
					boolean downsampled = false;
					double micronsPerPixel = 0d;
					double magnification = 0d;
					if (description != null && description.length > 0) {
						slf4jlogger.debug("WSIFrameOfReference(): description.length = {}",description.length);
						for (String d : description) {
							slf4jlogger.debug("WSIFrameOfReference(): String = {}",d);
							if (d.startsWith("<?xml")) {	// (001285)
								slf4jlogger.debug("WSIFrameOfReference(): Parsing OME-TIFF XML metadata");
								try {
									//DocumentBuilderFactory.setNamespaceAware(true);	// don't do this - stops XPath from recognizing attributes :(
									Document document = DocumentBuilderFactory.newInstance().newDocumentBuilder().parse(new InputSource(new StringReader(d)));
									XPathFactory xpf = XPathFactory.newInstance();
									// OME/Image/Pixels@ PhysicalSizeX="0.5022" PhysicalSizeXUnit="µm" PhysicalSizeY="0.5022" PhysicalSizeYUnit="µm"
									{
										String physicalSizeX = xpf.newXPath().evaluate("OME/Image/Pixels/@PhysicalSizeX",document);
										slf4jlogger.debug("WSIFrameOfReference(): found physicalSizeX {}",physicalSizeX);
										String physicalSizeY = xpf.newXPath().evaluate("OME/Image/Pixels/@PhysicalSizeY",document);
										slf4jlogger.debug("WSIFrameOfReference(): found PhysicalSizeY {}",physicalSizeY);
										String physicalSizeXUnit = xpf.newXPath().evaluate("OME/Image/Pixels/@PhysicalSizeXUnit",document);
										slf4jlogger.debug("WSIFrameOfReference(): found PhysicalSizeXUnit {}",physicalSizeXUnit);
										String physicalSizeYUnit = xpf.newXPath().evaluate("OME/Image/Pixels/@PhysicalSizeYUnit",document);
										slf4jlogger.debug("WSIFrameOfReference(): found PhysicalSizeYUnit {}",physicalSizeYUnit);
										// could theoretically handle non-square pixels in DICOM, but happen not to have in WSIFrameOfReference - revisit if ever encountered :(
										
										//only matches "µm" if ImageDescription bytes extracted as "UTF-8" not "US_ASCII", otherwise is pair of Unicode "unrecognized characters"
										if (physicalSizeX.length() > 0 && physicalSizeY.equals(physicalSizeX) && physicalSizeXUnit.equals("µm")) {
											micronsPerPixel = Double.parseDouble(physicalSizeX);
											slf4jlogger.debug("WSIFrameOfReference(): set micronsPerPixel to {}",micronsPerPixel);
										}
										else {
											slf4jlogger.debug("WSIFrameOfReference(): OME/Image/Pixels PhysicalSize attributes not as expected, not used");
										}
									}
									// OME/Instrument/Objective@ NominalMagnification="20.0"
									{
										String nominalMagnification = xpf.newXPath().evaluate("OME/Instrument/Objective/@NominalMagnification",document);
										slf4jlogger.debug("WSIFrameOfReference(): found nominalMagnification {}",nominalMagnification);
										if (nominalMagnification.length() > 0) {
											objectiveLensPower = Double.parseDouble(nominalMagnification);
										}
									}
									// OME/Image/Pixels/Channel@ ID="Channel:0:0" Name="NUCLEI" SamplesPerPixel="1"
									// OME/Image/Pixels/Channel@ Channel ID="Channel:0:1" Name="PD1" SamplesPerPixel="1"
									{
										NodeList channels = (NodeList)(xpf.newXPath().evaluate("OME/Image/Pixels/Channel",document,XPathConstants.NODESET));
										if (channels.getLength() == ifdlist.size()) {
											// assume channels in metadata match order of IFDs
											for (int i=0; i<channels.getLength(); ++i) {
												Node channel = channels.item(i);
												String channelID = xpf.newXPath().evaluate("@ID",channel);
												slf4jlogger.debug("WSIFrameOfReference(): found channelID {}",channelID);
												opticalPathIdentifier[i] = channelID.replace("Channel:0:","");
												slf4jlogger.debug("WSIFrameOfReference(): extracted opticalPathIdentifier {}",opticalPathIdentifier[i]);
												opticalPathDescription[i] = xpf.newXPath().evaluate("@Name",channel);
												slf4jlogger.debug("WSIFrameOfReference(): found Channel Name (OpticalPathDescription) {}",opticalPathDescription[i]);
											}
										}
										else {
											slf4jlogger.error("WSIFrameOfReference(): number of channels in OME-TIFF XML metadata {} does not match number of IFDs {}",channels.getLength(),ifdlist.size());
										}
									}
								}
								catch (Exception e) {
									slf4jlogger.error("Failed to parse OME-TIFF XML metadata in ImageDescription ",e);
								}
							}
							else if (d.contains("Aperio")) {
								// 46920x33014 [0,100 46000x32914] (256x256) JPEG/RGB Q=30|AppMag = 20|StripeWidth = 2040|ScanScope ID = CPAPERIOCS|Filename = CMU-1|Date = 12/29/09|Time = 09:59:15|User = b414003d-95c6-48b0-9369-8010ed517ba7|Parmset = USM Filter|MPP = 0.4990|Left = 25.691574|Top = 23.449873|LineCameraSkew = -0.000424|LineAreaXOffset = 0.019265|LineAreaYOffset = -0.000313|Focus Offset = 0.000000|ImageID = 1004486|OriginalWidth = 46920|Originalheight = 33014|Filtered = 5|ICC Profile = ScanScope v1

								// 46000x32914 -> 1024x732 - |AppMag = 20|StripeWidth = 2040|ScanScope ID = CPAPERIOCS|Filename = CMU-1|Date = 12/29/09|Time = 09:59:15|User = b414003d-95c6-48b0-9369-8010ed517ba7|Parmset = USM Filter|MPP = 0.4990|Left = 25.691574|Top = 23.449873|LineCameraSkew = -0.000424|LineAreaXOffset = 0.019265|LineAreaYOffset = -0.000313|Focus Offset = 0.000000|ImageID = 1004486|OriginalWidth = 46920|Originalheight = 33014|Filtered = 5|ICC Profile = ScanScope v1

								// 46920x33014 [0,100 46000x32914] (256x256) -> 11500x8228 JPEG/RGB Q=65
				
								// 46000x32914 [0,0 46000x32893] (240x240) J2K/KDU Q=30;CMU-1;Aperio Image Library v10.0.51
								// 46920x33014 [0,100 46000x32914] (256x256) JPEG/RGB Q=30|AppMag = 20|StripeWidth = 2040|ScanScope ID = CPAPERIOCS|Filename = CMU-1|Date = 12/29/09|Time = 09:59:15|User = b414003d-95c6-48b0-9369-8010ed517ba7|Parmset = USM Filter|MPP = 0.4990|Left = 25.691574|Top = 23.449873|LineCameraSkew = -0.000424|LineAreaXOffset = 0.019265|LineAreaYOffset = -0.000313|Focus Offset = 0.000000|ImageID = 1004486|OriginalWidth = 46920|Originalheight = 33014|Filtered = 5|OriginalWidth = 46000|OriginalHeight = 32914

								// 46000x32893 -> 1024x732 - ;CMU-1;Aperio Image Library v10.0.51
								// 46920x33014 [0,100 46000x32914] (256x256) JPEG/RGB Q=30|AppMag = 20|StripeWidth = 2040|ScanScope ID = CPAPERIOCS|Filename = CMU-1|Date = 12/29/09|Time = 09:59:15|User = b414003d-95c6-48b0-9369-8010ed517ba7|Parmset = USM Filter|MPP = 0.4990|Left = 25.691574|Top = 23.449873|LineCameraSkew = -0.000424|LineAreaXOffset = 0.019265|LineAreaYOffset = -0.000313|Focus Offset = 0.000000|ImageID = 1004486|OriginalWidth = 46920|Originalheight = 33014|Filtered = 5|OriginalWidth = 46000|OriginalHeight = 32914

								// 29600x42592 (256x256) J2K/KDU Q=70;BioImagene iScan|Scanner ID = BI10N0294|AppMag = 20|MPP = 0.46500
					
								// 152855x79623 [0,0,152855x79623] (256x256) JPEG/YCC Q=91|AppMag = 40|Date = 07/22/2020|Exposure Scale = 0.000001|Exposure Time = 8|Filtered = 3|Focus Offset = 0.089996|Gamma = 2.2|Left = 6.8904371261597|MPP = 0.263592|Rack = 1|ScanScope ID = SS45002|Slide = 1|StripeWidth = 4096|Time = 08:34:09|Time Zone = GMT+0100|Top = 23.217206954956

								downsampled = d.contains("->");	// need to detect this when MPP of base layer described for downsampled layer
								slf4jlogger.debug("WSIFrameOfReference(): found downsampled = {}",downsampled);
								
								try {
									BufferedReader r = new BufferedReader(new StringReader(d));
									String line = null;
									while ((line=r.readLine()) != null) {
										{
											// |MPP = 0.4990| or |MPP = 0.4990 end of line with no trailing delimeter
											Pattern p = Pattern.compile(".*[|]MPP[ ]*=[ ]*([0-9][0-9]*[.][0-9][0-9]*)[|]*.*");
											Matcher m = p.matcher(line);
											if (m.matches()) {
												slf4jlogger.debug("WSIFrameOfReference(): have MPP match");
												int groupCount = m.groupCount();
												if (groupCount == 1) {
													slf4jlogger.debug("WSIFrameOfReference(): have MPP correct groupCount");
													try {
														micronsPerPixel = Double.parseDouble(m.group(1));
														slf4jlogger.debug("WSIFrameOfReference(): found micronsPerPixel (MPP) {}",micronsPerPixel);
													}
													catch (NumberFormatException e) {
														slf4jlogger.error("Failed to parse MPP to double ",e);
													}
												}
											}
										}
										{
											// |AppMag = 20| or |AppMag = 20 end of line with no trailing delimeter
											Pattern p = Pattern.compile(".*[|]AppMag[ ]*=[ ]*([0-9][0-9]*)[|]*.*");
											Matcher m = p.matcher(line);
											if (m.matches()) {
												slf4jlogger.debug("WSIFrameOfReference(): have AppMag match");
												int groupCount = m.groupCount();
												if (groupCount == 1) {
													slf4jlogger.debug("WSIFrameOfReference(): have AppMag correct groupCount");
													try {
														objectiveLensPower = Double.parseDouble(m.group(1));
														slf4jlogger.debug("WSIFrameOfReference(): found objectiveLensPower (AppMag) {}",objectiveLensPower);
													}
													catch (NumberFormatException e) {
														slf4jlogger.error("Failed to parse AppMag to double ",e);
													}
												}
											}
										}
										{
											// |Left = 25.691574|
											Pattern p = Pattern.compile(".*[|]Left[ ]*=[ ]*([0-9][0-9]*[.][0-9][0-9]*)[|].*");
											Matcher m = p.matcher(line);
											if (m.matches()) {
												slf4jlogger.debug("WSIFrameOfReference(): have Left match");
												int groupCount = m.groupCount();
												if (groupCount == 1) {
													slf4jlogger.debug("WSIFrameOfReference(): have Left correct groupCount");
													try {
														// SVS, so can assume slide on its side with label on left
														distanceLongAxisSlideFromMacroLeftEdge = Double.parseDouble(m.group(1));
														slf4jlogger.debug("WSIFrameOfReference(): found Left {}",distanceLongAxisSlideFromMacroLeftEdge);
														haveOverviewRelativeLeft = true;
													}
													catch (NumberFormatException e) {
														slf4jlogger.error("Failed to parse Left to double ",e);
													}
												}
											}
										}
										{
											// |Top = 23.449873|
											Pattern p = Pattern.compile(".*[|]Top[ ]*=[ ]*([0-9][0-9]*[.][0-9][0-9]*)[|].*");
											Matcher m = p.matcher(line);
											if (m.matches()) {
												slf4jlogger.debug("WSIFrameOfReference(): have Top match");
												int groupCount = m.groupCount();
												if (groupCount == 1) {
													slf4jlogger.debug("WSIFrameOfReference(): have Top correct groupCount");
													try {
														// SVS, so can assume slide on its side with label on left
														distanceShortAxisSlideFromMacroBottomEdge = Double.parseDouble(m.group(1));
														slf4jlogger.debug("WSIFrameOfReference(): found Top {}",distanceShortAxisSlideFromMacroBottomEdge);
														haveOverviewRelativeTop = true;
													}
													catch (NumberFormatException e) {
														slf4jlogger.error("Failed to parse Top to double ",e);
													}
												}
											}
										}
									}
								}
								catch (IOException e) {
									slf4jlogger.error("Failed to parse ImageDescription ",e);
								}
							}
							else if (d.contains("X scan size")) {
								// encountered in 3D Histech uncompressed TIFF samples
								
								// ImageDescription: X scan size = 4.27mm
								// Y scan size = 28.90mm
								// X offset = 74.00mm
								// Y offset = 23.90mm
								// X resolution = 17067
								// Y resolution = 115600
								// Triple Simultaneous Acquisition
								// Resolution (um) = 0.25
								// Tissue Start Pixel = 40400
								// Tissue End Pixel = 108800
								// Source = Bright Field
								
								try {
									BufferedReader r = new BufferedReader(new StringReader(d));
									String line = null;
									while ((line=r.readLine()) != null) {
										{
											// Resolution (um) = 0.25
											Pattern p = Pattern.compile(".*Resolution[ ]*[(]um[)][ ]*=[ ]*([0-9][0-9]*[.][0-9][0-9]*).*");
											Matcher m = p.matcher(line);
											if (m.matches()) {
												slf4jlogger.debug("WSIFrameOfReference(): have Resolution (um) match");
												int groupCount = m.groupCount();
												if (groupCount == 1) {
													slf4jlogger.debug("WSIFrameOfReference(): have Resolution (um) correct groupCount");
													try {
														micronsPerPixel = Double.parseDouble(m.group(1));
														slf4jlogger.debug("WSIFrameOfReference(): found Resolution (um) {}",micronsPerPixel);
													}
													catch (NumberFormatException e) {
														slf4jlogger.error("Failed to parse Resolution to double ",e);
													}
												}
											}
										}
										{
											// X offset = 74.00mm
											Pattern p = Pattern.compile(".*X offset[ ]*=[ ]*([0-9][0-9]*[.][0-9][0-9]*)[ ]*mm.*");
											Matcher m = p.matcher(line);
											if (m.matches()) {
												slf4jlogger.debug("WSIFrameOfReference(): have X offset match");
												int groupCount = m.groupCount();
												if (groupCount == 1) {
													slf4jlogger.debug("WSIFrameOfReference(): have X offset correct groupCount");
													try {
														// just a guess that this is what 'X' is based on observed values ...
														double xOffset = Double.parseDouble(m.group(1));
														slf4jlogger.debug("WSIFrameOfReference(): found X offset (mm) {}",xOffset);
														yOffsetInSlideCoordinateSystemPyramid = xOffset;	// DICOM Slide Coordinate System is different X and Y
													}
													catch (NumberFormatException e) {
														slf4jlogger.error("Failed to parse X offset to double ",e);
													}
												}
											}
										}
										{
											// Y offset = 23.90mm
											Pattern p = Pattern.compile(".*Y offset[ ]*=[ ]*([0-9][0-9]*[.][0-9][0-9]*)[ ]*mm.*");
											Matcher m = p.matcher(line);
											if (m.matches()) {
												slf4jlogger.debug("WSIFrameOfReference(): have Y offset match");
												int groupCount = m.groupCount();
												if (groupCount == 1) {
													slf4jlogger.debug("WSIFrameOfReference(): have Y offset correct groupCount");
													try {
														double yOffset = Double.parseDouble(m.group(1));
														slf4jlogger.debug("WSIFrameOfReference(): found Y offset (mm) {}",yOffset);
														xOffsetInSlideCoordinateSystemPyramid = yOffset;	// DICOM Slide Coordinate System is different X and Y
													}
													catch (NumberFormatException e) {
														slf4jlogger.error("Failed to parse Y offset to double ",e);
													}
												}
											}
										}
									}
								}
								catch (IOException e) {
									slf4jlogger.error("Failed to parse ImageDescription ",e);
								}
							}
						}
						
						if (!downsampled && micronsPerPixel != 0) {		// take care not to use MPP of base layer specified for downsampled layer, e.g., for 2nd directory of strips
							mmPerPixel[dirNum] = micronsPerPixel/1000.0d;
							slf4jlogger.debug("WSIFrameOfReference(): mmPerPixel[{}] set to {} for non-downsampled layer",dirNum,mmPerPixel[dirNum]);
						}
					}
					else {
						slf4jlogger.debug("WSIFrameOfReference(): no ImageDescription");
					}
				}
				slf4jlogger.debug("mmPerPixel[{}] extracted from descriptionList={}",dirNum,mmPerPixel[dirNum]);

				if (mmPerPixel[dirNum] == 0) {
					slf4jlogger.debug("mmPerPixel is zero after parsing descriptionList");
					
					// XResolution (282) RATIONAL (5) 1<10>			Generic-TIFF/CMU-1.tiff - obviously invalid
					// YResolution (283) RATIONAL (5) 1<10>			Generic-TIFF/CMU-1.tiff - obviously invalid
					// XResolution (282) RATIONAL (5) 1<40000/1>
					// YResolution (283) RATIONAL (5) 1<40000/1>
					// XResolution (282) RATIONAL (5) 1<20576.4>	PESO - missing denominator
					// YResolution (283) RATIONAL (5) 1<20576.4>	PESO - missing denominator
					double xResolution = ifd.getSingleRationalValue(TIFFTags.XRESOLUTION,0,0);
					slf4jlogger.debug("xResolution={}",xResolution);
					double yResolution = ifd.getSingleRationalValue(TIFFTags.YRESOLUTION,0,0);
					slf4jlogger.debug("yResolution={}",yResolution);

					//if (xResolution > 0 && (!isWSI || xResolution > 10)) {		// not just greater than missing value of 0, but greater than meaningless incorrect value of 10 in Generic-TIFF/CMU-1.tiff
					if (xResolution > 10) {											// not just greater than missing value of 0, but greater than meaningless incorrect value of 10 in Generic-TIFF/CMU-1.tiff
						if (xResolution == yResolution) {
							// ResolutionUnit (296) SHORT (3) 1<3>
							long resolutionUnit = ifd.getSingleNumericValue(TIFFTags.RESOLUTIONUNIT,0,2);	// 1 = none, 2 = inch (default), 3 = cm
							slf4jlogger.debug("resolutionUnit={}",resolutionUnit);

							if (resolutionUnit == 2) {		// inch
								mmPerPixel[dirNum] = 25.4d / xResolution;
							}
							else if (resolutionUnit == 3) {	// cm
								mmPerPixel[dirNum] = 10.0d / xResolution;
							}
							else if (resolutionUnit == 1) {
								slf4jlogger.debug("not using no meaningful RESOLUTIONUNIT for mmPerPixel");
							}
							else {
								slf4jlogger.debug("not using unrecognized RESOLUTIONUNIT {} for mmPerPixel",resolutionUnit);
							}
						}
						else {
							slf4jlogger.debug("not using non-square or uncalibrated X/YRESOLUTION for mmPerPixel");
						}
					}
					else {
						slf4jlogger.debug("not using missing or obviously invalid XRESOLUTION of {} for mmPerPixel",xResolution);
					}
					slf4jlogger.debug("mmPerPixel is {} after checking X/YRESOLUTION and RESOLUTIONUNIT",mmPerPixel[dirNum]);
				}
				
				//if (isWSI) {
				{
					long imageWidth = ifd.getSingleNumericValue(TIFFTags.IMAGEWIDTH,0,0);
					slf4jlogger.debug("imageWidth={}",imageWidth);
					long imageLength = ifd.getSingleNumericValue(TIFFTags.IMAGELENGTH,0,0);
					slf4jlogger.debug("imageLength={}",imageLength);

					if (dirNum == 0) {
						widthOfBaseLayerInPixels = imageWidth;		// store this to calculate pixel spacing for subsequent (lower) layers of pyramid
					}
					if (mmPerPixel[dirNum] == 0) {
						slf4jlogger.debug("mmPerPixel is zero");
						if (dirNum == 0) {		// assume base layer is first ... could check this with extra pass through IFDs :(
							slf4jlogger.debug("using default {} mmPerPixel, assuming first directory is base layer for WSI",mmPerPixelBaseLayerDefault);
							mmPerPixel[dirNum] = mmPerPixelBaseLayerDefault;
							mmPerPixelBaseLayer = mmPerPixel[dirNum]; // (001265)
						}
						else {
							if (imageFlavorAndDerivationByIFD[dirNum][0].equals("OVERVIEW")) {
								slf4jlogger.debug("computing mmPerPixel for macro (OVERVIEW) from standard slide height");
								mmPerPixelOverviewImage = 25.4d / imageLength;	// (001267)
								mmPerPixel[dirNum] = mmPerPixelOverviewImage;
								dirNumOfOverview = dirNum;
								haveOverviewPixelSpacing = true;
								widthOfOverviewInPixels = imageWidth;		// store this to calculate overview origin in frame of reference
								lengthOfOverviewInPixels = imageLength;		// store this to calculate overview origin in frame of reference
							}
							else {
								slf4jlogger.debug("deriving mmPerPixel from pixel width {} relative to base layer width {} and pixel spacing {}",imageWidth,widthOfBaseLayerInPixels,mmPerPixelBaseLayer);
								mmPerPixel[dirNum] = mmPerPixelBaseLayer * widthOfBaseLayerInPixels/imageWidth;		// assumes all images are same physical width (001265)
							}
						}
					}
					else {
						if (dirNum == 0) {								// assume base layer is first ... could check this with extra pass through IFDs :(
							mmPerPixelBaseLayer = mmPerPixel[dirNum];	// keep track of this, for computing pixel spacing of lower layers of pyramid (001265)
						}
					}
					slf4jlogger.debug("Using mmPerPixel[{}]={}",dirNum,mmPerPixel[dirNum]);
					slf4jlogger.debug("Using mmPerPixelBaseLayer={}",mmPerPixelBaseLayer);
				}
				
				++dirNum;
			}

			slf4jlogger.debug("haveOverviewPixelSpacing={}",haveOverviewPixelSpacing);
			slf4jlogger.debug("haveOverviewRelativeLeft={}",haveOverviewRelativeLeft);
			slf4jlogger.debug("haveOverviewRelativeTop={}",haveOverviewRelativeTop);
			slf4jlogger.debug("xOffsetInSlideCoordinateSystemPyramid={}",xOffsetInSlideCoordinateSystemPyramid);
			slf4jlogger.debug("yOffsetInSlideCoordinateSystemPyramid={}",yOffsetInSlideCoordinateSystemPyramid);

			// (001256) (001268)
			if (haveOverviewPixelSpacing
			 && haveOverviewRelativeLeft
			 && haveOverviewRelativeTop) {
				slf4jlogger.debug("Establishing common frame of reference between OVERVIEW and pyramid");
				
				uidOverview = uidPyramid;

				// (001255)
				// Aperio SVS seems to always be slide on its side with label on left
				// In that orientation, DICOM origin is bottom right corner of slide, short axis is +X and long axis +Y (001269)
							
				xOffsetInSlideCoordinateSystemPyramid = distanceShortAxisSlideFromMacroBottomEdge;
				yOffsetInSlideCoordinateSystemPyramid = mmPerPixelOverviewImage * widthOfOverviewInPixels - distanceLongAxisSlideFromMacroLeftEdge; // (001269)

				xOffsetInSlideCoordinateSystemOverview = mmPerPixelOverviewImage * lengthOfOverviewInPixels;
				yOffsetInSlideCoordinateSystemOverview = mmPerPixelOverviewImage * widthOfOverviewInPixels; // (001269)
			}
			else if (xOffsetInSlideCoordinateSystemPyramid != 0 || yOffsetInSlideCoordinateSystemPyramid != 0) {
				slf4jlogger.debug("Have explicit X and Y offsets for pyramid frame of reference");
				uidOverview = u.getAnotherNewUID();		// probably is no OVERVIEW (in 3D Histech) but just in case
				// xOffsetInSlideCoordinateSystemPyramid and yOffsetInSlideCoordinateSystemPyramid already set
				xOffsetInSlideCoordinateSystemOverview = 0;
				yOffsetInSlideCoordinateSystemOverview = 0;
			}
			else {
				slf4jlogger.debug("Cannot establish common frame of reference between OVERVIEW and pyramid");

				uidOverview = u.getAnotherNewUID();
				
				xOffsetInSlideCoordinateSystemPyramid = 0;
				yOffsetInSlideCoordinateSystemPyramid = 0;
				
				xOffsetInSlideCoordinateSystemOverview = 0;
				yOffsetInSlideCoordinateSystemOverview = 0;
			}
			slf4jlogger.debug("xOffsetInSlideCoordinateSystemPyramid={}",xOffsetInSlideCoordinateSystemPyramid);
			slf4jlogger.debug("yOffsetInSlideCoordinateSystemPyramid={}",yOffsetInSlideCoordinateSystemPyramid);
			slf4jlogger.debug("xOffsetInSlideCoordinateSystemOverview={}",xOffsetInSlideCoordinateSystemOverview);
			slf4jlogger.debug("yOffsetInSlideCoordinateSystemOverview={}",yOffsetInSlideCoordinateSystemOverview);
		}
	}

	/**
	 * <p>Read a TIFF image input format file and create an image of a specified or appropriate SOP Class.</p>
	 *
	 * @param	jsonfile		JSON file describing the functional groups and attributes and values to be added or replaced
	 * @param	inputFileName
	 * @param	outputFilePrefix
	 * @param	outputFileSuffix
	 * @param	modality	may be null
	 * @param	sopClass	may be null
	 * @param	transferSyntax	may be null
	 * @param	addTIFF		whether or not to add a TIFF IFD in the DICOM preamble to make a dual=personality DICOM-TIFF file sharing the same pixel data
	 * @param	useBigTIFF	whether or not to create a BigTIFF rather than Classic TIFF file
	 * @param	addPyramid	whether or not to add multi-resolution pyramid (downsampled) layers to the TIFF IFD and a corresponding DICOM private data element in the same file
	 * @param	mergeStrips	whether or not to merge an image with more than one strip into a single DICOM image, or to create a separate image or frame for each strip
	 * @exception			IOException
	 * @exception			DicomException
	 * @exception			TIFFException
	 * @exception			NumberFormatException
	 */
	public TIFFToDicom(String jsonfile,String inputFileName,String outputFilePrefix,String outputFileSuffix,String modality,String sopClass,String transferSyntax,boolean addTIFF,boolean useBigTIFF,boolean addPyramid,boolean mergeStrips)
			throws IOException, DicomException, TIFFException, NumberFormatException {
			
		TIFFImageFileDirectories ifds = new TIFFImageFileDirectories();
		ifds.read(inputFileName);
		
		boolean isWSI = sopClass != null && sopClass.equals(SOPClass.VLWholeSlideMicroscopyImageStorage);
		slf4jlogger.debug("isWSI={}",isWSI);
		
		byte[] iccProfileOfBaseLayer = null;

		AttributeList commonDescriptionList = new AttributeList();		// keep track of stuff defined once but reusable for subsequent images
		{ Attribute a = new UniqueIdentifierAttribute(TagFromName.SeriesInstanceUID); a.addValue(u.getAnotherNewUID()); commonDescriptionList.put(a); }	// (001163)
		{ Attribute a = new UniqueIdentifierAttribute(TagFromName.StudyInstanceUID); a.addValue(u.getAnotherNewUID()); commonDescriptionList.put(a); }	// (001163)
		{
			java.util.Date currentDateTime = new java.util.Date();
			String currentDate = new java.text.SimpleDateFormat("yyyyMMdd").format(currentDateTime);
			String currentTime = new java.text.SimpleDateFormat("HHmmss.SSS").format(currentDateTime);
			{ Attribute a = new DateAttribute(TagFromName.StudyDate);   a.addValue(currentDate); commonDescriptionList.put(a); }
			{ Attribute a = new DateAttribute(TagFromName.SeriesDate);  a.addValue(currentDate); commonDescriptionList.put(a); }
			{ Attribute a = new DateAttribute(TagFromName.ContentDate); a.addValue(currentDate); commonDescriptionList.put(a); }
			{ Attribute a = new TimeAttribute(TagFromName.StudyTime);   a.addValue(currentTime); commonDescriptionList.put(a); }
			{ Attribute a = new TimeAttribute(TagFromName.SeriesTime);  a.addValue(currentTime); commonDescriptionList.put(a); }
			{ Attribute a = new TimeAttribute(TagFromName.ContentTime); a.addValue(currentTime); commonDescriptionList.put(a); }
		}
		
		String containerIdentifier = "SLIDE_1";
		String specimenIdentifier = "SPECIMEN_1";
		String specimenUID = u.getAnotherNewUID();
		
		ArrayList<TIFFImageFileDirectory> ifdlist = ifds.getListOfImageFileDirectories();
		
		String[][] imageFlavorAndDerivationByIFD = getImageFlavorAndDerivationByIFD(ifdlist);

		WSIFrameOfReference wsifor = new WSIFrameOfReference(ifdlist,imageFlavorAndDerivationByIFD);

		int dirNum = 0;
		for (TIFFImageFileDirectory ifd : ifdlist) {
			slf4jlogger.debug("Directory={}",dirNum);
		
			// SubFileType (254) LONG (4) 1<0>
			long imageWidth = ifd.getSingleNumericValue(TIFFTags.IMAGEWIDTH,0,0);
			slf4jlogger.debug("imageWidth={}",imageWidth);
			long imageLength = ifd.getSingleNumericValue(TIFFTags.IMAGELENGTH,0,0);
			slf4jlogger.debug("imageLength={}",imageLength);
			long bitsPerSample = ifd.getSingleNumericValue(TIFFTags.BITSPERSAMPLE,0,0);
			slf4jlogger.debug("bitsPerSample={}",bitsPerSample);
			long compression = ifd.getSingleNumericValue(TIFFTags.COMPRESSION,0,0);
			slf4jlogger.debug("compression={}",compression);
			long photometric = ifd.getSingleNumericValue(TIFFTags.PHOTOMETRIC,0,0);
			slf4jlogger.debug("photometric={}",photometric);
			// Orientation (274) SHORT (3) 1<1>
			long samplesPerPixel = ifd.getSingleNumericValue(TIFFTags.SAMPLESPERPIXEL,0,0);
			slf4jlogger.debug("samplesPerPixel={}",samplesPerPixel);

			long planarConfig = ifd.getSingleNumericValue(TIFFTags.PLANARCONFIG,0,1);	// default is 1 (chunky not planar format)
			slf4jlogger.debug("planarConfig={}",planarConfig);

			long sampleFormat = ifd.getSingleNumericValue(TIFFTags.SAMPLEFORMAT,0,1);	// assume unsigned if absent, and assume same for all samples (though that is not required)
			slf4jlogger.debug("sampleFormat={}",sampleFormat);

			byte[] jpegTables = null;
			if (compression == 7) {
				jpegTables = ifd.getByteValues(TIFFTags.JPEGTABLES);
				if (jpegTables != null) {
					slf4jlogger.debug("jpegTables present");
					jpegTables = stripSOIEOIMarkers(jpegTables);
				}
			}
			
			byte[] iccProfile = ifd.getByteValues(TIFFTags.ICCPROFILE);
			if (iccProfile != null) {
				slf4jlogger.debug("ICC profile present, of length {}",iccProfile.length);
			}
			if (iccProfile != null && iccProfile.length > 0) {
				if (dirNum == 0) {
					iccProfileOfBaseLayer = iccProfile;		// store this in case need not specified in subsequent layers
				}
			}
			else {
				if (isWSI && iccProfileOfBaseLayer != null && iccProfileOfBaseLayer.length > 0) {
					slf4jlogger.debug("ICC profile absent or empty so using profile of base layer");
					iccProfile = iccProfileOfBaseLayer;		// use base layer profile if not specified in subsequent layers
				}
			}
			
			boolean makeMultiFrame = SOPClass.isMultiframeImageStorage(sopClass);
			slf4jlogger.debug("makeMultiFrame={}",makeMultiFrame);

			// PageNumber (297) SHORT (3) 2<4 5>
			long tileWidth = ifd.getSingleNumericValue(TIFFTags.TILEWIDTH,0,0);
			slf4jlogger.debug("tileWidth={}",tileWidth);
			long tileLength = ifd.getSingleNumericValue(TIFFTags.TILELENGTH,0,0);
			slf4jlogger.debug("tileLength={}",tileLength);
			
			AttributeList descriptionList = new AttributeList();
			String imageComments = "";
			{
				String[] imageDescription = ifd.getStringValues(TIFFTags.IMAGEDESCRIPTION);
				parseTIFFImageDescription(imageDescription,descriptionList,commonDescriptionList);
				imageComments = mergeImageDescription(imageDescription);
			}
			
			String frameOfReferenceUID = wsifor.getFrameOfReferenceUIDForIFD(dirNum);	// (001256)
			slf4jlogger.debug("Using frameOfReferenceUID[{}] from WSI Frame of Reference {} to make DICOM file",dirNum,frameOfReferenceUID);
			double mmPerPixel = wsifor.getmmPerPixelForIFD(dirNum);
			slf4jlogger.debug("Using mmPerPixel[{}] from WSI Frame of Reference {} to make DICOM file",dirNum,mmPerPixel);
			double objectiveLensPower = wsifor.getObjectiveLensPower();
			slf4jlogger.debug("Using objectiveLensPower from WSI Frame of Reference {} to make DICOM file",objectiveLensPower);
			String opticalPathIdentifier = wsifor.getOpticalPathIdentifierForIFD(dirNum);	// (001285)
			slf4jlogger.debug("Using opticalPathIdentifier from WSI Frame of Reference {} to make DICOM file",opticalPathIdentifier);
			String opticalPathDescription = wsifor.getOpticalPathDescriptionForIFD(dirNum);	// (001285)
			slf4jlogger.debug("Using opticalPathDescription from WSI Frame of Reference {} to make DICOM file",opticalPathDescription);
			double xOffsetInSlideCoordinateSystem = wsifor.getXOffsetInSlideCoordinateSystemForIFD(dirNum);
			slf4jlogger.debug("Using xOffsetInSlideCoordinateSystem[{}] from WSI Frame of Reference {} to make DICOM file",dirNum,xOffsetInSlideCoordinateSystem);
			double yOffsetInSlideCoordinateSystem = wsifor.getYOffsetInSlideCoordinateSystemForIFD(dirNum);
			slf4jlogger.debug("Using yOffsetInSlideCoordinateSystem[{}] from WSI Frame of Reference {} to make DICOM file",dirNum,yOffsetInSlideCoordinateSystem);

			try {
				long[] tileOffsets = ifd.getNumericValues(TIFFTags.TILEOFFSETS);
				long[] tileByteCounts = ifd.getNumericValues(TIFFTags.TILEBYTECOUNTS);
				if (tileOffsets != null) {
					int numberOfTiles = tileOffsets.length;
					if (tileByteCounts.length != numberOfTiles) {
						throw new TIFFException("Number of tiles uncertain: tileOffsets length = "+tileOffsets.length+" different from tileByteCounts length "+tileByteCounts.length);
					}
					slf4jlogger.debug("numberOfTiles={}",numberOfTiles);
					if (makeMultiFrame) {
						String outputFileName = outputFilePrefix + "_" + dirNum + outputFileSuffix;
						slf4jlogger.info("outputFileName={}",outputFileName);
						int instanceNumber = dirNum+1;
						convertTIFFTilesToDicomMultiFrame(jsonfile,ifds.getFile(),outputFileName,instanceNumber,imageWidth,imageLength,tileOffsets,tileByteCounts,tileWidth,tileLength,bitsPerSample,compression,jpegTables,iccProfile,photometric,samplesPerPixel,planarConfig,sampleFormat,
														  frameOfReferenceUID,mmPerPixel,objectiveLensPower,
														  opticalPathIdentifier,opticalPathDescription,
														  xOffsetInSlideCoordinateSystem,yOffsetInSlideCoordinateSystem,
														  modality,sopClass,transferSyntax,containerIdentifier,specimenIdentifier,specimenUID,imageFlavorAndDerivationByIFD[dirNum][0],imageFlavorAndDerivationByIFD[dirNum][1],imageComments,commonDescriptionList,addTIFF,useBigTIFF,addPyramid);
					}
					else {
						for (int tileNumber=0; tileNumber<numberOfTiles; ++tileNumber) {
							String outputFileName = outputFilePrefix + "_" + dirNum + "_" + tileNumber + outputFileSuffix;
							slf4jlogger.info("outputFileName={}",outputFileName);
							int instanceNumber = (dirNum+1)*100000+(tileNumber+1);
							convertTIFFTilesToDicomSingleFrame(jsonfile,ifds.getFile(),outputFileName,instanceNumber,imageWidth,imageLength,tileOffsets[tileNumber],tileByteCounts[tileNumber],tileWidth,tileLength,bitsPerSample,compression,jpegTables,iccProfile,photometric,samplesPerPixel,planarConfig,sampleFormat,
												   modality,sopClass,transferSyntax,imageFlavorAndDerivationByIFD[dirNum][0],imageFlavorAndDerivationByIFD[dirNum][1],imageComments,commonDescriptionList,addTIFF,useBigTIFF);
						}
					}
				}
				else {
					long rowsPerStrip = ifd.getSingleNumericValue(TIFFTags.ROWSPERSTRIP,0,0);
					slf4jlogger.debug("rowsPerStrip={}",rowsPerStrip);
					long[] stripOffsets = ifd.getNumericValues(TIFFTags.STRIPOFFSETS);
					long[] stripByteCounts = ifd.getNumericValues(TIFFTags.STRIPBYTECOUNTS);
					if (stripByteCounts != null) {
						slf4jlogger.debug("Strips rather than tiled");
						int numberOfStrips = stripOffsets.length;
						slf4jlogger.debug("numberOfStrips={}",numberOfStrips);
						if (stripByteCounts.length != numberOfStrips) {
							throw new TIFFException("Number of strips uncertain: stripOffsets length = "+stripOffsets.length+" different from stripByteCounts length "+stripByteCounts.length);
						}
						if (rowsPerStrip == imageLength) {
							slf4jlogger.debug("Single strip for entire image");
							if (numberOfStrips != 1) {
								throw new TIFFException("Number of strips uncertain: stripOffsets length = "+stripOffsets.length+" > 1 but rowsPerStrip == imageLength of "+rowsPerStrip);
							}
							String outputFileName = outputFilePrefix + "_" + dirNum + outputFileSuffix;
							slf4jlogger.info("outputFileName={}",outputFileName);
							int instanceNumber = dirNum+1;
							convertTIFFTilesToDicomSingleFrame(jsonfile,ifds.getFile(),outputFileName,instanceNumber,imageWidth,imageLength,stripOffsets[0],stripByteCounts[0],imageWidth,rowsPerStrip,bitsPerSample,compression,jpegTables,iccProfile,photometric,samplesPerPixel,planarConfig,sampleFormat,
												   modality,sopClass,transferSyntax,imageFlavorAndDerivationByIFD[dirNum][0],imageFlavorAndDerivationByIFD[dirNum][1],imageComments,commonDescriptionList,addTIFF,useBigTIFF);
						}
						else {
							if (mergeStrips) {
								slf4jlogger.debug("Merging strips into single image");
								String outputFileName = outputFilePrefix + "_" + dirNum + outputFileSuffix;
								slf4jlogger.info("outputFileName={}",outputFileName);
								int instanceNumber = dirNum+1;
								// if merging strips, always output decompressed, since compression is per strip, don't have ability to merge compressed bitstreams, and do not want additional compression loss of recompressing
								convertTIFFTilesToDicomSingleFrameMergingStrips(jsonfile,ifds.getFile(),outputFileName,instanceNumber,imageWidth,imageLength,stripOffsets,stripByteCounts,imageWidth,rowsPerStrip,bitsPerSample,compression,jpegTables,iccProfile,photometric,samplesPerPixel,planarConfig,sampleFormat,
													frameOfReferenceUID,mmPerPixel,objectiveLensPower,
													opticalPathIdentifier,opticalPathDescription,
													xOffsetInSlideCoordinateSystem,yOffsetInSlideCoordinateSystem,
													modality,sopClass,TransferSyntax.ExplicitVRLittleEndian,containerIdentifier,specimenIdentifier,specimenUID,imageFlavorAndDerivationByIFD[dirNum][0],imageFlavorAndDerivationByIFD[dirNum][1],imageComments,commonDescriptionList,addTIFF,useBigTIFF);
							}
							else {
								slf4jlogger.debug("Not merging strips - each becomes single frame or image");
								if (makeMultiFrame) {
									String outputFileName = outputFilePrefix + "_" + dirNum + outputFileSuffix;
									slf4jlogger.info("outputFileName={}",outputFileName);
									int instanceNumber = dirNum+1;
									convertTIFFTilesToDicomMultiFrame(jsonfile,ifds.getFile(),outputFileName,instanceNumber,imageWidth,imageLength,stripOffsets,stripByteCounts,imageWidth,rowsPerStrip,bitsPerSample,compression,jpegTables,iccProfile,photometric,samplesPerPixel,planarConfig,sampleFormat,
													frameOfReferenceUID,mmPerPixel,objectiveLensPower,
													opticalPathIdentifier,opticalPathDescription,
													xOffsetInSlideCoordinateSystem,yOffsetInSlideCoordinateSystem,
													modality,sopClass,transferSyntax,containerIdentifier,specimenIdentifier,specimenUID,imageFlavorAndDerivationByIFD[dirNum][0],imageFlavorAndDerivationByIFD[dirNum][1],imageComments,commonDescriptionList,addTIFF,useBigTIFF,addPyramid);
								}
								else {
									for (int stripNumber=0; stripNumber<numberOfStrips; ++stripNumber) {
										String outputFileName = outputFilePrefix + "_" + dirNum + "_" + stripNumber + outputFileSuffix;
										slf4jlogger.info("outputFileName={}",outputFileName);
										int instanceNumber = (dirNum+1)*100000+(stripNumber+1);
										convertTIFFTilesToDicomSingleFrame(jsonfile,ifds.getFile(),outputFileName,instanceNumber,imageWidth,imageLength,stripOffsets[stripNumber],stripByteCounts[stripNumber],imageWidth,rowsPerStrip,bitsPerSample,compression,jpegTables,iccProfile,photometric,samplesPerPixel,planarConfig,sampleFormat,
													modality,sopClass,transferSyntax,imageFlavorAndDerivationByIFD[dirNum][0],imageFlavorAndDerivationByIFD[dirNum][1],imageComments,commonDescriptionList,addTIFF,useBigTIFF);
									}
								}
							}
						}
					}
					else {
						throw new TIFFException("Unsupported encoding");
					}
				}
			}
			catch (Exception e) {
				slf4jlogger.error("Failed to construct DICOM image: ",e);
			}
			++dirNum;
		}
	}
	
	/**
	 * <p>Read a TIFF image input format file consisting of one or more pages or tiles, and create one or more images of a specified or appropriate SOP Class.</p>
	 *
	 * <p>Options are:</p>
	 * <p>ADDTIFF | DONOTADDTIFF (default)</p>
	 * <p>USEBIGTIFF (default) | DONOTUSEBIGTIFF</p>
	 * <p>ADDPYRAMID | DONOTADDPYRAMID (default)</p>
	 * <p>MERGESTRIPS (default) | DONOTMERGESTRIPS</p>
	 *
	 * @param	arg	three, four or five parameters plus options, a JSON file describing the functional groups and attributes and values to be added or replaced, the TIFF inputFile, DICOM file outputFilePrefix, and optionally the modality, the SOP Class, and the Transfer Syntax to use, then various options controlling conversion
	 */
	public static void main(String arg[]) {
		try {
			boolean addTIFF = false;
			boolean useBigTIFF = true;
			boolean addPyramid = false;
			boolean mergeStrips = true;
			
			String outputFileSuffix = ".dcm";

			int numberOfFixedArguments = 3;
			int numberOfFixedAndOptionalArguments = 6;
			int endOptionsPosition = arg.length;
			boolean bad = false;
			
			if (endOptionsPosition < numberOfFixedArguments) {
				bad = true;
			}
			boolean keepLooking = true;
			while (keepLooking && endOptionsPosition > numberOfFixedArguments) {
				String option = arg[endOptionsPosition-1].trim().toUpperCase();
				switch (option) {
					case "ADDTIFF":				addTIFF = true;  --endOptionsPosition; break;
					case "DONOTADDTIFF":		addTIFF = false; --endOptionsPosition; break;
					
					case "USEBIGTIFF":			useBigTIFF = true;  --endOptionsPosition; break;
					case "DONOTUSEBIGTIFF":		useBigTIFF = false; --endOptionsPosition; break;

					case "ADDPYRAMID":			addPyramid = true;  --endOptionsPosition; break;
					case "DONOTADDPYRAMID":		addPyramid = false; --endOptionsPosition; break;

					case "MERGESTRIPS":			mergeStrips = true;  --endOptionsPosition; break;
					case "DONOTMERGESTRIPS":	mergeStrips = false; --endOptionsPosition; break;

					case "ADDDCMSUFFIX":		outputFileSuffix = ".dcm"; --endOptionsPosition; break;
					case "DONOTADDDCMSUFFIX":	outputFileSuffix = "";     --endOptionsPosition; break;
					
					default:	if (endOptionsPosition > numberOfFixedAndOptionalArguments) {
									slf4jlogger.error("Unrecognized argument {}",option);
									bad = true;
								}
								keepLooking = false;
								break;
				}
			}
			
			if (!bad) {
				String jsonfile = arg[0];
				String inputFile = arg[1];
				String outputFilePrefix = arg[2];
				String modality = null;
				String sopClass = null;
				String transferSyntax = null;

				if (endOptionsPosition >= 4) {
					modality = arg[3];
				}
				if (endOptionsPosition >= 5) {
					sopClass = arg[4];
				}
				if (endOptionsPosition >= 6) {
					transferSyntax = arg[5];
				}
				
				new TIFFToDicom(jsonfile,inputFile,outputFilePrefix,outputFileSuffix,modality,sopClass,transferSyntax,addTIFF,useBigTIFF,addPyramid,mergeStrips);
			}
			else {
				System.err.println("Error: Incorrect number of arguments or bad arguments");
				System.err.println("Usage: TIFFToDicom jsonfile inputFile outputFilePrefix [modality [SOPClass [TransferSyntax]]]"
					+" [ADDTIFF|DONOTADDTIFF]"
					+" [USEBIGTIFF|DONOTUSEBIGTIFF]"
					+" [ADDPYRAMID|DONOTADDPYRAMID]"
					+" [MERGESTRIPS|SPLITSTRIPS]"
				);
				System.exit(1);
			}
		}
		catch (Exception e) {
			e.printStackTrace();
		}
	}
}

