/************************************************************************/
/*                                                                      */
/*    vspline - a set of generic tools for creation and evaluation      */
/*              of uniform b-splines                                    */
/*                                                                      */
/*            Copyright 2015, 2016 by Kay F. Jahnke                     */
/*                                                                      */
/*    Permission is hereby granted, free of charge, to any person       */
/*    obtaining a copy of this software and associated documentation    */
/*    files (the "Software"), to deal in the Software without           */
/*    restriction, including without limitation the rights to use,      */
/*    copy, modify, merge, publish, distribute, sublicense, and/or      */
/*    sell copies of the Software, and to permit persons to whom the    */
/*    Software is furnished to do so, subject to the following          */
/*    conditions:                                                       */
/*                                                                      */
/*    The above copyright notice and this permission notice shall be    */
/*    included in all copies or substantial portions of the             */
/*    Software.                                                         */
/*                                                                      */
/*    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND    */
/*    EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES   */
/*    OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND          */
/*    NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT       */
/*    HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,      */
/*    WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING      */
/*    FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR     */
/*    OTHER DEALINGS IN THE SOFTWARE.                                   */
/*                                                                      */
/************************************************************************/

/// restore_test - create b-splines from random data and restore the original
/// data from the spline. This has grown to  function as a unit test
/// instantiating all sorts of splines and evaluators and using the evaluators
/// with the functions in transform.h.
///
/// Ideally, this test should result in restored data which are identical to
/// the initial random data, but several factors come into play:
///
/// the precision of the coefficient calculation is parametrizable in vspline,
/// via the 'tolerance' parameter, or, in some places, the 'horizon' parameter.
///
/// the data type of the spline is important - single precision data are unfit
/// to produce coefficients capable of reproducing the data very precisely
///
/// the extrapolation method may be EXPLICIT or BRACED (in this test), and
/// since the EXPLICIT extrapolation uses clever guesswork for the initial
/// causal and anticausal coefficients at the margin of it's extrapolated
/// data array, it usually comes out the winner precisionwise. But using
/// EXPLICIT prefilter strategy takes a long time for higher-D splines with
/// high precision, so I usually comment it out (look for 'EXPLICIT')
///
/// the spline degree is important. High degrees need wide horizons, but wide
/// horizons also need many calculation steps, which may introduce errors.
///
/// The dimension of the spline is also important. Since higher-dimension
/// splines need more calculations for prefiltering and evaluation, results are
/// less precise. This test program tets up to 4D.
///
/// With dimensions > 2 running this program in full takes a long time, since
/// it tries to be comprehensive to catch any corner cases which might have
/// escaped scrutiny.

#include <vigra/multi_array.hxx>
#include <vigra/accumulator.hxx>
#include <vigra/multi_math.hxx>
#include <iostream>
#include <typeinfo>
#include <random>

// for this test, we may define these two symbols, resulting in extra
// tests on the mappers used.

// #define ASSERT_IN_BOUNDS
// #define ASSERT_CONSISTENT
 
#include <vspline/vspline.h>

bool verbose = false ;

// 'condense' aggregate types (TinyVectors etc.) into a single value

template < typename T >
double condense ( const T & t , std::true_type )
{
  return std::abs ( t ) ;
}

template < typename T >
double condense ( const T & t , std::false_type )
{
  return sqrt ( sum ( t * t ) ) / t.size() ;
}

template < typename T >
double condense ( const std::complex<T> & t , std::false_type )
{
  return std::abs ( t ) ;
}

template < class T >
using is_singular = typename
  std::conditional
  < std::is_fundamental < T > :: value ,
    std::true_type ,
    std::false_type
  > :: type ;

template < typename T >
double condense ( const T & t )
{
  return condense ( t , is_singular<T>() ) ;
}

// compare two arrays and calculate the mean and maximum difference

template < int dim , typename T >
double check_diff ( vigra::MultiArrayView < dim , T > & reference ,
                    vigra::MultiArrayView < dim , T > & candidate )
{
  using namespace vigra::multi_math ;
  using namespace vigra::acc;
  
  assert ( reference.shape() == candidate.shape() ) ;
  
  vigra::MultiArray < 1 , double >
    error_array ( vigra::Shape1(reference.size() ) ) ;
    
  for ( int i = 0 ; i < reference.size() ; i++ )
  {
    auto help = candidate[i] - reference[i] ;
//     std::cerr << reference[i] << " <> " << candidate[i]
//               << " CFD " << help << std::endl ;
    error_array [ i ] = condense ( help ) ;
  }

  AccumulatorChain < double , Select < Mean, Maximum > > ac ;
  extractFeatures ( error_array.begin() , error_array.end() , ac ) ;
  double mean = get<Mean>(ac) ;
  double max = get<Maximum>(ac) ;
  if ( verbose )
  {
    std::cout << "delta Mean:    "
              << mean << std::endl;
    std::cout << "delta Maximum: "
              << max << std::endl;
  }
  return max ;
}

// template < int dim , class array_type , int spline_degree >
// struct get_vigra_coeffs
// {
//   void operator() ( array_type & target , const array_type & orig )
//   { } ;
// } ;
// 
// template < class array_type , int spline_degree >
// struct get_vigra_coeffs<2,array_type,spline_degree>
// {
//   void operator()
//    ( array_type & target , const array_type & orig )
//   {
//     vigra::SplineImageView<spline_degree, typename array_type::value_type>
//     view ( orig ) ;
//     vigra::MultiArrayView < 2 , typename array_type::value_type >
//       v ( view.image() ) ;
//     target = v ;
//   }
// } ;

/// do a restore test. This test fills the array that's
/// passed in with small random data, constructs a b-spline with the requested
/// parameters over the data, then calls vspline::restore(), which evaluates the
/// spline at discrete locations, either using transform() (for 1D data) or
/// grid_eval(), which is more efficient for dimensions > 1.
/// While this test fails to address several aspects (derivatives, behaviour
/// at locations which aren't discrete), it does make sure that prefiltering
/// has produced a correct result and reconstruction succeeds: If the spline
/// coefficients were wrong, reconstruction would fail just as it would if
/// the coefficients were right and the evaluation code was wrong. That both
/// should be wrong and accidentally produce a correct result is highly unlikely.
///
/// This routine has grown to be more of a unit test for all of vspline,
/// additional vspline functions are executed and the results are inspected.
/// This way we can assure that the transform-type routines are usable with
/// all supported data types and vectorized and unvectorized results are
/// consistent.

template < int dim , typename T >
double restore_test ( vigra::MultiArrayView < dim , T > & arr ,
                      vspline::bc_code bc ,
                      int spline_degree ,
                      vspline::prefilter_strategy pfs
                    )
{
  if ( verbose )
    std::cout << "**************************************" << std::endl
              << "testing type " << typeid(T()).name() << std::endl
              << " prefilter strategy "
              << ( pfs == vspline::BRACED ? "BRACED" : "EXPLICIT" )
              << std::endl ;
  typedef vigra::MultiArray < dim , T > array_type ;
  typedef vspline::bspline < T , dim > spline_type ;
  vigra::TinyVector < vspline::bc_code , dim > bcv { bc } ;
  
//   // note that the 'acceptable error' may be exceeded if the arithmetic type
//   // used for calculations isn't sufficient. When using double precision maths,
//   // specifying an acceptable error of .000000001 will limit the error to the
//   // specification, but with single precision, especially with higher degree
//   // splines, the maximum error will often be higher.
//   
//   double acceptable_error = .000000000000000000001 ;

  spline_type bsp  ( arr.shape() , spline_degree , bcv ,  pfs ) ;
//                      -1 , -1.0 , 0.0 , headroom ) ;

  if ( verbose )
    std::cout << "created b-spline:" << std::endl << bsp << std::endl ;
  
  std::random_device rd ;
  std::mt19937 gen ( rd() ) ;
//   gen.seed ( 765 ) ;              // level playing field
  std::uniform_real_distribution<> dis ( -1.0 , 1.0 ) ;
  for ( auto & e : arr )
    e = dis ( gen ) ;
  array_type reference = arr ;
  
  bsp.prefilter ( arr ) ;
  
  vspline::restore < dim , T > ( bsp , arr ) ;
  double emax = check_diff < dim , T > ( reference , arr ) ;
  
  if ( verbose )
  {
    std::cout << "after restoration of original data:" << std::endl ;
    // print a summary, can use '| grep CF' on cout
    std::cout
    << typeid(T()).name()
    << " CF "
    << vspline::pfs_name [ pfs ][0]
    << " D " << dim
    << " " << arr.shape()
    << " BC " << vspline::bc_name[bc]
    << " DG " << spline_degree
    << " TOL " << bsp.tolerance
    << " EMAX "
    << emax << std::endl ;
  }
  
  // test the factory functions make_evaluator and make_safe_evaluator

  auto raw_ev =  vspline::make_evaluator
                      < spline_type , float > ( bsp ) ;
                      
  typedef typename decltype ( raw_ev ) :: in_type fc_type ;
  auto rraw = raw_ev ( fc_type ( .123f ) ) ;
  fc_type cs ( .123f ) ;
  auto rs = raw_ev ( cs ) ;
  raw_ev.eval ( cs , rs ) ;

  // try evaluating the spline at it's lower and upper limit

  cs = fc_type ( vspline::unwrap ( bsp.lower_limit() ) ) ;
  raw_ev.eval ( cs , rs ) ;
  if ( verbose )
    std::cout << cs << " -> " << rs << std::endl ;
  
  cs = fc_type ( vspline::unwrap ( bsp.upper_limit() ) ) ;
  raw_ev.eval ( cs , rs ) ;
  if ( verbose )
    std::cout << cs << " -> " << rs << std::endl ;
  
// this code would only compile for vectorizable data,
// so the tests with long double's would not compile.
// a specific _vsize will only be accepted for vectorizable
// data types.

//   auto raw_ev_4 =  vspline::make_evaluator
//                       < spline_type , float , 4 > ( bsp ) ;
//                       
//   auto rraw4 = raw_ev_4 ( fc_type ( .123f ) ) ;

#ifdef USE_VC
  {
    // if the spline's data type is not vectorizable,
    // in_v will be the same as in_type, so this operation
    // wiil not be vectorized.

    typedef typename decltype ( raw_ev ) :: in_v fc_v ;
    fc_v cv ( .123f ) ;
    auto rv = raw_ev ( cv ) ;
    raw_ev.eval ( cv , rv ) ;
    
//     typedef typename decltype ( raw_ev_4 ) :: in_v fc_v4 ;
//     fc_v4 c4 ( .123f ) ;
//     auto r4 = raw_ev_4 ( c4 ) ;
  }
#endif

  // additionally, we perform a test with a 'safe evaluator' and random
  // coordinates. For a change we use double precision coordinates
  
  auto _ev = vspline::make_safe_evaluator
                      < spline_type , double > ( bsp ) ;

  enum { vsize = decltype ( _ev ) :: vsize } ;
  
  typedef typename decltype ( _ev ) :: in_type coordinate_type ;
  typedef typename decltype ( _ev ) :: in_ele_type rc_type ;
  typedef typename decltype ( _ev ) :: out_ele_type ele_type ;

  // throw a domain in to test that as well:

  auto dom = vspline::domain
             < coordinate_type , spline_type , vsize >
               ( bsp ,
                 coordinate_type(-.3377) ,
                 coordinate_type(3.11) ) ;

  auto ev = vspline::grok ( dom + _ev ) ;
  
  coordinate_type c ;
  
  vigra::MultiArray < 1 , coordinate_type > ca ( vigra::Shape1 ( 10003 ) ) ;
  vigra::MultiArray < 1 , T > ra ( vigra::Shape1 ( 10003 ) ) ;
  vigra::MultiArray < 1 , T > ra2 ( vigra::Shape1 ( 10003 ) ) ;
  
  auto pc = (rc_type*) &c ;
  // make sure we can evaluate at the lower and upper limit
  int k = 0 ;
  {
    for ( int e = 0 ; e < dim ; e++ )
      pc[e] = 0.0 ;
    ra[k] = ev ( c ) ;
    ca[k] = c ;
  }
  k = 1 ;
  {
    for ( int e = 0 ; e < dim ; e++ )
      pc[e] = 1.0 ;
    ra[k] = ev ( c ) ;
    ca[k] = c ;
  }
  k = 2 ;
  {
    for ( int e = 0 ; e < dim ; e++ )
      pc[e] = -2.0 ;
    ra[k] = ev ( c ) ;
    ca[k] = c ;
  }
  k = 3 ;
  {
    for ( int e = 2.0 ; e < dim ; e++ )
      pc[e] = 1.0 ;
    ra[k] = ev ( c ) ;
    ca[k] = c ;
  }
  
  // the remaining coordinates are picked randomly
  
  std::uniform_real_distribution<> dis2 ( -2.371 , 2.1113 ) ;
  for ( k = 4 ; k < 10003 ; k++ )
  {
    for ( int e = 0 ; e < dim ; e++ )
      pc[e] = dis2 ( gen ) ;
    ra[k] = ev ( c ) ;
    ca[k] = c ;
  }
  
  // run an index-based transform. With a domain in action, this
  // does *not* recreate the original data

  vspline::transform ( ev , arr ) ;
 
  // run an array-based transform. result should be identical
  // to single-value-eval above, within arithmetic precision limits
  // given the specific optimization level.
  
  // we can produce different views on the data to make sure feeding
  // the coordinates still works correctly:
  
  vigra::MultiArrayView < 2 , coordinate_type > ca2d
    ( vigra::Shape2 ( 99 , 97 ) , ca.data() ) ;
  vigra::MultiArrayView < 2 , T > ra2d
    ( vigra::Shape2 ( 99 , 97 ) , ra2.data() ) ;
    
  vigra::MultiArrayView < 2 , coordinate_type > ca2ds
    ( vigra::Shape2 ( 49 , 49 ) , vigra::Shape2 ( 2 , 200 ) , ca.data() ) ;
  vigra::MultiArrayView < 2 , T > ra2ds
    ( vigra::Shape2 ( 49 , 49 ) , vigra::Shape2 ( 2 , 200 ) , ra2.data() ) ;
    
  vigra::MultiArrayView < 3 , coordinate_type > ca3d
    ( vigra::Shape3 ( 20 , 21 , 19 ) , ca.data() ) ;
  vigra::MultiArrayView < 3 , T > ra3d
    ( vigra::Shape3 ( 20 , 21 , 19 ) , ra2.data() ) ;
  
  vspline::transform ( ev , ca , ra2 ) ;
  vspline::transform ( ev , ca2d , ra2d ) ;
  vspline::transform ( ev , ca2ds , ra2ds ) ;
  vspline::transform ( ev , ca3d , ra3d ) ;

  // vectorized and unvectorized operation may produce slightly
  // different results in optimized code.
  // usually assert ( ra == ra2 ) holds, but we don't consider
  // it an error if it doesn't and allow for a small difference
  
  auto dsv = check_diff < 1 , T > ( ra , ra2 ) ;
  auto tolerance = 10 * std::numeric_limits<ele_type>::epsilon() ;
  
  if ( dsv > tolerance )
    std::cout << vspline::bc_name[bc]
              << " - max difference single/vectorized eval "
              << dsv << std::endl ;
              
  if ( dsv > .0001 )
  {
    std::cout << vspline::bc_name[bc]
              << " - excessive difference single/vectorized eval "
              << dsv << std::endl ;
    for ( int k = 0 ; k < 10003 ; k++ )
    {
      if ( ra[k] != ra2[k] )
      {
        std::cout << "excessive at k = " << k
                  << ": " << ca[k] << " -> "
                  << ra[k] << ", " << ra2[k] << std::endl ;
      }
    }
  }
  
  // to test broadcasting, we create a broadcasting evaluator
  // from ev

  auto brd_ev = vspline::broadcast
                < decltype(ev) , vsize > ( ev ) ;
                
  vspline::transform ( brd_ev , ca , ra2 ) ;
  
  // with the broadcasted evaluator, this should hold:
  
  assert ( ra == ra2 ) ;
  
  return emax ;

// crossreferencing with vigra is problematic, for several reasons:
// - per default, vigra's SplineImageView will assume mirror BCs. vigra calls
//   them reflect, but it's mirroring on the bounds. If other BCs should be
//   tested, one would have to go one level deeper and directly use vigra's
//   filtering routines. Then, periodic BCs could be compared as well.
// - vigra's SplineImageViews only work on 2D data
// - vigra uses the spline's degree as a template argument and calls it ORDER
// - the acceptable error can't be parametrized, it's fixed at 0.00001
// - small arrays with high degrees need a wide horizon, but vigra's reflect BC
//   does not do multiple reflections, so the coffs come out wrong.
// with a widened eps parameter and sufficiently large 2D arrays, the coeffs
// came out nearly the same, but I omit the test for the reasons above.
  
//   if ( dim == 2 && bc == vspline::MIRROR )
//   {
//     std::cout << "coefficients crossreference with vigra:" << std::endl ;
//     array_type varr ;
//     get_vigra_coeffs<dim,array_type,spline_degree>() ( varr , reference ) ;
//     double emax = check_diff < dim , T > ( bsp.core , varr ) ;
//   }
}

using namespace vspline ;

template < class arr_t >
double view_test ( arr_t & arr )
{
  double emax = 0.0 ;
  
  enum { dimension = arr_t::actual_dimension } ;
  typedef typename arr_t::value_type value_type ;
  vspline::bc_code bc_seq[] { PERIODIC , MIRROR , REFLECT , NATURAL } ;
  vspline::prefilter_strategy pfs_seq [] { BRACED } ; // , EXPLICIT } ;
 

  // TODO: for degree-1 splines, I sometimes get different results
  // for unvectorized and vectorized operation. why?
  
  for ( int spline_degree = 0 ; spline_degree < 8 ; spline_degree++ )
  {
    for ( auto bc : bc_seq )
    {
      for ( auto pfs : pfs_seq )
      {
        auto e = restore_test < dimension , value_type >
                  ( arr , bc , spline_degree , pfs ) ;
        if ( e > emax )
          emax = e ;
      }
    }
  }
  return emax ;
}

int d0[] { 2 , 3 , 5 , 8 , 13 , 16 , 21 , 34 , 55 } ;
int d1[] { 2 , 3 , 5 , 8 , 13 , 16 , 21 , 34 } ;
int d2[] { 2 , 3 , 5 , 8 , 13 , 21 } ;
int d3[] { 2 , 3 , 5 , 8 , 13 } ;

int* dext[] { d0 , d1 , d2 , d3 } ;
int dsz[] { sizeof(d0) / sizeof(int) ,
            sizeof(d1) / sizeof(int) ,
            sizeof(d2) / sizeof(int) ,
            sizeof(d3) / sizeof(int) } ;

template < int dim , typename T >
struct test
{
  typedef vigra::TinyVector < int , dim > shape_type ;
  double emax = 0.0 ;
  
  double operator() ()
  {
    shape_type dshape ; 
    for ( int d = 0 ; d < dim ; d++ )
      dshape[d] = dsz[d] ;
    
    vigra::MultiCoordinateIterator<dim> i ( dshape ) ,
    end = i.getEndIterator() ;  
    
    while ( i != end )
    {
      shape_type shape ;
      for ( int d = 0 ; d < dim ; d++ )
        shape[d] = * ( dext[d] + (*i)[d] ) ;
      
      vigra::MultiArray < dim , T > _arr ( 2 * shape + 1 ) ;
      auto stride = _arr.stride() * 2 ;
      vigra::MultiArrayView < dim , T >
        arr ( shape , stride , _arr.data() + long ( sum ( stride ) ) ) ;
      auto e = view_test ( arr ) ;
      if ( e > emax )
        emax = e ;
      ++i ;
      
      // make sure that we have only written back to 'arr', leaving
      // _arr untouched
      for ( auto & e : arr )
        e = T(0.0) ;
      for ( auto e : _arr )
        assert ( e == T(0.0) ) ;
    }
    return emax ;
  }
} ;

template < typename T >
struct test < 0 , T >
{ 
  double operator() ()
  { 
    return 0.0 ;
  } ;
} ;

template < int dim ,
           typename tuple_type ,
           int ntypes = std::tuple_size<tuple_type>::value >
struct multitest
{
  void operator() ()
  {
    typedef typename std::tuple_element<ntypes-1,tuple_type>::type T ;
    auto e = test < dim , T >() () ;
    std::cout << "test for type " << typeid(T()).name()
              << ": max error = " << e << std::endl ;
    multitest < dim , tuple_type , ntypes - 1 >() () ;
  }
} ;

template < int dim ,
           typename tuple_type >
struct multitest < dim , tuple_type , 0 >
{
  void operator() ()
  {
  }
} ;

int main ( int argc , char * argv[] )
{
  std::cout << std::fixed << std::showpos << std::showpoint
            << std::setprecision(18);
  std::cerr << std::fixed << std::showpos << std::showpoint
            << std::setprecision(18);

  int test_dim = 2 ;
  if ( argc > 1 )
    test_dim = std::atoi ( argv[1] ) ;
  if ( test_dim > 4 )
    test_dim = 4 ;
  
  std::cout << "testing with " << test_dim << " dimensions" << std::endl ;
  
//   std::cout << "dvtype<float> "
//             << vspline::dvtype < float , std::true_type > :: size
//             << " "
//             << vspline::dvtype < float , std::false_type > :: size
//             << std::endl ;
//             
//   std::cout << "vtraits<float> "
//             << vspline::vector_traits < float > :: isv :: value
//             << " "
//             << vspline::vector_traits < float > :: size
//             << std::endl ;
            
  typedef std::tuple <
                       vigra::TinyVector < double , 1 > ,
                       vigra::TinyVector < float , 1 > ,
                       vigra::TinyVector < double , 2 > ,
                       vigra::TinyVector < float , 2 > ,
                       vigra::TinyVector < long double , 3 > ,
                       vigra::TinyVector < double , 3 > ,
                       vigra::TinyVector < float , 3 >  ,
                       vigra::RGBValue<long double> ,
                       vigra::RGBValue<double> ,
                       vigra::RGBValue<float,2,0,1> ,
                       std::complex < long double > ,
                       std::complex < double > ,
                       std::complex < float > ,
                       long double ,
                       double ,
                       float
 > tuple_type ;
 
 switch ( test_dim )
 {
   case 1:
    multitest < 1 , tuple_type >() () ;
    break ;
   case 2:
    multitest < 2 , tuple_type >() () ;
    break ;
//    case 3:
//     multitest < 3 , tuple_type >() () ;
//     break ;
//    case 4:
//     multitest < 4 , tuple_type >() () ;
//     break ;
   default:
     break ;
 }
 std::cout << "terminating" << std::endl ;
}

