/*
 *  This file is part of X-File Manager XFM
 *  ----------------------------------------------------------------------
  FmDelete.c

  (c) Simon Marlow 1990-92
  (c) Albert Graef 1994

  modified 2004,2006,2007 by Bernhard R. Link (see Changelog)

  Functions for implementing the delete and empty operations
 *  ----------------------------------------------------------------------
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.

 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.

 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <xfmconfig.h>

#include <assert.h>
#include <X11/Intrinsic.h>

#include "global.h"

#include "Fm.h"

void deleteItems(UNUSED(Widget w), FileWindowRec *fw,
                 UNUSED(XtPointer call_data))
{
  char error_string[0xff];
  int i, n_deleted = 0;
  Boolean aborted;

  assert( fw != NULL );

  if (!fw->n_selections) return;

  if (chdir(fw->directory)) {
    sysError("System error:");
    return;
  }

  freeze = True;

  if (resources.confirm_deletes) {
    sprintf(error_string, "Deleting %d item%s from", fw->n_selections,
	    fw->n_selections > 1 ? "s" : "" );
    if (!fwConfirm(fw, error_string, fw->directory, "", NULL))
      goto out;
  }

  for (i=0; i < fw->n_files; i++)
    if (fw->files[i]->selected) {
      if (!fw->files[i]->info.symlink && S_ISDIR(fw->files[i]->stats.st_mode)) {
	if (!strcmp(fw->files[i]->name, ".") ||
	    !strcmp(fw->files[i]->name, "..")) {
	  error("Cannot delete . or ..", "");
	  continue;
	}
	/* first try to only delete the folder, do not delete recursively
	 * or even ask for that if it is already empty */
	if (rmdir(fw->files[i]->name)==0) {
	  n_deleted++;
	  continue;
	}
	if (resources.confirm_delete_folder &&
		 !fwConfirm(fw, "Do you REALLY wish to delete folder",
			  fw->files[i]->name,
			  "and ALL items contained in it?", &aborted)) {
	  if (aborted)
	    break;
          else
	    continue;
	}
      }
      if (rdel(fw->files[i]->name)) {
	sysErrorFmt("Error deleting %s:", fw->files[i]->name);
      } else
	n_deleted++;
    }

  if (n_deleted) {
    markForUpdate(fw->directory);
    intUpdate();
  }

 out:
  freeze = False;
}

/*---------------------------------------------------------------------------*/

void emptyDir(UNUSED(Widget w), FileWindowRec *fw, UNUSED(XtPointer call_data))
{
  char error_string[0xff];
  int i, n_deleted = 0;
  Boolean aborted;

  if (chdir(fw->directory)) {
    sysError("System error:");
    return;
  }

  freeze = True;

  if (resources.confirm_deletes) {
    int n_files = fw->display_type == Tree?fw->n_files-2:fw->n_files-1;

    sprintf(error_string, "Deleting %d item%s from", n_files,
	    n_files > 1 ? "s" : "" );
    if (!fwConfirm(fw, error_string, fw->directory, "", NULL))
      goto out;
  }

  for (i=0; i < fw->n_files; i++)
    if (strcmp(fw->files[i]->name, ".") && strcmp(fw->files[i]->name, "..")) {
      if (!fw->files[i]->info.symlink && S_ISDIR(fw->files[i]->stats.st_mode) &&
	  resources.confirm_delete_folder &&
	  !fwConfirm(fw, "Do you REALLY wish to delete folder", fw->files[i]->name,
		   "and ALL items contained in it?", &aborted)) {
	if (aborted)
	  break;
        else
	  continue;
      }
      if (rdel(fw->files[i]->name)) {
	sysErrorFmt("Error deleting %s:", fw->files[i]->name);
      } else
	n_deleted++;
    }

  if (n_deleted) {
    markForUpdate(fw->directory);
    intUpdate();
  }

 out:
  freeze = False;
}
