// SPDX-License-Identifier: GPL-2.0
/*
 * Copyright (C) 2021 MediaTek Inc.
 *
 * Author: Guodong Liu <guodong.liu@mediatek.com>
 *
 */

#include "pinctrl-mtk-mt8186.h"
#include "pinctrl-paris.h"

/* MT8186 have multiple bases to program pin configuration listed as the below:
 * iocfg[0]:0x10005000, iocfg[1]:0x10002000, iocfg[2]:0x10002200,
 * iocfg[3]:0x10002400, iocfg[4]:0x10002600, iocfg[5]:0x10002800,
 * iocfg[6]:0x10002C00.
 * _i_based could be used to indicate what base the pin should be mapped into.
 */

#define PIN_FIELD_BASE(s_pin, e_pin, i_base, s_addr, x_addrs, s_bit, x_bits) \
	PIN_FIELD_CALC(s_pin, e_pin, i_base, s_addr, x_addrs, s_bit, x_bits, 32, 0)

#define PINS_FIELD_BASE(s_pin, e_pin, i_base, s_addr, x_addrs, s_bit, x_bits) \
	PIN_FIELD_CALC(s_pin, e_pin, i_base, s_addr, x_addrs, s_bit, x_bits, 32, 1)

static const struct mtk_pin_field_calc mt8186_pin_mode_range[] = {
	PIN_FIELD(0, 184, 0x300, 0x10, 0, 4),
};

static const struct mtk_pin_field_calc mt8186_pin_dir_range[] = {
	PIN_FIELD(0, 184, 0x0, 0x10, 0, 1),
};

static const struct mtk_pin_field_calc mt8186_pin_di_range[] = {
	PIN_FIELD(0, 184, 0x200, 0x10, 0, 1),
};

static const struct mtk_pin_field_calc mt8186_pin_do_range[] = {
	PIN_FIELD(0, 184, 0x100, 0x10, 0, 1),
};

static const struct mtk_pin_field_calc mt8186_pin_ies_range[] = {
	PIN_FIELD_BASE(0, 0, 6, 0x0030, 0x10, 13, 1),
	PIN_FIELD_BASE(1, 1, 6, 0x0030, 0x10, 14, 1),
	PIN_FIELD_BASE(2, 2, 6, 0x0030, 0x10, 17, 1),
	PIN_FIELD_BASE(3, 3, 6, 0x0030, 0x10, 18, 1),
	PIN_FIELD_BASE(4, 4, 6, 0x0030, 0x10, 19, 1),
	PIN_FIELD_BASE(5, 5, 6, 0x0030, 0x10, 20, 1),
	PIN_FIELD_BASE(6, 6, 4, 0x0020, 0x10, 19, 1),
	PIN_FIELD_BASE(7, 7, 4, 0x0020, 0x10, 20, 1),
	PIN_FIELD_BASE(8, 8, 4, 0x0020, 0x10, 21, 1),
	PIN_FIELD_BASE(9, 9, 4, 0x0020, 0x10, 22, 1),
	PIN_FIELD_BASE(10, 10, 4, 0x0020, 0x10, 16, 1),
	PIN_FIELD_BASE(11, 11, 4, 0x0020, 0x10, 17, 1),
	PIN_FIELD_BASE(12, 12, 4, 0x0020, 0x10, 18, 1),
	PIN_FIELD_BASE(13, 13, 3, 0x0040, 0x10, 0, 1),
	PIN_FIELD_BASE(14, 14, 3, 0x0040, 0x10, 1, 1),
	PIN_FIELD_BASE(15, 15, 6, 0x0030, 0x10, 15, 1),
	PIN_FIELD_BASE(16, 16, 6, 0x0030, 0x10, 16, 1),
	PIN_FIELD_BASE(17, 17, 5, 0x0050, 0x10, 9, 1),
	PIN_FIELD_BASE(18, 18, 5, 0x0050, 0x10, 10, 1),
	PIN_FIELD_BASE(19, 19, 5, 0x0050, 0x10, 3, 1),
	PIN_FIELD_BASE(20, 20, 5, 0x0050, 0x10, 6, 1),
	PIN_FIELD_BASE(21, 21, 5, 0x0050, 0x10, 4, 1),
	PIN_FIELD_BASE(22, 22, 5, 0x0050, 0x10, 7, 1),
	PIN_FIELD_BASE(23, 23, 5, 0x0050, 0x10, 5, 1),
	PIN_FIELD_BASE(24, 24, 5, 0x0050, 0x10, 8, 1),
	PIN_FIELD_BASE(25, 25, 5, 0x0050, 0x10, 18, 1),
	PIN_FIELD_BASE(26, 26, 5, 0x0050, 0x10, 15, 1),
	PIN_FIELD_BASE(27, 27, 5, 0x0050, 0x10, 17, 1),
	PIN_FIELD_BASE(28, 28, 5, 0x0050, 0x10, 16, 1),
	PIN_FIELD_BASE(29, 29, 6, 0x0030, 0x10, 0, 1),
	PIN_FIELD_BASE(30, 30, 6, 0x0030, 0x10, 1, 1),
	PIN_FIELD_BASE(31, 31, 6, 0x0030, 0x10, 2, 1),
	PIN_FIELD_BASE(32, 32, 2, 0x0040, 0x10, 25, 1),
	PIN_FIELD_BASE(33, 33, 2, 0x0040, 0x10, 27, 1),
	PIN_FIELD_BASE(34, 34, 2, 0x0040, 0x10, 26, 1),
	PIN_FIELD_BASE(35, 35, 2, 0x0040, 0x10, 28, 1),
	PIN_FIELD_BASE(36, 36, 2, 0x0040, 0x10, 9, 1),
	PIN_FIELD_BASE(37, 37, 2, 0x0040, 0x10, 10, 1),
	PIN_FIELD_BASE(38, 38, 2, 0x0040, 0x10, 12, 1),
	PIN_FIELD_BASE(39, 39, 2, 0x0040, 0x10, 11, 1),
	PIN_FIELD_BASE(40, 40, 2, 0x0040, 0x10, 13, 1),
	PIN_FIELD_BASE(41, 41, 2, 0x0040, 0x10, 14, 1),
	PIN_FIELD_BASE(42, 42, 2, 0x0040, 0x10, 16, 1),
	PIN_FIELD_BASE(43, 43, 2, 0x0040, 0x10, 15, 1),
	PIN_FIELD_BASE(44, 44, 5, 0x0060, 0x10, 0, 1),
	PIN_FIELD_BASE(45, 45, 5, 0x0060, 0x10, 1, 1),
	PIN_FIELD_BASE(46, 46, 5, 0x0060, 0x10, 3, 1),
	PIN_FIELD_BASE(47, 47, 5, 0x0060, 0x10, 2, 1),
	PIN_FIELD_BASE(48, 48, 2, 0x0040, 0x10, 17, 1),
	PIN_FIELD_BASE(49, 49, 2, 0x0040, 0x10, 18, 1),
	PIN_FIELD_BASE(50, 50, 2, 0x0040, 0x10, 20, 1),
	PIN_FIELD_BASE(51, 51, 2, 0x0040, 0x10, 19, 1),
	PIN_FIELD_BASE(52, 52, 3, 0x0040, 0x10, 18, 1),
	PIN_FIELD_BASE(53, 53, 3, 0x0040, 0x10, 19, 1),
	PIN_FIELD_BASE(54, 54, 3, 0x0040, 0x10, 21, 1),
	PIN_FIELD_BASE(55, 55, 3, 0x0040, 0x10, 20, 1),
	PIN_FIELD_BASE(56, 56, 5, 0x0050, 0x10, 12, 1),
	PIN_FIELD_BASE(57, 57, 5, 0x0050, 0x10, 11, 1),
	PIN_FIELD_BASE(58, 58, 5, 0x0050, 0x10, 13, 1),
	PIN_FIELD_BASE(59, 59, 5, 0x0050, 0x10, 14, 1),
	PIN_FIELD_BASE(60, 60, 3, 0x0040, 0x10, 27, 1),
	PIN_FIELD_BASE(61, 61, 3, 0x0040, 0x10, 22, 1),
	PIN_FIELD_BASE(62, 62, 3, 0x0040, 0x10, 28, 1),
	PIN_FIELD_BASE(63, 63, 3, 0x0040, 0x10, 23, 1),
	PIN_FIELD_BASE(64, 64, 3, 0x0040, 0x10, 24, 1),
	PIN_FIELD_BASE(65, 65, 3, 0x0040, 0x10, 25, 1),
	PIN_FIELD_BASE(66, 66, 3, 0x0040, 0x10, 26, 1),
	PIN_FIELD_BASE(67, 67, 1, 0x0050, 0x10, 18, 1),
	PIN_FIELD_BASE(68, 68, 1, 0x0050, 0x10, 8, 1),
	PIN_FIELD_BASE(69, 69, 1, 0x0050, 0x10, 9, 1),
	PIN_FIELD_BASE(70, 70, 1, 0x0050, 0x10, 19, 1),
	PIN_FIELD_BASE(71, 71, 1, 0x0050, 0x10, 10, 1),
	PIN_FIELD_BASE(72, 72, 1, 0x0050, 0x10, 11, 1),
	PIN_FIELD_BASE(73, 73, 1, 0x0050, 0x10, 12, 1),
	PIN_FIELD_BASE(74, 74, 1, 0x0050, 0x10, 13, 1),
	PIN_FIELD_BASE(75, 75, 1, 0x0050, 0x10, 14, 1),
	PIN_FIELD_BASE(76, 76, 1, 0x0050, 0x10, 15, 1),
	PIN_FIELD_BASE(77, 77, 1, 0x0050, 0x10, 16, 1),
	PIN_FIELD_BASE(78, 78, 1, 0x0050, 0x10, 17, 1),
	PIN_FIELD_BASE(79, 79, 5, 0x0050, 0x10, 24, 1),
	PIN_FIELD_BASE(80, 80, 5, 0x0050, 0x10, 25, 1),
	PIN_FIELD_BASE(81, 81, 5, 0x0050, 0x10, 26, 1),
	PIN_FIELD_BASE(82, 82, 5, 0x0050, 0x10, 27, 1),
	PIN_FIELD_BASE(83, 83, 6, 0x0030, 0x10, 3, 1),
	PIN_FIELD_BASE(84, 84, 3, 0x0040, 0x10, 4, 1),
	PIN_FIELD_BASE(85, 85, 3, 0x0040, 0x10, 5, 1),
	PIN_FIELD_BASE(86, 86, 3, 0x0040, 0x10, 6, 1),
	PIN_FIELD_BASE(87, 87, 3, 0x0040, 0x10, 7, 1),
	PIN_FIELD_BASE(88, 88, 3, 0x0040, 0x10, 8, 1),
	PIN_FIELD_BASE(89, 89, 3, 0x0040, 0x10, 9, 1),
	PIN_FIELD_BASE(90, 90, 3, 0x0040, 0x10, 2, 1),
	PIN_FIELD_BASE(91, 91, 3, 0x0040, 0x10, 29, 1),
	PIN_FIELD_BASE(92, 92, 3, 0x0040, 0x10, 31, 1),
	PIN_FIELD_BASE(93, 93, 3, 0x0040, 0x10, 3, 1),
	PIN_FIELD_BASE(94, 94, 3, 0x0040, 0x10, 30, 1),
	PIN_FIELD_BASE(95, 95, 3, 0x0050, 0x10, 0, 1),
	PIN_FIELD_BASE(96, 96, 2, 0x0040, 0x10, 1, 1),
	PIN_FIELD_BASE(97, 97, 2, 0x0040, 0x10, 0, 1),
	PIN_FIELD_BASE(98, 98, 2, 0x0040, 0x10, 2, 1),
	PIN_FIELD_BASE(99, 99, 4, 0x0020, 0x10, 14, 1),
	PIN_FIELD_BASE(100, 100, 4, 0x0020, 0x10, 15, 1),
	PIN_FIELD_BASE(101, 101, 4, 0x0020, 0x10, 13, 1),
	PIN_FIELD_BASE(102, 102, 4, 0x0020, 0x10, 12, 1),
	PIN_FIELD_BASE(103, 103, 4, 0x0020, 0x10, 0, 1),
	PIN_FIELD_BASE(104, 104, 4, 0x0020, 0x10, 1, 1),
	PIN_FIELD_BASE(105, 105, 4, 0x0020, 0x10, 4, 1),
	PIN_FIELD_BASE(106, 106, 4, 0x0020, 0x10, 5, 1),
	PIN_FIELD_BASE(107, 107, 4, 0x0020, 0x10, 6, 1),
	PIN_FIELD_BASE(108, 108, 4, 0x0020, 0x10, 7, 1),
	PIN_FIELD_BASE(109, 109, 4, 0x0020, 0x10, 8, 1),
	PIN_FIELD_BASE(110, 110, 4, 0x0020, 0x10, 9, 1),
	PIN_FIELD_BASE(111, 111, 4, 0x0020, 0x10, 10, 1),
	PIN_FIELD_BASE(112, 112, 4, 0x0020, 0x10, 11, 1),
	PIN_FIELD_BASE(113, 113, 4, 0x0020, 0x10, 2, 1),
	PIN_FIELD_BASE(114, 114, 4, 0x0020, 0x10, 3, 1),
	PIN_FIELD_BASE(115, 115, 3, 0x0040, 0x10, 10, 1),
	PIN_FIELD_BASE(116, 116, 3, 0x0040, 0x10, 13, 1),
	PIN_FIELD_BASE(117, 117, 3, 0x0040, 0x10, 11, 1),
	PIN_FIELD_BASE(118, 118, 3, 0x0040, 0x10, 12, 1),
	PIN_FIELD_BASE(119, 119, 5, 0x0050, 0x10, 22, 1),
	PIN_FIELD_BASE(120, 120, 5, 0x0050, 0x10, 19, 1),
	PIN_FIELD_BASE(121, 121, 5, 0x0050, 0x10, 20, 1),
	PIN_FIELD_BASE(122, 122, 5, 0x0050, 0x10, 21, 1),
	PIN_FIELD_BASE(123, 123, 5, 0x0050, 0x10, 23, 1),
	PIN_FIELD_BASE(124, 124, 5, 0x0050, 0x10, 0, 1),
	PIN_FIELD_BASE(125, 125, 5, 0x0050, 0x10, 1, 1),
	PIN_FIELD_BASE(126, 126, 5, 0x0050, 0x10, 2, 1),
	PIN_FIELD_BASE(127, 127, 3, 0x0040, 0x10, 14, 1),
	PIN_FIELD_BASE(128, 128, 3, 0x0040, 0x10, 16, 1),
	PIN_FIELD_BASE(129, 129, 5, 0x0050, 0x10, 28, 1),
	PIN_FIELD_BASE(130, 130, 5, 0x0050, 0x10, 30, 1),
	PIN_FIELD_BASE(131, 131, 5, 0x0050, 0x10, 29, 1),
	PIN_FIELD_BASE(132, 132, 5, 0x0050, 0x10, 31, 1),
	PIN_FIELD_BASE(133, 133, 1, 0x0050, 0x10, 21, 1),
	PIN_FIELD_BASE(134, 134, 1, 0x0050, 0x10, 24, 1),
	PIN_FIELD_BASE(135, 135, 6, 0x0030, 0x10, 21, 1),
	PIN_FIELD_BASE(136, 136, 6, 0x0030, 0x10, 24, 1),
	PIN_FIELD_BASE(137, 137, 1, 0x0050, 0x10, 22, 1),
	PIN_FIELD_BASE(138, 138, 1, 0x0050, 0x10, 25, 1),
	PIN_FIELD_BASE(139, 139, 2, 0x0040, 0x10, 7, 1),
	PIN_FIELD_BASE(140, 140, 2, 0x0040, 0x10, 8, 1),
	PIN_FIELD_BASE(141, 141, 3, 0x0040, 0x10, 15, 1),
	PIN_FIELD_BASE(142, 142, 3, 0x0040, 0x10, 17, 1),
	PIN_FIELD_BASE(143, 143, 6, 0x0030, 0x10, 22, 1),
	PIN_FIELD_BASE(144, 144, 6, 0x0030, 0x10, 25, 1),
	PIN_FIELD_BASE(145, 145, 6, 0x0030, 0x10, 23, 1),
	PIN_FIELD_BASE(146, 146, 6, 0x0030, 0x10, 26, 1),
	PIN_FIELD_BASE(147, 147, 4, 0x0020, 0x10, 23, 1),
	PIN_FIELD_BASE(148, 148, 4, 0x0020, 0x10, 24, 1),
	PIN_FIELD_BASE(149, 149, 4, 0x0020, 0x10, 25, 1),
	PIN_FIELD_BASE(150, 150, 4, 0x0020, 0x10, 26, 1),
	PIN_FIELD_BASE(151, 151, 4, 0x0020, 0x10, 27, 1),
	PIN_FIELD_BASE(152, 152, 4, 0x0020, 0x10, 28, 1),
	PIN_FIELD_BASE(153, 153, 4, 0x0020, 0x10, 29, 1),
	PIN_FIELD_BASE(154, 154, 4, 0x0020, 0x10, 30, 1),
	PIN_FIELD_BASE(155, 155, 4, 0x0020, 0x10, 31, 1),
	PIN_FIELD_BASE(156, 156, 4, 0x0030, 0x10, 0, 1),
	PIN_FIELD_BASE(157, 157, 2, 0x0040, 0x10, 4, 1),
	PIN_FIELD_BASE(158, 158, 2, 0x0040, 0x10, 3, 1),
	PIN_FIELD_BASE(159, 159, 2, 0x0040, 0x10, 6, 1),
	PIN_FIELD_BASE(160, 160, 2, 0x0040, 0x10, 5, 1),
	PIN_FIELD_BASE(161, 161, 2, 0x0040, 0x10, 23, 1),
	PIN_FIELD_BASE(162, 162, 2, 0x0040, 0x10, 24, 1),
	PIN_FIELD_BASE(163, 163, 1, 0x0050, 0x10, 23, 1),
	PIN_FIELD_BASE(164, 164, 1, 0x0050, 0x10, 20, 1),
	PIN_FIELD_BASE(165, 165, 1, 0x0050, 0x10, 26, 1),
	PIN_FIELD_BASE(166, 166, 1, 0x0050, 0x10, 1, 1),
	PIN_FIELD_BASE(167, 167, 1, 0x0050, 0x10, 7, 1),
	PIN_FIELD_BASE(168, 168, 1, 0x0050, 0x10, 4, 1),
	PIN_FIELD_BASE(169, 169, 1, 0x0050, 0x10, 5, 1),
	PIN_FIELD_BASE(170, 170, 1, 0x0050, 0x10, 0, 1),
	PIN_FIELD_BASE(171, 171, 1, 0x0050, 0x10, 6, 1),
	PIN_FIELD_BASE(172, 172, 1, 0x0050, 0x10, 2, 1),
	PIN_FIELD_BASE(173, 173, 1, 0x0050, 0x10, 3, 1),
	PIN_FIELD_BASE(174, 174, 6, 0x0030, 0x10, 7, 1),
	PIN_FIELD_BASE(175, 175, 6, 0x0030, 0x10, 8, 1),
	PIN_FIELD_BASE(176, 176, 6, 0x0030, 0x10, 4, 1),
	PIN_FIELD_BASE(177, 177, 6, 0x0030, 0x10, 5, 1),
	PIN_FIELD_BASE(178, 178, 6, 0x0030, 0x10, 6, 1),
	PIN_FIELD_BASE(179, 179, 6, 0x0030, 0x10, 9, 1),
	PIN_FIELD_BASE(180, 180, 6, 0x0030, 0x10, 10, 1),
	PIN_FIELD_BASE(181, 181, 6, 0x0030, 0x10, 11, 1),
	PIN_FIELD_BASE(182, 182, 6, 0x0030, 0x10, 12, 1),
	PIN_FIELD_BASE(183, 183, 2, 0x0040, 0x10, 21, 1),
	PIN_FIELD_BASE(184, 184, 2, 0x0040, 0x10, 22, 1),
};

static const struct mtk_pin_field_calc mt8186_pin_smt_range[] = {
	PIN_FIELD_BASE(0, 0, 6, 0x0080, 0x10, 0, 1),
	PIN_FIELD_BASE(1, 1, 6, 0x0080, 0x10, 0, 1),
	PIN_FIELD_BASE(2, 2, 6, 0x0080, 0x10, 0, 1),
	PIN_FIELD_BASE(3, 3, 6, 0x0080, 0x10, 0, 1),
	PIN_FIELD_BASE(4, 4, 6, 0x0080, 0x10, 1, 1),
	PIN_FIELD_BASE(5, 5, 6, 0x0080, 0x10, 1, 1),
	PIN_FIELD_BASE(6, 6, 4, 0x0090, 0x10, 0, 1),
	PIN_FIELD_BASE(7, 7, 4, 0x0090, 0x10, 1, 1),
	PIN_FIELD_BASE(8, 8, 4, 0x0090, 0x10, 1, 1),
	PIN_FIELD_BASE(9, 9, 4, 0x0090, 0x10, 1, 1),
	PIN_FIELD_BASE(10, 10, 4, 0x0090, 0x10, 0, 1),
	PIN_FIELD_BASE(11, 11, 4, 0x0090, 0x10, 0, 1),
	PIN_FIELD_BASE(12, 12, 4, 0x0090, 0x10, 0, 1),
	PIN_FIELD_BASE(13, 13, 3, 0x00e0, 0x10, 11, 1),
	PIN_FIELD_BASE(14, 14, 3, 0x00e0, 0x10, 11, 1),
	PIN_FIELD_BASE(15, 15, 6, 0x0080, 0x10, 2, 1),
	PIN_FIELD_BASE(16, 16, 6, 0x0080, 0x10, 2, 1),
	PIN_FIELD_BASE(17, 17, 5, 0x00e0, 0x10, 3, 1),
	PIN_FIELD_BASE(18, 18, 5, 0x00e0, 0x10, 7, 1),
	PIN_FIELD_BASE(19, 19, 5, 0x00e0, 0x10, 6, 1),
	PIN_FIELD_BASE(20, 20, 5, 0x00e0, 0x10, 5, 1),
	PIN_FIELD_BASE(21, 21, 5, 0x00e0, 0x10, 6, 1),
	PIN_FIELD_BASE(22, 22, 5, 0x00e0, 0x10, 5, 1),
	PIN_FIELD_BASE(23, 23, 5, 0x00e0, 0x10, 7, 1),
	PIN_FIELD_BASE(24, 24, 5, 0x00e0, 0x10, 10, 1),
	PIN_FIELD_BASE(25, 25, 5, 0x00e0, 0x10, 7, 1),
	PIN_FIELD_BASE(26, 26, 5, 0x00e0, 0x10, 6, 1),
	PIN_FIELD_BASE(27, 27, 5, 0x00e0, 0x10, 6, 1),
	PIN_FIELD_BASE(28, 28, 5, 0x00e0, 0x10, 7, 1),
	PIN_FIELD_BASE(29, 29, 6, 0x0080, 0x10, 1, 1),
	PIN_FIELD_BASE(30, 30, 6, 0x0080, 0x10, 1, 1),
	PIN_FIELD_BASE(31, 31, 6, 0x0080, 0x10, 2, 1),
	PIN_FIELD_BASE(32, 32, 2, 0x0090, 0x10, 2, 1),
	PIN_FIELD_BASE(33, 33, 2, 0x0090, 0x10, 2, 1),
	PIN_FIELD_BASE(34, 34, 2, 0x0090, 0x10, 1, 1),
	PIN_FIELD_BASE(35, 35, 2, 0x0090, 0x10, 1, 1),
	PIN_FIELD_BASE(36, 36, 2, 0x0090, 0x10, 3, 1),
	PIN_FIELD_BASE(37, 37, 2, 0x0090, 0x10, 3, 1),
	PIN_FIELD_BASE(38, 38, 2, 0x0090, 0x10, 4, 1),
	PIN_FIELD_BASE(39, 39, 2, 0x0090, 0x10, 3, 1),
	PIN_FIELD_BASE(40, 40, 2, 0x0090, 0x10, 5, 1),
	PIN_FIELD_BASE(41, 41, 2, 0x0090, 0x10, 4, 1),
	PIN_FIELD_BASE(42, 42, 2, 0x0090, 0x10, 4, 1),
	PIN_FIELD_BASE(43, 43, 2, 0x0090, 0x10, 4, 1),
	PIN_FIELD_BASE(44, 44, 5, 0x00e0, 0x10, 9, 1),
	PIN_FIELD_BASE(45, 45, 5, 0x00e0, 0x10, 8, 1),
	PIN_FIELD_BASE(46, 46, 5, 0x00e0, 0x10, 8, 1),
	PIN_FIELD_BASE(47, 47, 5, 0x00e0, 0x10, 9, 1),
	PIN_FIELD_BASE(48, 48, 2, 0x0090, 0x10, 5, 1),
	PIN_FIELD_BASE(49, 49, 2, 0x0090, 0x10, 5, 1),
	PIN_FIELD_BASE(50, 50, 2, 0x0090, 0x10, 5, 1),
	PIN_FIELD_BASE(51, 51, 2, 0x0090, 0x10, 6, 1),
	PIN_FIELD_BASE(52, 52, 3, 0x00e0, 0x10, 9, 1),
	PIN_FIELD_BASE(53, 53, 3, 0x00e0, 0x10, 10, 1),
	PIN_FIELD_BASE(54, 54, 3, 0x00e0, 0x10, 9, 1),
	PIN_FIELD_BASE(55, 55, 3, 0x00e0, 0x10, 9, 1),
	PIN_FIELD_BASE(56, 56, 5, 0x00e0, 0x10, 8, 1),
	PIN_FIELD_BASE(57, 57, 5, 0x00e0, 0x10, 9, 1),
	PIN_FIELD_BASE(58, 58, 5, 0x00e0, 0x10, 9, 1),
	PIN_FIELD_BASE(59, 59, 5, 0x00e0, 0x10, 8, 1),
	PIN_FIELD_BASE(60, 60, 3, 0x00e0, 0x10, 6, 1),
	PIN_FIELD_BASE(61, 61, 3, 0x00e0, 0x10, 6, 1),
	PIN_FIELD_BASE(62, 62, 3, 0x00e0, 0x10, 6, 1),
	PIN_FIELD_BASE(63, 63, 3, 0x00e0, 0x10, 6, 1),
	PIN_FIELD_BASE(64, 64, 3, 0x00e0, 0x10, 7, 1),
	PIN_FIELD_BASE(65, 65, 3, 0x00e0, 0x10, 7, 1),
	PIN_FIELD_BASE(66, 66, 3, 0x00e0, 0x10, 7, 1),
	PIN_FIELD_BASE(67, 67, 1, 0x00f0, 0x10, 10, 1),
	PIN_FIELD_BASE(68, 68, 1, 0x00f0, 0x10, 0, 1),
	PIN_FIELD_BASE(69, 69, 1, 0x00f0, 0x10, 1, 1),
	PIN_FIELD_BASE(70, 70, 1, 0x00f0, 0x10, 11, 1),
	PIN_FIELD_BASE(71, 71, 1, 0x00f0, 0x10, 2, 1),
	PIN_FIELD_BASE(72, 72, 1, 0x00f0, 0x10, 3, 1),
	PIN_FIELD_BASE(73, 73, 1, 0x00f0, 0x10, 4, 1),
	PIN_FIELD_BASE(74, 74, 1, 0x00f0, 0x10, 5, 1),
	PIN_FIELD_BASE(75, 75, 1, 0x00f0, 0x10, 6, 1),
	PIN_FIELD_BASE(76, 76, 1, 0x00f0, 0x10, 7, 1),
	PIN_FIELD_BASE(77, 77, 1, 0x00f0, 0x10, 8, 1),
	PIN_FIELD_BASE(78, 78, 1, 0x00f0, 0x10, 9, 1),
	PIN_FIELD_BASE(79, 79, 5, 0x00e0, 0x10, 5, 1),
	PIN_FIELD_BASE(80, 80, 5, 0x00e0, 0x10, 5, 1),
	PIN_FIELD_BASE(81, 81, 5, 0x00e0, 0x10, 4, 1),
	PIN_FIELD_BASE(82, 82, 5, 0x00e0, 0x10, 4, 1),
	PIN_FIELD_BASE(83, 83, 6, 0x0080, 0x10, 2, 1),
	PIN_FIELD_BASE(84, 84, 3, 0x00e0, 0x10, 0, 1),
	PIN_FIELD_BASE(85, 85, 3, 0x00e0, 0x10, 1, 1),
	PIN_FIELD_BASE(86, 86, 3, 0x00e0, 0x10, 2, 1),
	PIN_FIELD_BASE(87, 87, 3, 0x00e0, 0x10, 3, 1),
	PIN_FIELD_BASE(88, 88, 3, 0x00e0, 0x10, 4, 1),
	PIN_FIELD_BASE(89, 89, 3, 0x00e0, 0x10, 5, 1),
	PIN_FIELD_BASE(90, 90, 3, 0x00e0, 0x10, 7, 1),
	PIN_FIELD_BASE(91, 91, 3, 0x00e0, 0x10, 8, 1),
	PIN_FIELD_BASE(92, 92, 3, 0x00e0, 0x10, 8, 1),
	PIN_FIELD_BASE(93, 93, 3, 0x00e0, 0x10, 8, 1),
	PIN_FIELD_BASE(94, 94, 3, 0x00e0, 0x10, 8, 1),
	PIN_FIELD_BASE(95, 95, 3, 0x00e0, 0x10, 9, 1),
	PIN_FIELD_BASE(96, 96, 2, 0x0090, 0x10, 3, 1),
	PIN_FIELD_BASE(97, 97, 2, 0x0090, 0x10, 2, 1),
	PIN_FIELD_BASE(98, 98, 2, 0x0090, 0x10, 1, 1),
	PIN_FIELD_BASE(99, 99, 4, 0x0090, 0x10, 4, 1),
	PIN_FIELD_BASE(100, 100, 4, 0x0090, 0x10, 4, 1),
	PIN_FIELD_BASE(101, 101, 4, 0x0090, 0x10, 4, 1),
	PIN_FIELD_BASE(102, 102, 4, 0x0090, 0x10, 5, 1),
	PIN_FIELD_BASE(103, 103, 4, 0x0090, 0x10, 5, 1),
	PIN_FIELD_BASE(104, 104, 4, 0x0090, 0x10, 5, 1),
	PIN_FIELD_BASE(105, 105, 4, 0x0090, 0x10, 5, 1),
	PIN_FIELD_BASE(106, 106, 4, 0x0090, 0x10, 6, 1),
	PIN_FIELD_BASE(107, 107, 4, 0x0090, 0x10, 6, 1),
	PIN_FIELD_BASE(108, 108, 4, 0x0090, 0x10, 6, 1),
	PIN_FIELD_BASE(109, 109, 4, 0x0090, 0x10, 6, 1),
	PIN_FIELD_BASE(110, 110, 4, 0x0090, 0x10, 7, 1),
	PIN_FIELD_BASE(111, 111, 4, 0x0090, 0x10, 7, 1),
	PIN_FIELD_BASE(112, 112, 4, 0x0090, 0x10, 7, 1),
	PIN_FIELD_BASE(113, 113, 4, 0x0090, 0x10, 7, 1),
	PIN_FIELD_BASE(114, 114, 4, 0x0090, 0x10, 8, 1),
	PIN_FIELD_BASE(115, 115, 3, 0x00e0, 0x10, 10, 1),
	PIN_FIELD_BASE(116, 116, 3, 0x00e0, 0x10, 11, 1),
	PIN_FIELD_BASE(117, 117, 3, 0x00e0, 0x10, 10, 1),
	PIN_FIELD_BASE(118, 118, 3, 0x00e0, 0x10, 10, 1),
	PIN_FIELD_BASE(119, 119, 5, 0x00e0, 0x10, 4, 1),
	PIN_FIELD_BASE(120, 120, 5, 0x00e0, 0x10, 4, 1),
	PIN_FIELD_BASE(121, 121, 5, 0x00e0, 0x10, 3, 1),
	PIN_FIELD_BASE(122, 122, 5, 0x00e0, 0x10, 3, 1),
	PIN_FIELD_BASE(123, 123, 5, 0x00e0, 0x10, 3, 1),
	PIN_FIELD_BASE(124, 124, 5, 0x00e0, 0x10, 0, 1),
	PIN_FIELD_BASE(125, 125, 5, 0x00e0, 0x10, 1, 1),
	PIN_FIELD_BASE(126, 126, 5, 0x00e0, 0x10, 2, 1),
	PIN_FIELD_BASE(127, 127, 3, 0x00e0, 0x10, 12, 1),
	PIN_FIELD_BASE(128, 128, 3, 0x00e0, 0x10, 14, 1),
	PIN_FIELD_BASE(129, 129, 5, 0x00e0, 0x10, 11, 1),
	PIN_FIELD_BASE(130, 130, 5, 0x00e0, 0x10, 13, 1),
	PIN_FIELD_BASE(131, 131, 5, 0x00e0, 0x10, 12, 1),
	PIN_FIELD_BASE(132, 132, 5, 0x00e0, 0x10, 14, 1),
	PIN_FIELD_BASE(133, 133, 1, 0x00f0, 0x10, 15, 1),
	PIN_FIELD_BASE(134, 134, 1, 0x00f0, 0x10, 17, 1),
	PIN_FIELD_BASE(135, 135, 6, 0x0080, 0x10, 6, 1),
	PIN_FIELD_BASE(136, 136, 6, 0x0080, 0x10, 9, 1),
	PIN_FIELD_BASE(137, 137, 1, 0x00f0, 0x10, 16, 1),
	PIN_FIELD_BASE(138, 138, 1, 0x00f0, 0x10, 18, 1),
	PIN_FIELD_BASE(139, 139, 2, 0x0090, 0x10, 7, 1),
	PIN_FIELD_BASE(140, 140, 2, 0x0090, 0x10, 8, 1),
	PIN_FIELD_BASE(141, 141, 3, 0x00e0, 0x10, 13, 1),
	PIN_FIELD_BASE(142, 142, 3, 0x00e0, 0x10, 15, 1),
	PIN_FIELD_BASE(143, 143, 6, 0x0080, 0x10, 7, 1),
	PIN_FIELD_BASE(144, 144, 6, 0x0080, 0x10, 10, 1),
	PIN_FIELD_BASE(145, 145, 6, 0x0080, 0x10, 8, 1),
	PIN_FIELD_BASE(146, 146, 6, 0x0080, 0x10, 11, 1),
	PIN_FIELD_BASE(147, 147, 4, 0x0090, 0x10, 2, 1),
	PIN_FIELD_BASE(148, 148, 4, 0x0090, 0x10, 2, 1),
	PIN_FIELD_BASE(149, 149, 4, 0x0090, 0x10, 2, 1),
	PIN_FIELD_BASE(150, 150, 4, 0x0090, 0x10, 2, 1),
	PIN_FIELD_BASE(151, 151, 4, 0x0090, 0x10, 3, 1),
	PIN_FIELD_BASE(152, 152, 4, 0x0090, 0x10, 1, 1),
	PIN_FIELD_BASE(153, 153, 4, 0x0090, 0x10, 3, 1),
	PIN_FIELD_BASE(154, 154, 4, 0x0090, 0x10, 3, 1),
	PIN_FIELD_BASE(155, 155, 4, 0x0090, 0x10, 3, 1),
	PIN_FIELD_BASE(156, 156, 4, 0x0090, 0x10, 4, 1),
	PIN_FIELD_BASE(157, 157, 2, 0x0090, 0x10, 0, 1),
	PIN_FIELD_BASE(158, 158, 2, 0x0090, 0x10, 0, 1),
	PIN_FIELD_BASE(159, 159, 2, 0x0090, 0x10, 0, 1),
	PIN_FIELD_BASE(160, 160, 2, 0x0090, 0x10, 0, 1),
	PIN_FIELD_BASE(161, 161, 2, 0x0090, 0x10, 2, 1),
	PIN_FIELD_BASE(162, 162, 2, 0x0090, 0x10, 1, 1),
	PIN_FIELD_BASE(163, 163, 1, 0x00f0, 0x10, 14, 1),
	PIN_FIELD_BASE(164, 164, 1, 0x00f0, 0x10, 12, 1),
	PIN_FIELD_BASE(165, 165, 1, 0x00f0, 0x10, 12, 1),
	PIN_FIELD_BASE(166, 166, 1, 0x00f0, 0x10, 13, 1),
	PIN_FIELD_BASE(167, 167, 1, 0x00f0, 0x10, 13, 1),
	PIN_FIELD_BASE(168, 168, 1, 0x00f0, 0x10, 12, 1),
	PIN_FIELD_BASE(169, 169, 1, 0x00f0, 0x10, 14, 1),
	PIN_FIELD_BASE(170, 170, 1, 0x00f0, 0x10, 13, 1),
	PIN_FIELD_BASE(171, 171, 1, 0x00f0, 0x10, 13, 1),
	PIN_FIELD_BASE(172, 172, 1, 0x00f0, 0x10, 14, 1),
	PIN_FIELD_BASE(173, 173, 1, 0x00f0, 0x10, 12, 1),
	PIN_FIELD_BASE(174, 174, 6, 0x0080, 0x10, 3, 1),
	PIN_FIELD_BASE(175, 175, 6, 0x0080, 0x10, 3, 1),
	PIN_FIELD_BASE(176, 176, 6, 0x0080, 0x10, 4, 1),
	PIN_FIELD_BASE(177, 177, 6, 0x0080, 0x10, 4, 1),
	PIN_FIELD_BASE(178, 178, 6, 0x0080, 0x10, 3, 1),
	PIN_FIELD_BASE(179, 179, 6, 0x0080, 0x10, 3, 1),
	PIN_FIELD_BASE(180, 180, 6, 0x0080, 0x10, 4, 1),
	PIN_FIELD_BASE(181, 181, 6, 0x0080, 0x10, 4, 1),
	PIN_FIELD_BASE(182, 182, 6, 0x0080, 0x10, 5, 1),
	PIN_FIELD_BASE(183, 183, 2, 0x0090, 0x10, 9, 1),
	PIN_FIELD_BASE(184, 184, 2, 0x0090, 0x10, 10, 1),
};

static const struct mtk_pin_field_calc mt8186_pin_pu_range[] = {
	PIN_FIELD_BASE(0, 0, 6, 0x0050, 0x10, 13, 1),
	PIN_FIELD_BASE(1, 1, 6, 0x0050, 0x10, 14, 1),
	PIN_FIELD_BASE(2, 2, 6, 0x0050, 0x10, 17, 1),
	PIN_FIELD_BASE(3, 3, 6, 0x0050, 0x10, 18, 1),
	PIN_FIELD_BASE(4, 4, 6, 0x0050, 0x10, 19, 1),
	PIN_FIELD_BASE(5, 5, 6, 0x0050, 0x10, 20, 1),
	PIN_FIELD_BASE(6, 6, 4, 0x0060, 0x10, 19, 1),
	PIN_FIELD_BASE(7, 7, 4, 0x0060, 0x10, 20, 1),
	PIN_FIELD_BASE(8, 8, 4, 0x0060, 0x10, 21, 1),
	PIN_FIELD_BASE(9, 9, 4, 0x0060, 0x10, 22, 1),
	PIN_FIELD_BASE(10, 10, 4, 0x0060, 0x10, 16, 1),
	PIN_FIELD_BASE(11, 11, 4, 0x0060, 0x10, 17, 1),
	PIN_FIELD_BASE(12, 12, 4, 0x0060, 0x10, 18, 1),
	PIN_FIELD_BASE(13, 13, 3, 0x0080, 0x10, 0, 1),
	PIN_FIELD_BASE(14, 14, 3, 0x0080, 0x10, 1, 1),
	PIN_FIELD_BASE(15, 15, 6, 0x0050, 0x10, 15, 1),
	PIN_FIELD_BASE(16, 16, 6, 0x0050, 0x10, 16, 1),
	PIN_FIELD_BASE(17, 17, 5, 0x0090, 0x10, 9, 1),
	PIN_FIELD_BASE(18, 18, 5, 0x0090, 0x10, 10, 1),
	PIN_FIELD_BASE(19, 19, 5, 0x0090, 0x10, 3, 1),
	PIN_FIELD_BASE(20, 20, 5, 0x0090, 0x10, 6, 1),
	PIN_FIELD_BASE(21, 21, 5, 0x0090, 0x10, 4, 1),
	PIN_FIELD_BASE(22, 22, 5, 0x0090, 0x10, 7, 1),
	PIN_FIELD_BASE(23, 23, 5, 0x0090, 0x10, 5, 1),
	PIN_FIELD_BASE(24, 24, 5, 0x0090, 0x10, 8, 1),
	PIN_FIELD_BASE(25, 25, 5, 0x0090, 0x10, 18, 1),
	PIN_FIELD_BASE(26, 26, 5, 0x0090, 0x10, 15, 1),
	PIN_FIELD_BASE(27, 27, 5, 0x0090, 0x10, 17, 1),
	PIN_FIELD_BASE(28, 28, 5, 0x0090, 0x10, 16, 1),
	PIN_FIELD_BASE(29, 29, 6, 0x0050, 0x10, 0, 1),
	PIN_FIELD_BASE(30, 30, 6, 0x0050, 0x10, 1, 1),
	PIN_FIELD_BASE(31, 31, 6, 0x0050, 0x10, 2, 1),
	PIN_FIELD_BASE(32, 32, 2, 0x0060, 0x10, 25, 1),
	PIN_FIELD_BASE(33, 33, 2, 0x0060, 0x10, 27, 1),
	PIN_FIELD_BASE(34, 34, 2, 0x0060, 0x10, 26, 1),
	PIN_FIELD_BASE(35, 35, 2, 0x0060, 0x10, 28, 1),
	PIN_FIELD_BASE(36, 36, 2, 0x0060, 0x10, 9, 1),
	PIN_FIELD_BASE(37, 37, 2, 0x0060, 0x10, 10, 1),
	PIN_FIELD_BASE(38, 38, 2, 0x0060, 0x10, 12, 1),
	PIN_FIELD_BASE(39, 39, 2, 0x0060, 0x10, 11, 1),
	PIN_FIELD_BASE(40, 40, 2, 0x0060, 0x10, 13, 1),
	PIN_FIELD_BASE(41, 41, 2, 0x0060, 0x10, 14, 1),
	PIN_FIELD_BASE(42, 42, 2, 0x0060, 0x10, 16, 1),
	PIN_FIELD_BASE(43, 43, 2, 0x0060, 0x10, 15, 1),
	PIN_FIELD_BASE(44, 44, 5, 0x0090, 0x10, 28, 1),
	PIN_FIELD_BASE(45, 45, 5, 0x0090, 0x10, 29, 1),
	PIN_FIELD_BASE(46, 46, 5, 0x0090, 0x10, 31, 1),
	PIN_FIELD_BASE(47, 47, 5, 0x0090, 0x10, 30, 1),
	PIN_FIELD_BASE(48, 48, 2, 0x0060, 0x10, 17, 1),
	PIN_FIELD_BASE(49, 49, 2, 0x0060, 0x10, 18, 1),
	PIN_FIELD_BASE(50, 50, 2, 0x0060, 0x10, 20, 1),
	PIN_FIELD_BASE(51, 51, 2, 0x0060, 0x10, 19, 1),
	PIN_FIELD_BASE(52, 52, 3, 0x0080, 0x10, 12, 1),
	PIN_FIELD_BASE(53, 53, 3, 0x0080, 0x10, 13, 1),
	PIN_FIELD_BASE(54, 54, 3, 0x0080, 0x10, 15, 1),
	PIN_FIELD_BASE(55, 55, 3, 0x0080, 0x10, 14, 1),
	PIN_FIELD_BASE(56, 56, 5, 0x0090, 0x10, 12, 1),
	PIN_FIELD_BASE(57, 57, 5, 0x0090, 0x10, 11, 1),
	PIN_FIELD_BASE(58, 58, 5, 0x0090, 0x10, 13, 1),
	PIN_FIELD_BASE(59, 59, 5, 0x0090, 0x10, 14, 1),
	PIN_FIELD_BASE(60, 60, 3, 0x0080, 0x10, 21, 1),
	PIN_FIELD_BASE(61, 61, 3, 0x0080, 0x10, 16, 1),
	PIN_FIELD_BASE(62, 62, 3, 0x0080, 0x10, 22, 1),
	PIN_FIELD_BASE(63, 63, 3, 0x0080, 0x10, 17, 1),
	PIN_FIELD_BASE(64, 64, 3, 0x0080, 0x10, 18, 1),
	PIN_FIELD_BASE(65, 65, 3, 0x0080, 0x10, 19, 1),
	PIN_FIELD_BASE(66, 66, 3, 0x0080, 0x10, 20, 1),
	PIN_FIELD_BASE(83, 83, 6, 0x0050, 0x10, 3, 1),
	PIN_FIELD_BASE(90, 90, 3, 0x0080, 0x10, 2, 1),
	PIN_FIELD_BASE(91, 91, 3, 0x0080, 0x10, 23, 1),
	PIN_FIELD_BASE(92, 92, 3, 0x0080, 0x10, 25, 1),
	PIN_FIELD_BASE(93, 93, 3, 0x0080, 0x10, 3, 1),
	PIN_FIELD_BASE(94, 94, 3, 0x0080, 0x10, 24, 1),
	PIN_FIELD_BASE(95, 95, 3, 0x0080, 0x10, 26, 1),
	PIN_FIELD_BASE(96, 96, 2, 0x0060, 0x10, 1, 1),
	PIN_FIELD_BASE(97, 97, 2, 0x0060, 0x10, 0, 1),
	PIN_FIELD_BASE(98, 98, 2, 0x0060, 0x10, 2, 1),
	PIN_FIELD_BASE(99, 99, 4, 0x0060, 0x10, 14, 1),
	PIN_FIELD_BASE(100, 100, 4, 0x0060, 0x10, 15, 1),
	PIN_FIELD_BASE(101, 101, 4, 0x0060, 0x10, 13, 1),
	PIN_FIELD_BASE(102, 102, 4, 0x0060, 0x10, 12, 1),
	PIN_FIELD_BASE(103, 103, 4, 0x0060, 0x10, 0, 1),
	PIN_FIELD_BASE(104, 104, 4, 0x0060, 0x10, 1, 1),
	PIN_FIELD_BASE(105, 105, 4, 0x0060, 0x10, 4, 1),
	PIN_FIELD_BASE(106, 106, 4, 0x0060, 0x10, 5, 1),
	PIN_FIELD_BASE(107, 107, 4, 0x0060, 0x10, 6, 1),
	PIN_FIELD_BASE(108, 108, 4, 0x0060, 0x10, 7, 1),
	PIN_FIELD_BASE(109, 109, 4, 0x0060, 0x10, 8, 1),
	PIN_FIELD_BASE(110, 110, 4, 0x0060, 0x10, 9, 1),
	PIN_FIELD_BASE(111, 111, 4, 0x0060, 0x10, 10, 1),
	PIN_FIELD_BASE(112, 112, 4, 0x0060, 0x10, 11, 1),
	PIN_FIELD_BASE(113, 113, 4, 0x0060, 0x10, 2, 1),
	PIN_FIELD_BASE(114, 114, 4, 0x0060, 0x10, 3, 1),
	PIN_FIELD_BASE(115, 115, 3, 0x0080, 0x10, 4, 1),
	PIN_FIELD_BASE(116, 116, 3, 0x0080, 0x10, 7, 1),
	PIN_FIELD_BASE(117, 117, 3, 0x0080, 0x10, 5, 1),
	PIN_FIELD_BASE(118, 118, 3, 0x0080, 0x10, 6, 1),
	PIN_FIELD_BASE(119, 119, 5, 0x0090, 0x10, 22, 1),
	PIN_FIELD_BASE(120, 120, 5, 0x0090, 0x10, 19, 1),
	PIN_FIELD_BASE(121, 121, 5, 0x0090, 0x10, 20, 1),
	PIN_FIELD_BASE(122, 122, 5, 0x0090, 0x10, 21, 1),
	PIN_FIELD_BASE(123, 123, 5, 0x0090, 0x10, 23, 1),
	PIN_FIELD_BASE(124, 124, 5, 0x0090, 0x10, 0, 1),
	PIN_FIELD_BASE(125, 125, 5, 0x0090, 0x10, 1, 1),
	PIN_FIELD_BASE(126, 126, 5, 0x0090, 0x10, 2, 1),
	PIN_FIELD_BASE(127, 127, 3, 0x0080, 0x10, 8, 1),
	PIN_FIELD_BASE(128, 128, 3, 0x0080, 0x10, 10, 1),
	PIN_FIELD_BASE(129, 129, 5, 0x0090, 0x10, 24, 1),
	PIN_FIELD_BASE(130, 130, 5, 0x0090, 0x10, 26, 1),
	PIN_FIELD_BASE(131, 131, 5, 0x0090, 0x10, 25, 1),
	PIN_FIELD_BASE(132, 132, 5, 0x0090, 0x10, 27, 1),
	PIN_FIELD_BASE(133, 133, 1, 0x0080, 0x10, 9, 1),
	PIN_FIELD_BASE(134, 134, 1, 0x0080, 0x10, 12, 1),
	PIN_FIELD_BASE(135, 135, 6, 0x0050, 0x10, 21, 1),
	PIN_FIELD_BASE(136, 136, 6, 0x0050, 0x10, 24, 1),
	PIN_FIELD_BASE(137, 137, 1, 0x0080, 0x10, 10, 1),
	PIN_FIELD_BASE(138, 138, 1, 0x0080, 0x10, 13, 1),
	PIN_FIELD_BASE(139, 139, 2, 0x0060, 0x10, 7, 1),
	PIN_FIELD_BASE(140, 140, 2, 0x0060, 0x10, 8, 1),
	PIN_FIELD_BASE(141, 141, 3, 0x0080, 0x10, 9, 1),
	PIN_FIELD_BASE(142, 142, 3, 0x0080, 0x10, 11, 1),
	PIN_FIELD_BASE(143, 143, 6, 0x0050, 0x10, 22, 1),
	PIN_FIELD_BASE(144, 144, 6, 0x0050, 0x10, 25, 1),
	PIN_FIELD_BASE(145, 145, 6, 0x0050, 0x10, 23, 1),
	PIN_FIELD_BASE(146, 146, 6, 0x0050, 0x10, 26, 1),
	PIN_FIELD_BASE(147, 147, 4, 0x0060, 0x10, 23, 1),
	PIN_FIELD_BASE(148, 148, 4, 0x0060, 0x10, 24, 1),
	PIN_FIELD_BASE(149, 149, 4, 0x0060, 0x10, 25, 1),
	PIN_FIELD_BASE(150, 150, 4, 0x0060, 0x10, 26, 1),
	PIN_FIELD_BASE(151, 151, 4, 0x0060, 0x10, 27, 1),
	PIN_FIELD_BASE(152, 152, 4, 0x0060, 0x10, 28, 1),
	PIN_FIELD_BASE(153, 153, 4, 0x0060, 0x10, 29, 1),
	PIN_FIELD_BASE(154, 154, 4, 0x0060, 0x10, 30, 1),
	PIN_FIELD_BASE(155, 155, 4, 0x0060, 0x10, 31, 1),
	PIN_FIELD_BASE(156, 156, 4, 0x0070, 0x10, 0, 1),
	PIN_FIELD_BASE(157, 157, 2, 0x0060, 0x10, 4, 1),
	PIN_FIELD_BASE(158, 158, 2, 0x0060, 0x10, 3, 1),
	PIN_FIELD_BASE(159, 159, 2, 0x0060, 0x10, 6, 1),
	PIN_FIELD_BASE(160, 160, 2, 0x0060, 0x10, 5, 1),
	PIN_FIELD_BASE(161, 161, 2, 0x0060, 0x10, 23, 1),
	PIN_FIELD_BASE(162, 162, 2, 0x0060, 0x10, 24, 1),
	PIN_FIELD_BASE(163, 163, 1, 0x0080, 0x10, 11, 1),
	PIN_FIELD_BASE(164, 164, 1, 0x0080, 0x10, 8, 1),
	PIN_FIELD_BASE(165, 165, 1, 0x0080, 0x10, 16, 1),
	PIN_FIELD_BASE(166, 166, 1, 0x0080, 0x10, 1, 1),
	PIN_FIELD_BASE(167, 167, 1, 0x0080, 0x10, 7, 1),
	PIN_FIELD_BASE(168, 168, 1, 0x0080, 0x10, 4, 1),
	PIN_FIELD_BASE(169, 169, 1, 0x0080, 0x10, 5, 1),
	PIN_FIELD_BASE(170, 170, 1, 0x0080, 0x10, 0, 1),
	PIN_FIELD_BASE(171, 171, 1, 0x0080, 0x10, 6, 1),
	PIN_FIELD_BASE(172, 172, 1, 0x0080, 0x10, 2, 1),
	PIN_FIELD_BASE(173, 173, 1, 0x0080, 0x10, 3, 1),
	PIN_FIELD_BASE(174, 174, 6, 0x0050, 0x10, 7, 1),
	PIN_FIELD_BASE(175, 175, 6, 0x0050, 0x10, 8, 1),
	PIN_FIELD_BASE(176, 176, 6, 0x0050, 0x10, 4, 1),
	PIN_FIELD_BASE(177, 177, 6, 0x0050, 0x10, 5, 1),
	PIN_FIELD_BASE(178, 178, 6, 0x0050, 0x10, 6, 1),
	PIN_FIELD_BASE(179, 179, 6, 0x0050, 0x10, 9, 1),
	PIN_FIELD_BASE(180, 180, 6, 0x0050, 0x10, 10, 1),
	PIN_FIELD_BASE(181, 181, 6, 0x0050, 0x10, 11, 1),
	PIN_FIELD_BASE(182, 182, 6, 0x0050, 0x10, 12, 1),
	PIN_FIELD_BASE(183, 183, 2, 0x0060, 0x10, 21, 1),
	PIN_FIELD_BASE(184, 184, 2, 0x0060, 0x10, 22, 1),
};

static const struct mtk_pin_field_calc mt8186_pin_pd_range[] = {
	PIN_FIELD_BASE(0, 0, 6, 0x0040, 0x10, 13, 1),
	PIN_FIELD_BASE(1, 1, 6, 0x0040, 0x10, 14, 1),
	PIN_FIELD_BASE(2, 2, 6, 0x0040, 0x10, 17, 1),
	PIN_FIELD_BASE(3, 3, 6, 0x0040, 0x10, 18, 1),
	PIN_FIELD_BASE(4, 4, 6, 0x0040, 0x10, 19, 1),
	PIN_FIELD_BASE(5, 5, 6, 0x0040, 0x10, 20, 1),
	PIN_FIELD_BASE(6, 6, 4, 0x0040, 0x10, 19, 1),
	PIN_FIELD_BASE(7, 7, 4, 0x0040, 0x10, 20, 1),
	PIN_FIELD_BASE(8, 8, 4, 0x0040, 0x10, 21, 1),
	PIN_FIELD_BASE(9, 9, 4, 0x0040, 0x10, 22, 1),
	PIN_FIELD_BASE(10, 10, 4, 0x0040, 0x10, 16, 1),
	PIN_FIELD_BASE(11, 11, 4, 0x0040, 0x10, 17, 1),
	PIN_FIELD_BASE(12, 12, 4, 0x0040, 0x10, 18, 1),
	PIN_FIELD_BASE(13, 13, 3, 0x0060, 0x10, 0, 1),
	PIN_FIELD_BASE(14, 14, 3, 0x0060, 0x10, 1, 1),
	PIN_FIELD_BASE(15, 15, 6, 0x0040, 0x10, 15, 1),
	PIN_FIELD_BASE(16, 16, 6, 0x0040, 0x10, 16, 1),
	PIN_FIELD_BASE(17, 17, 5, 0x0070, 0x10, 9, 1),
	PIN_FIELD_BASE(18, 18, 5, 0x0070, 0x10, 10, 1),
	PIN_FIELD_BASE(19, 19, 5, 0x0070, 0x10, 3, 1),
	PIN_FIELD_BASE(20, 20, 5, 0x0070, 0x10, 6, 1),
	PIN_FIELD_BASE(21, 21, 5, 0x0070, 0x10, 4, 1),
	PIN_FIELD_BASE(22, 22, 5, 0x0070, 0x10, 7, 1),
	PIN_FIELD_BASE(23, 23, 5, 0x0070, 0x10, 5, 1),
	PIN_FIELD_BASE(24, 24, 5, 0x0070, 0x10, 8, 1),
	PIN_FIELD_BASE(25, 25, 5, 0x0070, 0x10, 18, 1),
	PIN_FIELD_BASE(26, 26, 5, 0x0070, 0x10, 15, 1),
	PIN_FIELD_BASE(27, 27, 5, 0x0070, 0x10, 17, 1),
	PIN_FIELD_BASE(28, 28, 5, 0x0070, 0x10, 16, 1),
	PIN_FIELD_BASE(29, 29, 6, 0x0040, 0x10, 0, 1),
	PIN_FIELD_BASE(30, 30, 6, 0x0040, 0x10, 1, 1),
	PIN_FIELD_BASE(31, 31, 6, 0x0040, 0x10, 2, 1),
	PIN_FIELD_BASE(32, 32, 2, 0x0050, 0x10, 25, 1),
	PIN_FIELD_BASE(33, 33, 2, 0x0050, 0x10, 27, 1),
	PIN_FIELD_BASE(34, 34, 2, 0x0050, 0x10, 26, 1),
	PIN_FIELD_BASE(35, 35, 2, 0x0050, 0x10, 28, 1),
	PIN_FIELD_BASE(36, 36, 2, 0x0050, 0x10, 9, 1),
	PIN_FIELD_BASE(37, 37, 2, 0x0050, 0x10, 10, 1),
	PIN_FIELD_BASE(38, 38, 2, 0x0050, 0x10, 12, 1),
	PIN_FIELD_BASE(39, 39, 2, 0x0050, 0x10, 11, 1),
	PIN_FIELD_BASE(40, 40, 2, 0x0050, 0x10, 13, 1),
	PIN_FIELD_BASE(41, 41, 2, 0x0050, 0x10, 14, 1),
	PIN_FIELD_BASE(42, 42, 2, 0x0050, 0x10, 16, 1),
	PIN_FIELD_BASE(43, 43, 2, 0x0050, 0x10, 15, 1),
	PIN_FIELD_BASE(44, 44, 5, 0x0070, 0x10, 28, 1),
	PIN_FIELD_BASE(45, 45, 5, 0x0070, 0x10, 29, 1),
	PIN_FIELD_BASE(46, 46, 5, 0x0070, 0x10, 31, 1),
	PIN_FIELD_BASE(47, 47, 5, 0x0070, 0x10, 30, 1),
	PIN_FIELD_BASE(48, 48, 2, 0x0050, 0x10, 17, 1),
	PIN_FIELD_BASE(49, 49, 2, 0x0050, 0x10, 18, 1),
	PIN_FIELD_BASE(50, 50, 2, 0x0050, 0x10, 20, 1),
	PIN_FIELD_BASE(51, 51, 2, 0x0050, 0x10, 19, 1),
	PIN_FIELD_BASE(52, 52, 3, 0x0060, 0x10, 12, 1),
	PIN_FIELD_BASE(53, 53, 3, 0x0060, 0x10, 13, 1),
	PIN_FIELD_BASE(54, 54, 3, 0x0060, 0x10, 15, 1),
	PIN_FIELD_BASE(55, 55, 3, 0x0060, 0x10, 14, 1),
	PIN_FIELD_BASE(56, 56, 5, 0x0070, 0x10, 12, 1),
	PIN_FIELD_BASE(57, 57, 5, 0x0070, 0x10, 11, 1),
	PIN_FIELD_BASE(58, 58, 5, 0x0070, 0x10, 13, 1),
	PIN_FIELD_BASE(59, 59, 5, 0x0070, 0x10, 14, 1),
	PIN_FIELD_BASE(60, 60, 3, 0x0060, 0x10, 21, 1),
	PIN_FIELD_BASE(61, 61, 3, 0x0060, 0x10, 16, 1),
	PIN_FIELD_BASE(62, 62, 3, 0x0060, 0x10, 22, 1),
	PIN_FIELD_BASE(63, 63, 3, 0x0060, 0x10, 17, 1),
	PIN_FIELD_BASE(64, 64, 3, 0x0060, 0x10, 18, 1),
	PIN_FIELD_BASE(65, 65, 3, 0x0060, 0x10, 19, 1),
	PIN_FIELD_BASE(66, 66, 3, 0x0060, 0x10, 20, 1),
	PIN_FIELD_BASE(83, 83, 6, 0x0040, 0x10, 3, 1),
	PIN_FIELD_BASE(90, 90, 3, 0x0060, 0x10, 2, 1),
	PIN_FIELD_BASE(91, 91, 3, 0x0060, 0x10, 23, 1),
	PIN_FIELD_BASE(92, 92, 3, 0x0060, 0x10, 25, 1),
	PIN_FIELD_BASE(93, 93, 3, 0x0060, 0x10, 3, 1),
	PIN_FIELD_BASE(94, 94, 3, 0x0060, 0x10, 24, 1),
	PIN_FIELD_BASE(95, 95, 3, 0x0060, 0x10, 26, 1),
	PIN_FIELD_BASE(96, 96, 2, 0x0050, 0x10, 1, 1),
	PIN_FIELD_BASE(97, 97, 2, 0x0050, 0x10, 0, 1),
	PIN_FIELD_BASE(98, 98, 2, 0x0050, 0x10, 2, 1),
	PIN_FIELD_BASE(99, 99, 4, 0x0040, 0x10, 14, 1),
	PIN_FIELD_BASE(100, 100, 4, 0x0040, 0x10, 15, 1),
	PIN_FIELD_BASE(101, 101, 4, 0x0040, 0x10, 13, 1),
	PIN_FIELD_BASE(102, 102, 4, 0x0040, 0x10, 12, 1),
	PIN_FIELD_BASE(103, 103, 4, 0x0040, 0x10, 0, 1),
	PIN_FIELD_BASE(104, 104, 4, 0x0040, 0x10, 1, 1),
	PIN_FIELD_BASE(105, 105, 4, 0x0040, 0x10, 4, 1),
	PIN_FIELD_BASE(106, 106, 4, 0x0040, 0x10, 5, 1),
	PIN_FIELD_BASE(107, 107, 4, 0x0040, 0x10, 6, 1),
	PIN_FIELD_BASE(108, 108, 4, 0x0040, 0x10, 7, 1),
	PIN_FIELD_BASE(109, 109, 4, 0x0040, 0x10, 8, 1),
	PIN_FIELD_BASE(110, 110, 4, 0x0040, 0x10, 9, 1),
	PIN_FIELD_BASE(111, 111, 4, 0x0040, 0x10, 10, 1),
	PIN_FIELD_BASE(112, 112, 4, 0x0040, 0x10, 11, 1),
	PIN_FIELD_BASE(113, 113, 4, 0x0040, 0x10, 2, 1),
	PIN_FIELD_BASE(114, 114, 4, 0x0040, 0x10, 3, 1),
	PIN_FIELD_BASE(115, 115, 3, 0x0060, 0x10, 4, 1),
	PIN_FIELD_BASE(116, 116, 3, 0x0060, 0x10, 7, 1),
	PIN_FIELD_BASE(117, 117, 3, 0x0060, 0x10, 5, 1),
	PIN_FIELD_BASE(118, 118, 3, 0x0060, 0x10, 6, 1),
	PIN_FIELD_BASE(119, 119, 5, 0x0070, 0x10, 22, 1),
	PIN_FIELD_BASE(120, 120, 5, 0x0070, 0x10, 19, 1),
	PIN_FIELD_BASE(121, 121, 5, 0x0070, 0x10, 20, 1),
	PIN_FIELD_BASE(122, 122, 5, 0x0070, 0x10, 21, 1),
	PIN_FIELD_BASE(123, 123, 5, 0x0070, 0x10, 23, 1),
	PIN_FIELD_BASE(124, 124, 5, 0x0070, 0x10, 0, 1),
	PIN_FIELD_BASE(125, 125, 5, 0x0070, 0x10, 1, 1),
	PIN_FIELD_BASE(126, 126, 5, 0x0070, 0x10, 2, 1),
	PIN_FIELD_BASE(127, 127, 3, 0x0060, 0x10, 8, 1),
	PIN_FIELD_BASE(128, 128, 3, 0x0060, 0x10, 10, 1),
	PIN_FIELD_BASE(129, 129, 5, 0x0070, 0x10, 24, 1),
	PIN_FIELD_BASE(130, 130, 5, 0x0070, 0x10, 26, 1),
	PIN_FIELD_BASE(131, 131, 5, 0x0070, 0x10, 25, 1),
	PIN_FIELD_BASE(132, 132, 5, 0x0070, 0x10, 27, 1),
	PIN_FIELD_BASE(133, 133, 1, 0x0060, 0x10, 9, 1),
	PIN_FIELD_BASE(134, 134, 1, 0x0060, 0x10, 12, 1),
	PIN_FIELD_BASE(135, 135, 6, 0x0040, 0x10, 21, 1),
	PIN_FIELD_BASE(136, 136, 6, 0x0040, 0x10, 24, 1),
	PIN_FIELD_BASE(137, 137, 1, 0x0060, 0x10, 10, 1),
	PIN_FIELD_BASE(138, 138, 1, 0x0060, 0x10, 13, 1),
	PIN_FIELD_BASE(139, 139, 2, 0x0050, 0x10, 7, 1),
	PIN_FIELD_BASE(140, 140, 2, 0x0050, 0x10, 8, 1),
	PIN_FIELD_BASE(141, 141, 3, 0x0060, 0x10, 9, 1),
	PIN_FIELD_BASE(142, 142, 3, 0x0060, 0x10, 11, 1),
	PIN_FIELD_BASE(143, 143, 6, 0x0040, 0x10, 22, 1),
	PIN_FIELD_BASE(144, 144, 6, 0x0040, 0x10, 25, 1),
	PIN_FIELD_BASE(145, 145, 6, 0x0040, 0x10, 23, 1),
	PIN_FIELD_BASE(146, 146, 6, 0x0040, 0x10, 26, 1),
	PIN_FIELD_BASE(147, 147, 4, 0x0040, 0x10, 23, 1),
	PIN_FIELD_BASE(148, 148, 4, 0x0040, 0x10, 24, 1),
	PIN_FIELD_BASE(149, 149, 4, 0x0040, 0x10, 25, 1),
	PIN_FIELD_BASE(150, 150, 4, 0x0040, 0x10, 26, 1),
	PIN_FIELD_BASE(151, 151, 4, 0x0040, 0x10, 27, 1),
	PIN_FIELD_BASE(152, 152, 4, 0x0040, 0x10, 28, 1),
	PIN_FIELD_BASE(153, 153, 4, 0x0040, 0x10, 29, 1),
	PIN_FIELD_BASE(154, 154, 4, 0x0040, 0x10, 30, 1),
	PIN_FIELD_BASE(155, 155, 4, 0x0040, 0x10, 31, 1),
	PIN_FIELD_BASE(156, 156, 4, 0x0050, 0x10, 0, 1),
	PIN_FIELD_BASE(157, 157, 2, 0x0050, 0x10, 4, 1),
	PIN_FIELD_BASE(158, 158, 2, 0x0050, 0x10, 3, 1),
	PIN_FIELD_BASE(159, 159, 2, 0x0050, 0x10, 6, 1),
	PIN_FIELD_BASE(160, 160, 2, 0x0050, 0x10, 5, 1),
	PIN_FIELD_BASE(161, 161, 2, 0x0050, 0x10, 23, 1),
	PIN_FIELD_BASE(162, 162, 2, 0x0050, 0x10, 24, 1),
	PIN_FIELD_BASE(163, 163, 1, 0x0060, 0x10, 11, 1),
	PIN_FIELD_BASE(164, 164, 1, 0x0060, 0x10, 8, 1),
	PIN_FIELD_BASE(165, 165, 1, 0x0060, 0x10, 16, 1),
	PIN_FIELD_BASE(166, 166, 1, 0x0060, 0x10, 1, 1),
	PIN_FIELD_BASE(167, 167, 1, 0x0060, 0x10, 7, 1),
	PIN_FIELD_BASE(168, 168, 1, 0x0060, 0x10, 4, 1),
	PIN_FIELD_BASE(169, 169, 1, 0x0060, 0x10, 5, 1),
	PIN_FIELD_BASE(170, 170, 1, 0x0060, 0x10, 0, 1),
	PIN_FIELD_BASE(171, 171, 1, 0x0060, 0x10, 6, 1),
	PIN_FIELD_BASE(172, 172, 1, 0x0060, 0x10, 2, 1),
	PIN_FIELD_BASE(173, 173, 1, 0x0060, 0x10, 3, 1),
	PIN_FIELD_BASE(174, 174, 6, 0x0040, 0x10, 7, 1),
	PIN_FIELD_BASE(175, 175, 6, 0x0040, 0x10, 8, 1),
	PIN_FIELD_BASE(176, 176, 6, 0x0040, 0x10, 4, 1),
	PIN_FIELD_BASE(177, 177, 6, 0x0040, 0x10, 5, 1),
	PIN_FIELD_BASE(178, 178, 6, 0x0040, 0x10, 6, 1),
	PIN_FIELD_BASE(179, 179, 6, 0x0040, 0x10, 9, 1),
	PIN_FIELD_BASE(180, 180, 6, 0x0040, 0x10, 10, 1),
	PIN_FIELD_BASE(181, 181, 6, 0x0040, 0x10, 11, 1),
	PIN_FIELD_BASE(182, 182, 6, 0x0040, 0x10, 12, 1),
	PIN_FIELD_BASE(183, 183, 2, 0x0050, 0x10, 21, 1),
	PIN_FIELD_BASE(184, 184, 2, 0x0050, 0x10, 22, 1),
};

static const struct mtk_pin_field_calc mt8186_pin_pupd_range[] = {
	PIN_FIELD_BASE(67, 67, 1, 0x0070, 0x10, 10, 1),
	PIN_FIELD_BASE(68, 68, 1, 0x0070, 0x10, 0, 1),
	PIN_FIELD_BASE(69, 69, 1, 0x0070, 0x10, 1, 1),
	PIN_FIELD_BASE(70, 70, 1, 0x0070, 0x10, 11, 1),
	PIN_FIELD_BASE(71, 71, 1, 0x0070, 0x10, 2, 1),
	PIN_FIELD_BASE(72, 72, 1, 0x0070, 0x10, 3, 1),
	PIN_FIELD_BASE(73, 73, 1, 0x0070, 0x10, 4, 1),
	PIN_FIELD_BASE(74, 74, 1, 0x0070, 0x10, 5, 1),
	PIN_FIELD_BASE(75, 75, 1, 0x0070, 0x10, 6, 1),
	PIN_FIELD_BASE(76, 76, 1, 0x0070, 0x10, 7, 1),
	PIN_FIELD_BASE(77, 77, 1, 0x0070, 0x10, 8, 1),
	PIN_FIELD_BASE(78, 78, 1, 0x0070, 0x10, 9, 1),
	PIN_FIELD_BASE(79, 79, 5, 0x0080, 0x10, 0, 1),
	PIN_FIELD_BASE(80, 80, 5, 0x0080, 0x10, 1, 1),
	PIN_FIELD_BASE(81, 81, 5, 0x0080, 0x10, 2, 1),
	PIN_FIELD_BASE(82, 82, 5, 0x0080, 0x10, 3, 1),
	PIN_FIELD_BASE(84, 84, 3, 0x0070, 0x10, 0, 1),
	PIN_FIELD_BASE(85, 85, 3, 0x0070, 0x10, 1, 1),
	PIN_FIELD_BASE(86, 86, 3, 0x0070, 0x10, 2, 1),
	PIN_FIELD_BASE(87, 87, 3, 0x0070, 0x10, 3, 1),
	PIN_FIELD_BASE(88, 88, 3, 0x0070, 0x10, 4, 1),
	PIN_FIELD_BASE(89, 89, 3, 0x0070, 0x10, 5, 1),
};

static const struct mtk_pin_field_calc mt8186_pin_r0_range[] = {
	PIN_FIELD_BASE(67, 67, 1, 0x0090, 0x10, 10, 1),
	PIN_FIELD_BASE(68, 68, 1, 0x0090, 0x10, 0, 1),
	PIN_FIELD_BASE(69, 69, 1, 0x0090, 0x10, 1, 1),
	PIN_FIELD_BASE(70, 70, 1, 0x0090, 0x10, 11, 1),
	PIN_FIELD_BASE(71, 71, 1, 0x0090, 0x10, 2, 1),
	PIN_FIELD_BASE(72, 72, 1, 0x0090, 0x10, 3, 1),
	PIN_FIELD_BASE(73, 73, 1, 0x0090, 0x10, 4, 1),
	PIN_FIELD_BASE(74, 74, 1, 0x0090, 0x10, 5, 1),
	PIN_FIELD_BASE(75, 75, 1, 0x0090, 0x10, 6, 1),
	PIN_FIELD_BASE(76, 76, 1, 0x0090, 0x10, 7, 1),
	PIN_FIELD_BASE(77, 77, 1, 0x0090, 0x10, 8, 1),
	PIN_FIELD_BASE(78, 78, 1, 0x0090, 0x10, 9, 1),
	PIN_FIELD_BASE(79, 79, 5, 0x00a0, 0x10, 0, 1),
	PIN_FIELD_BASE(80, 80, 5, 0x00a0, 0x10, 1, 1),
	PIN_FIELD_BASE(81, 81, 5, 0x00a0, 0x10, 2, 1),
	PIN_FIELD_BASE(82, 82, 5, 0x00a0, 0x10, 3, 1),
	PIN_FIELD_BASE(84, 84, 3, 0x0090, 0x10, 0, 1),
	PIN_FIELD_BASE(85, 85, 3, 0x0090, 0x10, 1, 1),
	PIN_FIELD_BASE(86, 86, 3, 0x0090, 0x10, 2, 1),
	PIN_FIELD_BASE(87, 87, 3, 0x0090, 0x10, 3, 1),
	PIN_FIELD_BASE(88, 88, 3, 0x0090, 0x10, 4, 1),
	PIN_FIELD_BASE(89, 89, 3, 0x0090, 0x10, 5, 1),
};

static const struct mtk_pin_field_calc mt8186_pin_r1_range[] = {
	PIN_FIELD_BASE(67, 67, 1, 0x00a0, 0x10, 10, 1),
	PIN_FIELD_BASE(68, 68, 1, 0x00a0, 0x10, 0, 1),
	PIN_FIELD_BASE(69, 69, 1, 0x00a0, 0x10, 1, 1),
	PIN_FIELD_BASE(70, 70, 1, 0x00a0, 0x10, 11, 1),
	PIN_FIELD_BASE(71, 71, 1, 0x00a0, 0x10, 2, 1),
	PIN_FIELD_BASE(72, 72, 1, 0x00a0, 0x10, 3, 1),
	PIN_FIELD_BASE(73, 73, 1, 0x00a0, 0x10, 4, 1),
	PIN_FIELD_BASE(74, 74, 1, 0x00a0, 0x10, 5, 1),
	PIN_FIELD_BASE(75, 75, 1, 0x00a0, 0x10, 6, 1),
	PIN_FIELD_BASE(76, 76, 1, 0x00a0, 0x10, 7, 1),
	PIN_FIELD_BASE(77, 77, 1, 0x00a0, 0x10, 8, 1),
	PIN_FIELD_BASE(78, 78, 1, 0x00a0, 0x10, 9, 1),
	PIN_FIELD_BASE(79, 79, 5, 0x00b0, 0x10, 0, 1),
	PIN_FIELD_BASE(80, 80, 5, 0x00b0, 0x10, 1, 1),
	PIN_FIELD_BASE(81, 81, 5, 0x00b0, 0x10, 2, 1),
	PIN_FIELD_BASE(82, 82, 5, 0x00b0, 0x10, 3, 1),
	PIN_FIELD_BASE(84, 84, 3, 0x00a0, 0x10, 0, 1),
	PIN_FIELD_BASE(85, 85, 3, 0x00a0, 0x10, 1, 1),
	PIN_FIELD_BASE(86, 86, 3, 0x00a0, 0x10, 2, 1),
	PIN_FIELD_BASE(87, 87, 3, 0x00a0, 0x10, 3, 1),
	PIN_FIELD_BASE(88, 88, 3, 0x00a0, 0x10, 4, 1),
	PIN_FIELD_BASE(89, 89, 3, 0x00a0, 0x10, 5, 1),
};

static const struct mtk_pin_field_calc mt8186_pin_drv_range[] = {
	PIN_FIELD_BASE(0, 0, 6, 0x0000, 0x10, 27, 3),
	PIN_FIELD_BASE(1, 1, 6, 0x0000, 0x10, 27, 3),
	PIN_FIELD_BASE(2, 2, 6, 0x0000, 0x10, 27, 3),
	PIN_FIELD_BASE(3, 3, 6, 0x0000, 0x10, 27, 3),
	PIN_FIELD_BASE(4, 4, 6, 0x0010, 0x10, 0, 3),
	PIN_FIELD_BASE(5, 5, 6, 0x0010, 0x10, 0, 3),
	PIN_FIELD_BASE(6, 6, 4, 0x0000, 0x10, 9, 3),
	PIN_FIELD_BASE(7, 7, 4, 0x0000, 0x10, 9, 3),
	PIN_FIELD_BASE(8, 8, 4, 0x0000, 0x10, 9, 3),
	PIN_FIELD_BASE(9, 9, 4, 0x0000, 0x10, 12, 3),
	PIN_FIELD_BASE(10, 10, 4, 0x0000, 0x10, 0, 3),
	PIN_FIELD_BASE(11, 11, 4, 0x0000, 0x10, 3, 3),
	PIN_FIELD_BASE(12, 12, 4, 0x0000, 0x10, 6, 3),
	PIN_FIELD_BASE(13, 13, 3, 0x0000, 0x10, 0, 3),
	PIN_FIELD_BASE(14, 14, 3, 0x0000, 0x10, 3, 3),
	PIN_FIELD_BASE(15, 15, 6, 0x0010, 0x10, 3, 3),
	PIN_FIELD_BASE(16, 16, 6, 0x0010, 0x10, 3, 3),
	PIN_FIELD_BASE(17, 17, 5, 0x0000, 0x10, 21, 3),
	PIN_FIELD_BASE(18, 18, 5, 0x0010, 0x10, 0, 3),
	PIN_FIELD_BASE(19, 19, 5, 0x0000, 0x10, 27, 3),
	PIN_FIELD_BASE(20, 20, 5, 0x0000, 0x10, 24, 3),
	PIN_FIELD_BASE(21, 21, 5, 0x0000, 0x10, 27, 3),
	PIN_FIELD_BASE(22, 22, 5, 0x0000, 0x10, 24, 3),
	PIN_FIELD_BASE(23, 23, 5, 0x0010, 0x10, 0, 3),
	PIN_FIELD_BASE(24, 24, 5, 0x0010, 0x10, 9, 3),
	PIN_FIELD_BASE(25, 25, 5, 0x0010, 0x10, 0, 3),
	PIN_FIELD_BASE(26, 26, 5, 0x0000, 0x10, 27, 3),
	PIN_FIELD_BASE(27, 27, 5, 0x0000, 0x10, 27, 3),
	PIN_FIELD_BASE(28, 28, 5, 0x0010, 0x10, 0, 3),
	PIN_FIELD_BASE(29, 29, 6, 0x0010, 0x10, 0, 3),
	PIN_FIELD_BASE(30, 30, 6, 0x0010, 0x10, 0, 3),
	PIN_FIELD_BASE(31, 31, 6, 0x0010, 0x10, 3, 3),
	PIN_FIELD_BASE(32, 32, 2, 0x0000, 0x10, 6, 3),
	PIN_FIELD_BASE(33, 33, 2, 0x0000, 0x10, 6, 3),
	PIN_FIELD_BASE(34, 34, 2, 0x0000, 0x10, 3, 3),
	PIN_FIELD_BASE(35, 35, 2, 0x0000, 0x10, 3, 3),
	PIN_FIELD_BASE(36, 36, 2, 0x0000, 0x10, 9, 3),
	PIN_FIELD_BASE(37, 37, 2, 0x0000, 0x10, 9, 3),
	PIN_FIELD_BASE(38, 38, 2, 0x0000, 0x10, 12, 3),
	PIN_FIELD_BASE(39, 39, 2, 0x0000, 0x10, 9, 3),
	PIN_FIELD_BASE(40, 40, 2, 0x0000, 0x10, 15, 3),
	PIN_FIELD_BASE(41, 41, 2, 0x0000, 0x10, 12, 3),
	PIN_FIELD_BASE(42, 42, 2, 0x0000, 0x10, 12, 3),
	PIN_FIELD_BASE(43, 43, 2, 0x0000, 0x10, 12, 3),
	PIN_FIELD_BASE(44, 44, 5, 0x0010, 0x10, 6, 3),
	PIN_FIELD_BASE(45, 45, 5, 0x0010, 0x10, 3, 3),
	PIN_FIELD_BASE(46, 46, 5, 0x0010, 0x10, 3, 3),
	PIN_FIELD_BASE(47, 47, 5, 0x0010, 0x10, 6, 3),
	PIN_FIELD_BASE(48, 48, 2, 0x0000, 0x10, 15, 3),
	PIN_FIELD_BASE(49, 49, 2, 0x0000, 0x10, 15, 3),
	PIN_FIELD_BASE(50, 50, 2, 0x0000, 0x10, 15, 3),
	PIN_FIELD_BASE(51, 51, 2, 0x0000, 0x10, 18, 3),
	PIN_FIELD_BASE(52, 52, 3, 0x0010, 0x10, 3, 3),
	PIN_FIELD_BASE(53, 53, 3, 0x0010, 0x10, 6, 3),
	PIN_FIELD_BASE(54, 54, 3, 0x0010, 0x10, 3, 3),
	PIN_FIELD_BASE(55, 55, 3, 0x0010, 0x10, 3, 3),
	PIN_FIELD_BASE(56, 56, 5, 0x0010, 0x10, 3, 3),
	PIN_FIELD_BASE(57, 57, 5, 0x0010, 0x10, 6, 3),
	PIN_FIELD_BASE(58, 58, 5, 0x0010, 0x10, 6, 3),
	PIN_FIELD_BASE(59, 59, 5, 0x0010, 0x10, 3, 3),
	PIN_FIELD_BASE(60, 60, 3, 0x0000, 0x10, 24, 3),
	PIN_FIELD_BASE(61, 61, 3, 0x0000, 0x10, 24, 3),
	PIN_FIELD_BASE(62, 62, 3, 0x0000, 0x10, 24, 3),
	PIN_FIELD_BASE(63, 63, 3, 0x0000, 0x10, 24, 3),
	PIN_FIELD_BASE(64, 64, 3, 0x0000, 0x10, 27, 3),
	PIN_FIELD_BASE(65, 65, 3, 0x0000, 0x10, 27, 3),
	PIN_FIELD_BASE(66, 66, 3, 0x0000, 0x10, 27, 3),
	PIN_FIELD_BASE(67, 67, 1, 0x0010, 0x10, 0, 3),
	PIN_FIELD_BASE(68, 68, 1, 0x0000, 0x10, 0, 3),
	PIN_FIELD_BASE(69, 69, 1, 0x0000, 0x10, 3, 3),
	PIN_FIELD_BASE(70, 70, 1, 0x0010, 0x10, 3, 3),
	PIN_FIELD_BASE(71, 71, 1, 0x0000, 0x10, 6, 3),
	PIN_FIELD_BASE(72, 72, 1, 0x0000, 0x10, 9, 3),
	PIN_FIELD_BASE(73, 73, 1, 0x0000, 0x10, 12, 3),
	PIN_FIELD_BASE(74, 74, 1, 0x0000, 0x10, 15, 3),
	PIN_FIELD_BASE(75, 75, 1, 0x0000, 0x10, 18, 3),
	PIN_FIELD_BASE(76, 76, 1, 0x0000, 0x10, 21, 3),
	PIN_FIELD_BASE(77, 77, 1, 0x0000, 0x10, 24, 3),
	PIN_FIELD_BASE(78, 78, 1, 0x0000, 0x10, 27, 3),
	PIN_FIELD_BASE(79, 79, 5, 0x0000, 0x10, 24, 3),
	PIN_FIELD_BASE(80, 80, 5, 0x0000, 0x10, 24, 3),
	PIN_FIELD_BASE(81, 81, 5, 0x0000, 0x10, 21, 3),
	PIN_FIELD_BASE(82, 82, 5, 0x0000, 0x10, 21, 3),
	PIN_FIELD_BASE(83, 83, 6, 0x0010, 0x10, 3, 3),
	PIN_FIELD_BASE(84, 84, 3, 0x0000, 0x10, 6, 3),
	PIN_FIELD_BASE(85, 85, 3, 0x0000, 0x10, 9, 3),
	PIN_FIELD_BASE(86, 86, 3, 0x0000, 0x10, 12, 3),
	PIN_FIELD_BASE(87, 87, 3, 0x0000, 0x10, 15, 3),
	PIN_FIELD_BASE(88, 88, 3, 0x0000, 0x10, 18, 3),
	PIN_FIELD_BASE(89, 89, 3, 0x0000, 0x10, 21, 3),
	PIN_FIELD_BASE(90, 90, 3, 0x0000, 0x10, 27, 3),
	PIN_FIELD_BASE(91, 91, 3, 0x0010, 0x10, 0, 3),
	PIN_FIELD_BASE(92, 92, 3, 0x0010, 0x10, 0, 3),
	PIN_FIELD_BASE(93, 93, 3, 0x0010, 0x10, 0, 3),
	PIN_FIELD_BASE(94, 94, 3, 0x0010, 0x10, 0, 3),
	PIN_FIELD_BASE(95, 95, 3, 0x0010, 0x10, 3, 3),
	PIN_FIELD_BASE(96, 96, 2, 0x0000, 0x10, 9, 3),
	PIN_FIELD_BASE(97, 97, 2, 0x0000, 0x10, 6, 3),
	PIN_FIELD_BASE(98, 98, 2, 0x0000, 0x10, 3, 3),
	PIN_FIELD_BASE(99, 99, 4, 0x0000, 0x10, 18, 3),
	PIN_FIELD_BASE(100, 100, 4, 0x0000, 0x10, 18, 3),
	PIN_FIELD_BASE(101, 101, 4, 0x0000, 0x10, 21, 3),
	PIN_FIELD_BASE(102, 102, 4, 0x0000, 0x10, 21, 3),
	PIN_FIELD_BASE(103, 103, 4, 0x0000, 0x10, 21, 3),
	PIN_FIELD_BASE(104, 104, 4, 0x0000, 0x10, 21, 3),
	PIN_FIELD_BASE(105, 105, 4, 0x0000, 0x10, 24, 3),
	PIN_FIELD_BASE(106, 106, 4, 0x0000, 0x10, 24, 3),
	PIN_FIELD_BASE(107, 107, 4, 0x0000, 0x10, 24, 3),
	PIN_FIELD_BASE(108, 108, 4, 0x0000, 0x10, 24, 3),
	PIN_FIELD_BASE(109, 109, 4, 0x0000, 0x10, 27, 3),
	PIN_FIELD_BASE(110, 110, 4, 0x0000, 0x10, 27, 3),
	PIN_FIELD_BASE(111, 111, 4, 0x0000, 0x10, 27, 3),
	PIN_FIELD_BASE(112, 112, 4, 0x0000, 0x10, 27, 3),
	PIN_FIELD_BASE(113, 113, 4, 0x0010, 0x10, 0, 3),
	PIN_FIELD_BASE(114, 114, 4, 0x0010, 0x10, 0, 3),
	PIN_FIELD_BASE(115, 115, 3, 0x0010, 0x10, 6, 3),
	PIN_FIELD_BASE(116, 116, 3, 0x0010, 0x10, 9, 3),
	PIN_FIELD_BASE(117, 117, 3, 0x0010, 0x10, 6, 3),
	PIN_FIELD_BASE(118, 118, 3, 0x0010, 0x10, 6, 3),
	PIN_FIELD_BASE(119, 119, 5, 0x0000, 0x10, 18, 3),
	PIN_FIELD_BASE(120, 120, 5, 0x0000, 0x10, 9, 3),
	PIN_FIELD_BASE(121, 121, 5, 0x0000, 0x10, 12, 3),
	PIN_FIELD_BASE(122, 122, 5, 0x0000, 0x10, 15, 3),
	PIN_FIELD_BASE(123, 123, 5, 0x0000, 0x10, 21, 3),
	PIN_FIELD_BASE(124, 124, 5, 0x0000, 0x10, 0, 3),
	PIN_FIELD_BASE(125, 125, 5, 0x0000, 0x10, 3, 3),
	PIN_FIELD_BASE(126, 126, 5, 0x0000, 0x10, 6, 3),
	PIN_FIELD_BASE(127, 127, 3, 0x0010, 0x10, 12, 3),
	PIN_FIELD_BASE(128, 128, 3, 0x0010, 0x10, 18, 3),
	PIN_FIELD_BASE(129, 129, 5, 0x0010, 0x10, 12, 3),
	PIN_FIELD_BASE(130, 130, 5, 0x0010, 0x10, 18, 3),
	PIN_FIELD_BASE(131, 131, 5, 0x0010, 0x10, 15, 3),
	PIN_FIELD_BASE(132, 132, 5, 0x0010, 0x10, 21, 3),
	PIN_FIELD_BASE(133, 133, 1, 0x0010, 0x10, 15, 3),
	PIN_FIELD_BASE(134, 134, 1, 0x0010, 0x10, 21, 3),
	PIN_FIELD_BASE(135, 135, 6, 0x0010, 0x10, 6, 3),
	PIN_FIELD_BASE(136, 136, 6, 0x0010, 0x10, 15, 3),
	PIN_FIELD_BASE(137, 137, 1, 0x0010, 0x10, 18, 3),
	PIN_FIELD_BASE(138, 138, 1, 0x0010, 0x10, 24, 3),
	PIN_FIELD_BASE(139, 139, 2, 0x0000, 0x10, 21, 3),
	PIN_FIELD_BASE(140, 140, 2, 0x0000, 0x10, 24, 3),
	PIN_FIELD_BASE(141, 141, 3, 0x0010, 0x10, 15, 3),
	PIN_FIELD_BASE(142, 142, 3, 0x0010, 0x10, 21, 3),
	PIN_FIELD_BASE(143, 143, 6, 0x0010, 0x10, 9, 3),
	PIN_FIELD_BASE(144, 144, 6, 0x0010, 0x10, 18, 3),
	PIN_FIELD_BASE(145, 145, 6, 0x0010, 0x10, 12, 3),
	PIN_FIELD_BASE(146, 146, 6, 0x0010, 0x10, 21, 3),
	PIN_FIELD_BASE(147, 147, 4, 0x0000, 0x10, 12, 3),
	PIN_FIELD_BASE(148, 148, 4, 0x0000, 0x10, 12, 3),
	PIN_FIELD_BASE(149, 149, 4, 0x0000, 0x10, 12, 3),
	PIN_FIELD_BASE(150, 150, 4, 0x0000, 0x10, 15, 3),
	PIN_FIELD_BASE(151, 151, 4, 0x0000, 0x10, 15, 3),
	PIN_FIELD_BASE(152, 152, 4, 0x0000, 0x10, 9, 3),
	PIN_FIELD_BASE(153, 153, 4, 0x0000, 0x10, 15, 3),
	PIN_FIELD_BASE(154, 154, 4, 0x0000, 0x10, 15, 3),
	PIN_FIELD_BASE(155, 155, 4, 0x0000, 0x10, 18, 3),
	PIN_FIELD_BASE(156, 156, 4, 0x0000, 0x10, 18, 3),
	PIN_FIELD_BASE(157, 157, 2, 0x0000, 0x10, 0, 3),
	PIN_FIELD_BASE(158, 158, 2, 0x0000, 0x10, 0, 3),
	PIN_FIELD_BASE(159, 159, 2, 0x0000, 0x10, 0, 3),
	PIN_FIELD_BASE(160, 160, 2, 0x0000, 0x10, 0, 3),
	PIN_FIELD_BASE(161, 161, 2, 0x0000, 0x10, 6, 3),
	PIN_FIELD_BASE(162, 162, 2, 0x0000, 0x10, 3, 3),
	PIN_FIELD_BASE(163, 163, 1, 0x0010, 0x10, 12, 3),
	PIN_FIELD_BASE(165, 165, 1, 0x0010, 0x10, 6, 3),
	PIN_FIELD_BASE(166, 166, 1, 0x0010, 0x10, 6, 3),
	PIN_FIELD_BASE(167, 167, 1, 0x0010, 0x10, 9, 3),
	PIN_FIELD_BASE(168, 168, 1, 0x0010, 0x10, 6, 3),
	PIN_FIELD_BASE(169, 169, 1, 0x0010, 0x10, 12, 3),
	PIN_FIELD_BASE(170, 170, 1, 0x0010, 0x10, 9, 3),
	PIN_FIELD_BASE(171, 171, 1, 0x0010, 0x10, 9, 3),
	PIN_FIELD_BASE(172, 172, 1, 0x0010, 0x10, 9, 3),
	PIN_FIELD_BASE(173, 173, 1, 0x0010, 0x10, 6, 3),
	PIN_FIELD_BASE(174, 174, 6, 0x0000, 0x10, 9, 3),
	PIN_FIELD_BASE(175, 175, 6, 0x0000, 0x10, 12, 3),
	PIN_FIELD_BASE(176, 176, 6, 0x0000, 0x10, 0, 3),
	PIN_FIELD_BASE(177, 177, 6, 0x0000, 0x10, 3, 3),
	PIN_FIELD_BASE(178, 178, 6, 0x0000, 0x10, 6, 3),
	PIN_FIELD_BASE(179, 179, 6, 0x0000, 0x10, 15, 3),
	PIN_FIELD_BASE(180, 180, 6, 0x0000, 0x10, 18, 3),
	PIN_FIELD_BASE(181, 181, 6, 0x0000, 0x10, 21, 3),
	PIN_FIELD_BASE(182, 182, 6, 0x0000, 0x10, 24, 3),
	PIN_FIELD_BASE(183, 183, 2, 0x0000, 0x10, 27, 3),
	PIN_FIELD_BASE(184, 184, 2, 0x0010, 0x10, 0, 3),
};

static const struct mtk_pin_field_calc mt8186_pin_drv_adv_range[] = {
	PIN_FIELD_BASE(127, 127, 3, 0x0030, 0x10, 0, 3),
	PIN_FIELD_BASE(128, 128, 3, 0x0030, 0x10, 6, 3),
	PIN_FIELD_BASE(129, 129, 5, 0x0030, 0x10, 0, 3),
	PIN_FIELD_BASE(130, 130, 5, 0x0030, 0x10, 6, 3),
	PIN_FIELD_BASE(131, 131, 5, 0x0030, 0x10, 3, 3),
	PIN_FIELD_BASE(132, 132, 5, 0x0030, 0x10, 9, 3),
	PIN_FIELD_BASE(133, 133, 1, 0x0030, 0x10, 0, 3),
	PIN_FIELD_BASE(134, 134, 1, 0x0030, 0x10, 6, 3),
	PIN_FIELD_BASE(135, 135, 6, 0x0020, 0x10, 0, 3),
	PIN_FIELD_BASE(136, 136, 6, 0x0020, 0x10, 9, 3),
	PIN_FIELD_BASE(137, 137, 1, 0x0030, 0x10, 3, 3),
	PIN_FIELD_BASE(138, 138, 1, 0x0030, 0x10, 9, 3),
	PIN_FIELD_BASE(139, 139, 2, 0x0020, 0x10, 0, 3),
	PIN_FIELD_BASE(140, 140, 2, 0x0020, 0x10, 3, 3),
	PIN_FIELD_BASE(141, 141, 3, 0x0030, 0x10, 3, 3),
	PIN_FIELD_BASE(142, 142, 3, 0x0030, 0x10, 9, 3),
	PIN_FIELD_BASE(143, 143, 6, 0x0020, 0x10, 3, 3),
	PIN_FIELD_BASE(144, 144, 6, 0x0020, 0x10, 12, 3),
	PIN_FIELD_BASE(145, 145, 6, 0x0020, 0x10, 6, 3),
	PIN_FIELD_BASE(146, 146, 6, 0x0020, 0x10, 15, 3),
};

static const struct mtk_pin_field_calc mt8186_pin_rsel_range[] = {
	PIN_FIELD_BASE(127, 127, 3, 0x00d0, 0x10, 0, 2),
	PIN_FIELD_BASE(128, 128, 3, 0x00d0, 0x10, 4, 2),
	PIN_FIELD_BASE(129, 129, 5, 0x00d0, 0x10, 0, 2),
	PIN_FIELD_BASE(130, 130, 5, 0x00d0, 0x10, 4, 2),
	PIN_FIELD_BASE(131, 131, 5, 0x00d0, 0x10, 2, 2),
	PIN_FIELD_BASE(132, 132, 5, 0x00d0, 0x10, 6, 2),
	PIN_FIELD_BASE(133, 133, 1, 0x00e0, 0x10, 0, 2),
	PIN_FIELD_BASE(134, 134, 1, 0x00e0, 0x10, 4, 2),
	PIN_FIELD_BASE(135, 135, 6, 0x0070, 0x10, 0, 2),
	PIN_FIELD_BASE(136, 136, 6, 0x0070, 0x10, 6, 2),
	PIN_FIELD_BASE(137, 137, 1, 0x00e0, 0x10, 2, 2),
	PIN_FIELD_BASE(138, 138, 1, 0x00e0, 0x10, 6, 2),
	PIN_FIELD_BASE(139, 139, 2, 0x0080, 0x10, 0, 2),
	PIN_FIELD_BASE(140, 140, 2, 0x0080, 0x10, 2, 2),
	PIN_FIELD_BASE(141, 141, 3, 0x00d0, 0x10, 2, 2),
	PIN_FIELD_BASE(142, 142, 3, 0x00d0, 0x10, 6, 2),
	PIN_FIELD_BASE(143, 143, 6, 0x0070, 0x10, 2, 2),
	PIN_FIELD_BASE(144, 144, 6, 0x0070, 0x10, 8, 2),
	PIN_FIELD_BASE(145, 145, 6, 0x0070, 0x10, 4, 2),
	PIN_FIELD_BASE(146, 146, 6, 0x0070, 0x10, 10, 2),
};

static const struct mtk_pin_rsel mt8186_pin_rsel_val_range[] = {
	PIN_RSEL(127, 128, 0x0, 75000, 75000),
	PIN_RSEL(127, 128, 0x1, 10000, 75000),
	PIN_RSEL(127, 128, 0x2, 5000, 5000),
	PIN_RSEL(127, 128, 0x3, 1000, 5000),
	PIN_RSEL(129, 130, 0x0, 75000, 75000),
	PIN_RSEL(129, 130, 0x1, 10000, 75000),
	PIN_RSEL(129, 130, 0x2, 5000, 5000),
	PIN_RSEL(129, 130, 0x3, 1000, 5000),
	PIN_RSEL(131, 132, 0x0, 75000, 75000),
	PIN_RSEL(131, 132, 0x1, 10000, 75000),
	PIN_RSEL(131, 132, 0x2, 5000, 5000),
	PIN_RSEL(131, 132, 0x3, 1000, 5000),
	PIN_RSEL(133, 134, 0x0, 75000, 75000),
	PIN_RSEL(133, 134, 0x1, 10000, 75000),
	PIN_RSEL(133, 134, 0x2, 5000, 5000),
	PIN_RSEL(133, 134, 0x3, 1000, 5000),
	PIN_RSEL(135, 136, 0x0, 75000, 75000),
	PIN_RSEL(135, 136, 0x1, 10000, 75000),
	PIN_RSEL(135, 136, 0x2, 5000, 5000),
	PIN_RSEL(135, 136, 0x3, 1000, 5000),
	PIN_RSEL(137, 138, 0x0, 75000, 75000),
	PIN_RSEL(137, 138, 0x1, 10000, 75000),
	PIN_RSEL(137, 138, 0x2, 5000, 5000),
	PIN_RSEL(137, 138, 0x3, 1000, 5000),
	PIN_RSEL(139, 140, 0x0, 75000, 75000),
	PIN_RSEL(139, 140, 0x1, 10000, 75000),
	PIN_RSEL(139, 140, 0x2, 5000, 5000),
	PIN_RSEL(139, 140, 0x3, 1000, 5000),
	PIN_RSEL(141, 142, 0x0, 75000, 75000),
	PIN_RSEL(141, 142, 0x1, 10000, 75000),
	PIN_RSEL(141, 142, 0x2, 5000, 5000),
	PIN_RSEL(141, 142, 0x3, 1000, 5000),
	PIN_RSEL(143, 144, 0x0, 75000, 75000),
	PIN_RSEL(143, 144, 0x1, 10000, 75000),
	PIN_RSEL(143, 144, 0x2, 5000, 5000),
	PIN_RSEL(143, 144, 0x3, 1000, 5000),
	PIN_RSEL(145, 146, 0x0, 75000, 75000),
	PIN_RSEL(145, 146, 0x1, 10000, 75000),
	PIN_RSEL(145, 146, 0x2, 5000, 5000),
	PIN_RSEL(145, 146, 0x3, 1000, 5000),
};

static const unsigned int mt8186_pull_type[] = {
	MTK_PULL_PU_PD_TYPE,/*0*/		MTK_PULL_PU_PD_TYPE,/*1*/
	MTK_PULL_PU_PD_TYPE,/*2*/		MTK_PULL_PU_PD_TYPE,/*3*/
	MTK_PULL_PU_PD_TYPE,/*4*/		MTK_PULL_PU_PD_TYPE,/*5*/
	MTK_PULL_PU_PD_TYPE,/*6*/		MTK_PULL_PU_PD_TYPE,/*7*/
	MTK_PULL_PU_PD_TYPE,/*8*/		MTK_PULL_PU_PD_TYPE,/*9*/
	MTK_PULL_PU_PD_TYPE,/*10*/		MTK_PULL_PU_PD_TYPE,/*11*/
	MTK_PULL_PU_PD_TYPE,/*12*/		MTK_PULL_PU_PD_TYPE,/*13*/
	MTK_PULL_PU_PD_TYPE,/*14*/		MTK_PULL_PU_PD_TYPE,/*15*/
	MTK_PULL_PU_PD_TYPE,/*16*/		MTK_PULL_PU_PD_TYPE,/*17*/
	MTK_PULL_PU_PD_TYPE,/*18*/		MTK_PULL_PU_PD_TYPE,/*19*/
	MTK_PULL_PU_PD_TYPE,/*20*/		MTK_PULL_PU_PD_TYPE,/*21*/
	MTK_PULL_PU_PD_TYPE,/*22*/		MTK_PULL_PU_PD_TYPE,/*23*/
	MTK_PULL_PU_PD_TYPE,/*24*/		MTK_PULL_PU_PD_TYPE,/*25*/
	MTK_PULL_PU_PD_TYPE,/*26*/		MTK_PULL_PU_PD_TYPE,/*27*/
	MTK_PULL_PU_PD_TYPE,/*28*/		MTK_PULL_PU_PD_TYPE,/*29*/
	MTK_PULL_PU_PD_TYPE,/*30*/		MTK_PULL_PU_PD_TYPE,/*31*/
	MTK_PULL_PU_PD_TYPE,/*32*/		MTK_PULL_PU_PD_TYPE,/*33*/
	MTK_PULL_PU_PD_TYPE,/*34*/		MTK_PULL_PU_PD_TYPE,/*35*/
	MTK_PULL_PU_PD_TYPE,/*36*/		MTK_PULL_PU_PD_TYPE,/*37*/
	MTK_PULL_PU_PD_TYPE,/*38*/		MTK_PULL_PU_PD_TYPE,/*39*/
	MTK_PULL_PU_PD_TYPE,/*40*/		MTK_PULL_PU_PD_TYPE,/*41*/
	MTK_PULL_PU_PD_TYPE,/*42*/		MTK_PULL_PU_PD_TYPE,/*43*/
	MTK_PULL_PU_PD_TYPE,/*44*/		MTK_PULL_PU_PD_TYPE,/*45*/
	MTK_PULL_PU_PD_TYPE,/*46*/		MTK_PULL_PU_PD_TYPE,/*47*/
	MTK_PULL_PU_PD_TYPE,/*48*/		MTK_PULL_PU_PD_TYPE,/*49*/
	MTK_PULL_PU_PD_TYPE,/*50*/		MTK_PULL_PU_PD_TYPE,/*51*/
	MTK_PULL_PU_PD_TYPE,/*52*/		MTK_PULL_PU_PD_TYPE,/*53*/
	MTK_PULL_PU_PD_TYPE,/*54*/		MTK_PULL_PU_PD_TYPE,/*55*/
	MTK_PULL_PU_PD_TYPE,/*56*/		MTK_PULL_PU_PD_TYPE,/*57*/
	MTK_PULL_PU_PD_TYPE,/*58*/		MTK_PULL_PU_PD_TYPE,/*59*/
	MTK_PULL_PU_PD_TYPE,/*60*/		MTK_PULL_PU_PD_TYPE,/*61*/
	MTK_PULL_PU_PD_TYPE,/*62*/		MTK_PULL_PU_PD_TYPE,/*63*/
	MTK_PULL_PU_PD_TYPE,/*64*/		MTK_PULL_PU_PD_TYPE,/*65*/
	MTK_PULL_PU_PD_TYPE,/*66*/		MTK_PULL_PUPD_R1R0_TYPE,/*67*/
	MTK_PULL_PUPD_R1R0_TYPE,/*68*/		MTK_PULL_PUPD_R1R0_TYPE,/*69*/
	MTK_PULL_PUPD_R1R0_TYPE,/*70*/		MTK_PULL_PUPD_R1R0_TYPE,/*71*/
	MTK_PULL_PUPD_R1R0_TYPE,/*72*/		MTK_PULL_PUPD_R1R0_TYPE,/*73*/
	MTK_PULL_PUPD_R1R0_TYPE,/*74*/		MTK_PULL_PUPD_R1R0_TYPE,/*75*/
	MTK_PULL_PUPD_R1R0_TYPE,/*76*/		MTK_PULL_PUPD_R1R0_TYPE,/*77*/
	MTK_PULL_PUPD_R1R0_TYPE,/*78*/		MTK_PULL_PUPD_R1R0_TYPE,/*79*/
	MTK_PULL_PUPD_R1R0_TYPE,/*80*/		MTK_PULL_PUPD_R1R0_TYPE,/*81*/
	MTK_PULL_PUPD_R1R0_TYPE,/*82*/		MTK_PULL_PU_PD_TYPE,/*83*/
	MTK_PULL_PUPD_R1R0_TYPE,/*84*/		MTK_PULL_PUPD_R1R0_TYPE,/*85*/
	MTK_PULL_PUPD_R1R0_TYPE,/*86*/		MTK_PULL_PUPD_R1R0_TYPE,/*87*/
	MTK_PULL_PUPD_R1R0_TYPE,/*88*/		MTK_PULL_PUPD_R1R0_TYPE,/*89*/
	MTK_PULL_PU_PD_TYPE,/*90*/		MTK_PULL_PU_PD_TYPE,/*91*/
	MTK_PULL_PU_PD_TYPE,/*92*/		MTK_PULL_PU_PD_TYPE,/*93*/
	MTK_PULL_PU_PD_TYPE,/*94*/		MTK_PULL_PU_PD_TYPE,/*95*/
	MTK_PULL_PU_PD_TYPE,/*96*/		MTK_PULL_PU_PD_TYPE,/*97*/
	MTK_PULL_PU_PD_TYPE,/*98*/		MTK_PULL_PU_PD_TYPE,/*99*/
	MTK_PULL_PU_PD_TYPE,/*100*/		MTK_PULL_PU_PD_TYPE,/*101*/
	MTK_PULL_PU_PD_TYPE,/*102*/		MTK_PULL_PU_PD_TYPE,/*103*/
	MTK_PULL_PU_PD_TYPE,/*104*/		MTK_PULL_PU_PD_TYPE,/*105*/
	MTK_PULL_PU_PD_TYPE,/*106*/		MTK_PULL_PU_PD_TYPE,/*107*/
	MTK_PULL_PU_PD_TYPE,/*108*/		MTK_PULL_PU_PD_TYPE,/*109*/
	MTK_PULL_PU_PD_TYPE,/*110*/		MTK_PULL_PU_PD_TYPE,/*111*/
	MTK_PULL_PU_PD_TYPE,/*112*/		MTK_PULL_PU_PD_TYPE,/*113*/
	MTK_PULL_PU_PD_TYPE,/*114*/		MTK_PULL_PU_PD_TYPE,/*115*/
	MTK_PULL_PU_PD_TYPE,/*116*/		MTK_PULL_PU_PD_TYPE,/*117*/
	MTK_PULL_PU_PD_TYPE,/*118*/		MTK_PULL_PU_PD_TYPE,/*119*/
	MTK_PULL_PU_PD_TYPE,/*120*/		MTK_PULL_PU_PD_TYPE,/*121*/
	MTK_PULL_PU_PD_TYPE,/*122*/		MTK_PULL_PU_PD_TYPE,/*123*/
	MTK_PULL_PU_PD_TYPE,/*124*/		MTK_PULL_PU_PD_TYPE,/*125*/
	MTK_PULL_PU_PD_TYPE,/*126*/		MTK_PULL_PU_PD_RSEL_TYPE,/*127*/
	MTK_PULL_PU_PD_RSEL_TYPE,/*128*/	MTK_PULL_PU_PD_RSEL_TYPE,/*129*/
	MTK_PULL_PU_PD_RSEL_TYPE,/*130*/	MTK_PULL_PU_PD_RSEL_TYPE,/*131*/
	MTK_PULL_PU_PD_RSEL_TYPE,/*132*/	MTK_PULL_PU_PD_RSEL_TYPE,/*133*/
	MTK_PULL_PU_PD_RSEL_TYPE,/*134*/	MTK_PULL_PU_PD_RSEL_TYPE,/*135*/
	MTK_PULL_PU_PD_RSEL_TYPE,/*136*/	MTK_PULL_PU_PD_RSEL_TYPE,/*137*/
	MTK_PULL_PU_PD_RSEL_TYPE,/*138*/	MTK_PULL_PU_PD_RSEL_TYPE,/*139*/
	MTK_PULL_PU_PD_RSEL_TYPE,/*140*/	MTK_PULL_PU_PD_RSEL_TYPE,/*141*/
	MTK_PULL_PU_PD_RSEL_TYPE,/*142*/	MTK_PULL_PU_PD_RSEL_TYPE,/*143*/
	MTK_PULL_PU_PD_RSEL_TYPE,/*144*/	MTK_PULL_PU_PD_RSEL_TYPE,/*145*/
	MTK_PULL_PU_PD_RSEL_TYPE,/*146*/	MTK_PULL_PU_PD_TYPE,/*147*/
	MTK_PULL_PU_PD_TYPE,/*148*/		MTK_PULL_PU_PD_TYPE,/*149*/
	MTK_PULL_PU_PD_TYPE,/*150*/		MTK_PULL_PU_PD_TYPE,/*151*/
	MTK_PULL_PU_PD_TYPE,/*152*/		MTK_PULL_PU_PD_TYPE,/*153*/
	MTK_PULL_PU_PD_TYPE,/*154*/		MTK_PULL_PU_PD_TYPE,/*155*/
	MTK_PULL_PU_PD_TYPE,/*156*/		MTK_PULL_PU_PD_TYPE,/*157*/
	MTK_PULL_PU_PD_TYPE,/*158*/		MTK_PULL_PU_PD_TYPE,/*159*/
	MTK_PULL_PU_PD_TYPE,/*160*/		MTK_PULL_PU_PD_TYPE,/*161*/
	MTK_PULL_PU_PD_TYPE,/*162*/		MTK_PULL_PU_PD_TYPE,/*163*/
	MTK_PULL_PU_PD_TYPE,/*164*/		MTK_PULL_PU_PD_TYPE,/*165*/
	MTK_PULL_PU_PD_TYPE,/*166*/		MTK_PULL_PU_PD_TYPE,/*167*/
	MTK_PULL_PU_PD_TYPE,/*168*/		MTK_PULL_PU_PD_TYPE,/*169*/
	MTK_PULL_PU_PD_TYPE,/*170*/		MTK_PULL_PU_PD_TYPE,/*171*/
	MTK_PULL_PU_PD_TYPE,/*172*/		MTK_PULL_PU_PD_TYPE,/*173*/
	MTK_PULL_PU_PD_TYPE,/*174*/		MTK_PULL_PU_PD_TYPE,/*175*/
	MTK_PULL_PU_PD_TYPE,/*176*/		MTK_PULL_PU_PD_TYPE,/*177*/
	MTK_PULL_PU_PD_TYPE,/*178*/		MTK_PULL_PU_PD_TYPE,/*179*/
	MTK_PULL_PU_PD_TYPE,/*180*/		MTK_PULL_PU_PD_TYPE,/*181*/
	MTK_PULL_PU_PD_TYPE,/*182*/		MTK_PULL_PU_PD_TYPE,/*183*/
	MTK_PULL_PU_PD_TYPE,/*184*/
};

static const struct mtk_pin_reg_calc mt8186_reg_cals[PINCTRL_PIN_REG_MAX] = {
	[PINCTRL_PIN_REG_MODE] = MTK_RANGE(mt8186_pin_mode_range),
	[PINCTRL_PIN_REG_DIR] = MTK_RANGE(mt8186_pin_dir_range),
	[PINCTRL_PIN_REG_DI] = MTK_RANGE(mt8186_pin_di_range),
	[PINCTRL_PIN_REG_DO] = MTK_RANGE(mt8186_pin_do_range),
	[PINCTRL_PIN_REG_SR] = MTK_RANGE(mt8186_pin_dir_range),
	[PINCTRL_PIN_REG_SMT] = MTK_RANGE(mt8186_pin_smt_range),
	[PINCTRL_PIN_REG_IES] = MTK_RANGE(mt8186_pin_ies_range),
	[PINCTRL_PIN_REG_PU] = MTK_RANGE(mt8186_pin_pu_range),
	[PINCTRL_PIN_REG_PD] = MTK_RANGE(mt8186_pin_pd_range),
	[PINCTRL_PIN_REG_DRV] = MTK_RANGE(mt8186_pin_drv_range),
	[PINCTRL_PIN_REG_PUPD] = MTK_RANGE(mt8186_pin_pupd_range),
	[PINCTRL_PIN_REG_R0] = MTK_RANGE(mt8186_pin_r0_range),
	[PINCTRL_PIN_REG_R1] = MTK_RANGE(mt8186_pin_r1_range),
	[PINCTRL_PIN_REG_DRV_ADV] = MTK_RANGE(mt8186_pin_drv_adv_range),
	[PINCTRL_PIN_REG_RSEL]	= MTK_RANGE(mt8186_pin_rsel_range),

};

static const char * const mt8186_pinctrl_register_base_names[] = {
	"iocfg0", "iocfg_lt", "iocfg_lm", "iocfg_lb", "iocfg_bl",
	"iocfg_rb", "iocfg_rt",
};

static const struct mtk_eint_hw mt8186_eint_hw = {
	.port_mask = 0xf,
	.ports     = 7,
	.ap_num    = 217,
	.db_cnt    = 32,
	.db_time   = debounce_time_mt6765,
};

static const struct mtk_pin_soc mt8186_data = {
	.reg_cal = mt8186_reg_cals,
	.pins = mtk_pins_mt8186,
	.npins = ARRAY_SIZE(mtk_pins_mt8186),
	.ngrps = ARRAY_SIZE(mtk_pins_mt8186),
	.nfuncs = 8,
	.gpio_m = 0,
	.eint_hw = &mt8186_eint_hw,
	.base_names = mt8186_pinctrl_register_base_names,
	.nbase_names = ARRAY_SIZE(mt8186_pinctrl_register_base_names),
	.pull_type = mt8186_pull_type,
	.pin_rsel = mt8186_pin_rsel_val_range,
	.npin_rsel = ARRAY_SIZE(mt8186_pin_rsel_val_range),
	.bias_set_combo = mtk_pinconf_bias_set_combo,
	.bias_get_combo = mtk_pinconf_bias_get_combo,
	.drive_set = mtk_pinconf_drive_set_rev1,
	.drive_get = mtk_pinconf_drive_get_rev1,
	.adv_drive_get = mtk_pinconf_adv_drive_get_raw,
	.adv_drive_set = mtk_pinconf_adv_drive_set_raw,
};

static const struct of_device_id mt8186_pinctrl_of_match[] = {
	{ .compatible = "mediatek,mt8186-pinctrl", .data = &mt8186_data },
	{ }
};

static struct platform_driver mt8186_pinctrl_driver = {
	.driver = {
		.name = "mt8186-pinctrl",
		.of_match_table = mt8186_pinctrl_of_match,
		.pm = &mtk_paris_pinctrl_pm_ops,
	},
	.probe = mtk_paris_pinctrl_probe,
};

static int __init mt8186_pinctrl_init(void)
{
	return platform_driver_register(&mt8186_pinctrl_driver);
}

arch_initcall(mt8186_pinctrl_init);
