/*
    This file is part of KolabAdmin.

    Copyright (C) 2006 Tobias Koenig <tobias.koenig@credativ.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

#include <QtCore/QAbstractListModel>

#include "connection.h"
#include "itemview.h"
#include "ldapsearchdialog.h"

#include "memberselectionfield.h"

using namespace Form;


class MemberModel : public QAbstractListModel
{
  public:
    MemberModel( QObject *parent = 0 )
      : QAbstractListModel( parent )
    {
    }

    virtual int columnCount( const QModelIndex &parent = QModelIndex() ) const
    {
      return 1;
    }

    virtual int rowCount( const QModelIndex &parent = QModelIndex() ) const
    {
      return mList.count();
    }

    virtual QVariant data( const QModelIndex &index, int role = Qt::DisplayRole ) const
    {
      if ( !index.isValid() )
        return QVariant();

      if ( role == Qt::DisplayRole )
        return mList[ index.row() ].displayName;
      else if ( role == Qt::UserRole )
        return mList[ index.row() ].dn;
      else
        return QVariant();
    }

    virtual bool setData( const QModelIndex &index, const QVariant &value, int role = Qt::EditRole )
    {
      if ( !index.isValid() ) {
        return false;
      }

      if ( !value.isValid() )
        return false;

      if ( role == Qt::UserRole ) {
        mList[ index.row() ].dn = value.toString();

        QLdapResponse response = Connection::self()->search( value.toString(), QLdap::Base, "objectClass=*" );
        if ( response.isValid() && !response.entries().isEmpty() ) {
          QLdapEntry entry = response.entries().first();
          mList[ index.row() ].displayName = QString( "%1 %2" ).arg( entry.value( "givenName" ), entry.value( "sn" ) );
        } else {
          mList[ index.row() ].displayName = QString( "<not found>" );
        }

        qSort( mList );
      }

      return true;
    }

    virtual bool insertRows( int row, int count, const QModelIndex &parent = QModelIndex() )
    {
      beginInsertRows( parent, row, row + count );

      for ( int i = 0; i < count; ++i )
        mList.insert( row, Entry() );

      endInsertRows();

      return true;
    }

    virtual bool removeRows( int row, int count, const QModelIndex &parent = QModelIndex() )
    {
      beginRemoveRows( parent, row, row + count );

      for ( int i = 0; i < count; ++i )
        mList.removeAt( row );

      endRemoveRows();

      return true;
    }

  private:
    class Entry
    {
      public:
        QString displayName;
        QString dn;

        bool operator< ( const Entry &other ) const
        {
          return QString::localeAwareCompare( this->displayName, other.displayName ) <= 0;
        }
    };

    QList<Entry> mList;
};


class MemberItemView : public ItemView
{
  public:
    MemberItemView( QWidget *parent )
      : ItemView( parent )
    {
    }

  protected:
    virtual void doAdd()
    {
      QStringList blacklist;
      blacklist << QString( "cn=manager,cn=internal,%1" ).arg( Connection::self()->baseDn() );
      blacklist << QString( "cn=calendar,cn=internal,%1" ).arg( Connection::self()->baseDn() );
      blacklist << QString( "cn=nobody,cn=internal,%1" ).arg( Connection::self()->baseDn() );

      LdapSearchDialog dlg( Connection::self(), this );
      dlg.setSearchFilter( "(&(objectClass=kolabInetOrgPerson)(|(givenName=*%i*)(sn=*%i*)))" );
      dlg.setDisplayFormat( "%sn%, %givenName% (%mail%)" );
      dlg.setBlackList( blacklist );

      if ( dlg.exec() ) {
        QStringList dns = dlg.selectedValues( "dn" );

        QAbstractItemModel *model = mListView->model();
        int j = model->rowCount();
        for ( int i = 0; i < dns.count(); ++i ) {

          // check whether dn is already part of the model
          bool found = false;
          for ( int k = 0; k < model->rowCount(); ++k ) {
            if ( dns[ i ] == model->data( model->index( k, 0 ), Qt::UserRole ) )
              found = true;
          }

          if ( !found ) {
            model->insertRow( j );
            model->setData( model->index( j, 0 ), dns[ i ], Qt::UserRole );
            ++j;
          }
        }
      }
    }
};


MemberSelectionField::MemberSelectionField( const QString &name, QWidget *parent )
  : Field( name, parent )
{
  mView = new MemberItemView( this );

  mModel = new MemberModel( this );
  mView->setModel( mModel );

  fieldLayout()->addWidget( mView, 0, 0 );
}

void MemberSelectionField::loadEntry( const Entry &entry )
{
  QStringList members = entry.values( "member" );
  for ( int i = 0; i < members.count(); ++i ) {
    mModel->insertRow( i );
    mModel->setData( mModel->index( i, 0 ), members[ i ], Qt::UserRole );
  }
}

void MemberSelectionField::saveEntry( Entry &entry ) const
{
  entry.clearValue( "member" );

  for ( int i = 0; i < mModel->rowCount(); ++i )
    entry.addValue( "member", mModel->data( mModel->index( i, 0 ), Qt::UserRole ).toString() );
}

bool MemberSelectionField::isValid( QString &errorMsg, PagePolicy::State ) const
{
  if ( mModel->rowCount() == 0 ) {
    errorMsg = QObject::tr( "You have to add at least one member!" );
    return false;
  } else {
    return true;
  }
}

void MemberSelectionField::setEditable( bool editable )
{
  mView->setEnabled( editable );
}

