/* $Id: Seq_loc.cpp 449954 2014-10-22 13:28:16Z ivanov $
 * ===========================================================================
 *
 *                            PUBLIC DOMAIN NOTICE
 *               National Center for Biotechnology Information
 *
 *  This software/database is a "United States Government Work" under the
 *  terms of the United States Copyright Act.  It was written as part of
 *  the author's official duties as a United States Government employee and
 *  thus cannot be copyrighted.  This software/database is freely available
 *  to the public for use. The National Library of Medicine and the U.S.
 *  Government have not placed any restriction on its use or reproduction.
 *
 *  Although all reasonable efforts have been taken to ensure the accuracy
 *  and reliability of the software and data, the NLM and the U.S.
 *  Government do not and cannot warrant the performance or results that
 *  may be obtained by using this software or data. The NLM and the U.S.
 *  Government disclaim all warranties, express or implied, including
 *  warranties of performance, merchantability or fitness for any particular
 *  purpose.
 *
 *  Please cite the author in any work or product based on this material.
 *
 * ===========================================================================
 *
 * Author:   Author:  Cliff Clausen, Eugene Vasilchenko
 *
 * File Description:
 *   .......
 *
 * Remark:
 *   This code was originally generated by application DATATOOL
 *   using specifications from the ASN data definition file
 *   'seqloc.asn'.
 *
 * ===========================================================================
 */

#include <ncbi_pch.hpp>
#include <serial/enumvalues.hpp>
#include <objects/general/Int_fuzz.hpp>
#include <objects/seqloc/Seq_point.hpp>
#include <objects/seqloc/Seq_interval.hpp>
#include <objects/seqloc/Packed_seqint.hpp>
#include <objects/seqloc/Packed_seqpnt.hpp>
#include <objects/seqloc/Seq_loc_mix.hpp>
#include <objects/seqloc/Seq_loc_equiv.hpp>
#include <objects/seqloc/Seq_bond.hpp>
#include <objects/seqloc/Seq_loc.hpp>
#include <objects/seqfeat/Feat_id.hpp>
#include <objects/misc/error_codes.hpp>
#include <util/range_coll.hpp>
#include <objects/seq/seq_id_handle.hpp>
#include <algorithm>


#define NCBI_USE_ERRCODE_X   Objects_SeqLoc

BEGIN_NCBI_SCOPE
BEGIN_objects_SCOPE // namespace ncbi::objects::

// constructors
CSeq_loc::CSeq_loc(E_Choice index)
{
    InvalidateCache();
    switch ( index ) {
    case e_Null:
        {
            SetNull();
            break;
        }
    case e_Empty:
        {
            SetEmpty();
            break;
        }
    case e_Whole:
        {
            SetWhole();
            break;
        }
    case e_Int:
        {
            SetInt();
            break;
        }
    case e_Packed_int:
        {
            SetPacked_int();
            break;
        }
    case e_Pnt:
        {
            SetPnt();
            break;
        }
    case e_Packed_pnt:
        {
            SetPacked_pnt();
            break;
        }
    case e_Mix:
        {
            SetMix();
            break;
        }
    case e_Equiv:
        {
            SetEquiv();
            break;
        }
    case e_Bond:
        {
            SetBond();
            break;
        }
    case e_Feat:
        {
            SetFeat();
            break;
        }
    case e_not_set:
    default:
        break;
    }
}


CSeq_loc::CSeq_loc(TId& id, TPoint point, TStrand strand)
{
    InvalidateCache();
    SetPnt(*new CSeq_point(id, point, strand));
}


CSeq_loc::CSeq_loc(TId& id, const TPoints& points, TStrand strand)
{
    InvalidateCache();
    if ( points.size() == 1 ) {
        SetPnt(*new CSeq_point(id, points.front(), strand));
    } else {
        SetPacked_pnt(*new CPacked_seqpnt(id, points, strand));
    }
}


CSeq_loc::CSeq_loc(TId& id, TPoint from, TPoint to, TStrand strand)
{
    InvalidateCache();
    SetInt(*new CSeq_interval(id, from, to, strand));
}


CSeq_loc::CSeq_loc(TId& id, TRanges ranges, TStrand strand)
{
    InvalidateCache();
    if ( ranges.size() == 1 ) {
        SetInt(*new CSeq_interval(id,
            ranges.front().GetFrom(), ranges.front().GetTo(), strand));
    } else {
        SetPacked_int(*new CPacked_seqint(id, ranges, strand));
    }
}


// destructor
CSeq_loc::~CSeq_loc(void)
{
}


inline
void x_Assign(CInt_fuzz& dst, const CInt_fuzz& src)
{
    switch ( src.Which() ) {
    case CInt_fuzz::e_not_set:
        dst.Reset();
        break;
    case CInt_fuzz::e_P_m:
        dst.SetP_m(src.GetP_m());
        break;
    case CInt_fuzz::e_Range:
        dst.SetRange().SetMin(src.GetRange().GetMin());
        dst.SetRange().SetMax(src.GetRange().GetMax());
        break;
    case CInt_fuzz::e_Pct:
        dst.SetPct(src.GetPct());
        break;
    case CInt_fuzz::e_Lim:
        dst.SetLim(src.GetLim());
        break;
    case CInt_fuzz::e_Alt:
        dst.SetAlt() = src.GetAlt();
        break;
    default:
        NCBI_THROW(CException, eUnknown,
            "Invalid Int-fuzz variant");
    }
}


inline
void x_Assign(CSeq_interval& dst, const CSeq_interval& src)
{
    dst.SetFrom(src.GetFrom());
    dst.SetTo(src.GetTo());
    if ( src.IsSetStrand() ) {
        dst.SetStrand(src.GetStrand());
    }
    else {
        dst.ResetStrand();
    }
    dst.SetId().Assign(src.GetId());
    if ( src.IsSetFuzz_from() ) {
        x_Assign(dst.SetFuzz_from(), src.GetFuzz_from());
    }
    else {
        dst.ResetFuzz_from();
    }
    if ( src.IsSetFuzz_to() ) {
        x_Assign(dst.SetFuzz_to(), src.GetFuzz_to());
    }
    else {
        dst.ResetFuzz_to();
    }
}


inline
void x_Assign(CSeq_point& dst, const CSeq_point& src)
{
    dst.SetPoint(src.GetPoint());
    if ( src.IsSetStrand() ) {
        dst.SetStrand(src.GetStrand());
    }
    else {
        dst.ResetStrand();
    }
    dst.SetId().Assign(src.GetId());
    if ( src.IsSetFuzz() ) {
        x_Assign(dst.SetFuzz(), src.GetFuzz());
    }
    else {
        dst.ResetFuzz();
    }
}


inline
void x_Assign(CPacked_seqint& dst, const CPacked_seqint& src)
{
    CPacked_seqint::Tdata& data = dst.Set();
    data.clear();
    ITERATE ( CPacked_seqint::Tdata, i, src.Get() ) {
        data.push_back(CRef<CSeq_interval>(new CSeq_interval));
        x_Assign(*data.back(), **i);
    }
}


inline
void x_Assign(CPacked_seqpnt& dst, const CPacked_seqpnt& src)
{
    if ( src.IsSetStrand() ) {
        dst.SetStrand(src.GetStrand());
    }
    else {
        dst.ResetStrand();
    }
    dst.SetId().Assign(src.GetId());
    if ( src.IsSetFuzz() ) {
        x_Assign(dst.SetFuzz(), src.GetFuzz());
    }
    else {
        dst.ResetFuzz();
    }
    dst.SetPoints() = src.GetPoints();
}


inline
void x_Assign(CSeq_bond& dst, const CSeq_bond& src)
{
    x_Assign(dst.SetA(), src.GetA());
    if ( src.IsSetB() ) {
        x_Assign(dst.SetB(), src.GetB());
    }
    else {
        dst.ResetB();
    }
}


inline
void x_Assign(CSeq_loc_mix& dst, const CSeq_loc_mix& src)
{
    CSeq_loc_mix::Tdata& data = dst.Set();
    data.clear();
    ITERATE ( CSeq_loc_mix::Tdata, i, src.Get() ) {
        data.push_back(CRef<CSeq_loc>(new CSeq_loc));
        data.back()->Assign(**i);
    }
}


inline
void x_Assign(CSeq_loc_equiv& dst, const CSeq_loc_equiv& src)
{
    CSeq_loc_equiv::Tdata& data = dst.Set();
    data.clear();
    ITERATE ( CSeq_loc_equiv::Tdata, i, src.Get() ) {
        data.push_back(CRef<CSeq_loc>(new CSeq_loc));
        data.back()->Assign(**i);
    }
}


void CSeq_loc::Assign(const CSerialObject& obj, ESerialRecursionMode how)
{
    InvalidateCache();
    if ( GetTypeInfo() == obj.GetThisTypeInfo() ) {
        const CSeq_loc& loc = static_cast<const CSeq_loc&>(obj);
        switch ( loc.Which() ) {
        case e_not_set:
            Reset();
            return;
        case e_Null:
            SetNull();
            return;
        case e_Empty:
            SetEmpty().Assign(loc.GetEmpty());
            return;
        case e_Whole:
            SetWhole().Assign(loc.GetWhole());
            return;
        case e_Int:
            x_Assign(SetInt(), loc.GetInt());
            return;
        case e_Pnt:
            x_Assign(SetPnt(), loc.GetPnt());
            return;
        case e_Packed_int:
            x_Assign(SetPacked_int(), loc.GetPacked_int());
            return;
        case e_Packed_pnt:
            x_Assign(SetPacked_pnt(), loc.GetPacked_pnt());
            return;
        case e_Mix:
            x_Assign(SetMix(), loc.GetMix());
            return;
        case e_Equiv:
            x_Assign(SetEquiv(), loc.GetEquiv());
            return;
        case e_Bond:
            x_Assign(SetBond(), loc.GetBond());
            return;
        case e_Feat:
            SetFeat().Assign(loc.GetFeat());
            return;
        }
    }
    CSerialObject::Assign(obj, how);
}


CSeq_loc::TRange CSeq_loc::x_UpdateTotalRange(void) const
{
    TRange range = m_TotalRangeCache;
    if ( range.GetFrom() == TSeqPos(kDirtyCache) ) {
        const CSeq_id* id = 0;
        range = x_CalculateTotalRangeCheckId(id);
        m_IdCache = id;
        m_TotalRangeCache.SetToOpen(range.GetToOpen());
        m_TotalRangeCache.SetFrom(range.GetFrom());
    }
    return range;
}


bool CSeq_loc::x_UpdateId(const CSeq_id*& total_id, const CSeq_id* id,
                          bool may_throw) const
{
    if ( total_id == id ) {
        return true;
    }

    if ( !total_id ) {
        total_id = id;
    } else if ( (id  &&  !total_id->Equals(*id)) ) {
        if (may_throw) {
            NCBI_THROW(CException, eUnknown,
                       "CSeq_loc::GetTotalRange() is not defined "
                       "for seq-loc with several different seq-ids");
        } else {
            return false;
        }
    }
    return true;
}



// returns enclosing location range
// the total range is meaningless if there are several seq-ids
// in the location
CSeq_loc::TRange CSeq_loc::x_CalculateTotalRangeCheckId(const CSeq_id*& id) const
{
    TRange total_range;
    switch ( Which() ) {
    case e_not_set:
    case e_Null:
        {
            // Ignore locations without id
            total_range = TRange::GetEmpty();
            break;
        }
    case e_Empty:
        {
            x_UpdateId(id, &GetEmpty());
            total_range = TRange::GetEmpty();
            break;
        }
    case e_Whole:
        {
            x_UpdateId(id, &GetWhole());
            total_range = TRange::GetWhole();
            break;
        }
    case e_Int:
        {
            const CSeq_interval& loc = GetInt();
            x_UpdateId(id, &loc.GetId());
            total_range.Set(loc.GetFrom(), loc.GetTo());
            break;
        }
    case e_Pnt:
        {
            const CSeq_point& pnt = GetPnt();
            x_UpdateId(id, &pnt.GetId());
            TSeqPos pos = pnt.GetPoint();
            total_range.Set(pos, pos);
            break;
        }
    case e_Packed_int:
        {
            // Check ID of each interval
            const CPacked_seqint& ints = GetPacked_int();
            total_range = TRange::GetEmpty();
            ITERATE ( CPacked_seqint::Tdata, ii, ints.Get() ) {
                const CSeq_interval& loc = **ii;
                x_UpdateId(id, &loc.GetId());
                total_range += TRange(loc.GetFrom(), loc.GetTo());
            }
            break;
        }
    case e_Packed_pnt:
        {
            const CPacked_seqpnt& pnts = GetPacked_pnt();
            x_UpdateId(id, &pnts.GetId());
            total_range = TRange::GetEmpty();
            ITERATE( CPacked_seqpnt::TPoints, pi, pnts.GetPoints() ) {
                TSeqPos pos = *pi;
                total_range += TRange(pos, pos);
            }
            break;
        }
    case e_Mix:
        {
            // Check ID of each sub-location.
            const CSeq_loc_mix& mix = GetMix();
            total_range = TRange::GetEmpty();
            ITERATE( CSeq_loc_mix::Tdata, li, mix.Get() ) {
                // instead of Get... to be able to check ID
                total_range += (*li)->x_CalculateTotalRangeCheckId(id);
            }
            break;
        }
    case e_Equiv:
        {
            // Does it make any sense to GetTotalRange() from an equiv?
            const CSeq_loc_equiv::Tdata& equiv = GetEquiv().Get();
            total_range = TRange::GetEmpty();
            ITERATE( CSeq_loc_equiv::Tdata, li, equiv ) {
                total_range += (*li)->x_CalculateTotalRangeCheckId(id);
            }
            break;
        }
    case e_Bond:
        {
            // Does it make any sense to GetTotalRange() from a bond?
            const CSeq_bond& bond = GetBond();
            const CSeq_point& pointA = bond.GetA();
            x_UpdateId(id, &pointA.GetId());
            TSeqPos pos = pointA.GetPoint();
            total_range = TRange(pos, pos);
            if ( bond.IsSetB() ) {
                const CSeq_point& pointB = bond.GetB();
                x_UpdateId(id, &pointB.GetId());
                pos = pointB.GetPoint();
                total_range += TRange(pos, pos);
            }
            break;
        }
    case e_Feat:
    default:
        {
            NCBI_THROW(CException, eUnknown,
                       "CSeq_loc::CalculateTotalRange -- "
                       "unsupported location type");
        }
    }

    return total_range;
}


int CSeq_loc::x_CompareSingleId(const CSeq_loc& loc, const CSeq_id* id1,
                                const CSeq_id* id2) const
{
    if ( !id1 || !id2 ) {
        NCBI_THROW(CException, eUnknown,
                   "CSeq_loc::Compare(): "
                   "cannot compare locations with several different seq-ids");
    }
    if ( int diff = INT_ID_TO(int, id1->CompareOrdered(*id2)) ) {
        // ids are different - order by them
        return diff;
    }

    TSeqPos from1 = GetStart(eExtreme_Positional);
    TSeqPos to1 = GetStop(eExtreme_Positional);
    TSeqPos from2 = loc.GetStart(eExtreme_Positional);
    TSeqPos to2 = loc.GetStop(eExtreme_Positional);

    // (from > to) means circular location.
    // Any circular location is less than (before) non-circular one.
    // If both are circular, compare them regular way.
    bool circular1 = from1 > to1;
    bool circular2 = from2 > to2;
    if ( int diff = circular2 - circular1 ) {
        return diff;
    }

    // smallest left extreme first
    if ( from1 != from2 ) {
        return from1 < from2? -1: 1;
    }
    // longest feature first
    if ( to1 != to2 ) {
        return to1 > to2? -1: 1;
    }

    return 0;
}


int CSeq_loc::Compare(const CSeq_loc& loc) const
{
    // first try fast single-id comparison
    const CSeq_id* id1 = GetId();
    const CSeq_id* id2 = id1 == NULL ? NULL : loc.GetId();
    if (id1 != NULL  &&  id2 != NULL) {
        return x_CompareSingleId(loc, id1, id2);
    }
    // Slow comparison of ranges on each Seq-id separately.
    CSeq_loc_CI iter1(*this, CSeq_loc_CI::eEmpty_Allow);
    CSeq_loc_CI iter2(  loc, CSeq_loc_CI::eEmpty_Allow);
    for ( ; iter1 && iter2; ) {
        CRef<CSeq_loc> loc1, loc2;
        for ( int k = 0; k < 2; ++k ) {
            CSeq_loc_CI& iter = k? iter2: iter1;
            CRef<CSeq_loc>& loc = k? loc2: loc1;
            // skip null locations (no Seq-id)
            while ( iter && iter.GetSeq_id().Which() == CSeq_id::e_not_set ) {
                ++iter;
            }
            if ( !iter ) {
                // all the remaining segments were null -> end of location
                loc = null;
                continue;
            }
            const CSeq_id& id = iter.GetSeq_id();
            loc = const_cast<CSeq_loc*>(&*iter.GetRangeAsSeq_loc());
            // skip all sub-locations with the same Seq-id
            while ( ++iter ) {
                if ( !iter.GetSeq_id().Equals(id) ) {
                    if ( iter.GetSeq_id().Which() == CSeq_id::e_not_set ) {
                        // skip null sub-location
                        continue;
                    }
                    else {
                        // different non-null locations (has Seq-id)
                        break;
                    }
                }
                if ( !loc->IsMix() ) {
                    CRef<CSeq_loc> tmp = loc;
                    loc = new CSeq_loc;
                    loc->SetMix().AddSeqLoc(*tmp);
                }
                loc->SetMix().AddSeqLoc(const_cast<CSeq_loc&>(*iter.GetRangeAsSeq_loc()));
            }
        }
        if ( loc1 && !loc2 ) {
            // *this location is longer -> *this > loc
            return 1;
        }
        if ( loc2 && !loc1 ) {
            // second location is longer -> *this < loc
            return -1;
        }
        if ( !loc1 && !loc2 ) {
            // both locations ended
            return 0;
        }
        id1 = loc1->GetId();
        id2 = loc2->GetId();
        if ( int diff = loc1->x_CompareSingleId(*loc2, id1, id2) ) {
            // next sub-locs are different
            return diff;
        }
    }
    if ( iter1 && !iter2 ) {
        // *this location is longer -> *this > loc
        return 1;
    }
    if ( iter2 && !iter1 ) {
        // second locatio is longer -> *this < loc
        return -1;
    }
    // same-length locations
    return 0;
}


void CSeq_loc::PostRead(void) const
{
    InvalidateCache();
}


BEGIN_LOCAL_NAMESPACE;


// Transform a CSeq_loc into a CRange< TSeqPos >,
// that will be its total range.
static inline
CRange<TSeqPos> s_LocToRange(const CSeq_loc& loc )
{
    try {
        return loc.GetTotalRange();
    }
    catch ( CException& /*ignored*/ ) {
        // assume empty
        return CRange<TSeqPos>::GetEmpty();
    }
}


// Transform a CSeq_interval into a CRange< TSeqPos >,
// that will be its total range.
static inline
CRange<TSeqPos> s_LocToRange(const CSeq_interval& interval )
{
    return CRange<TSeqPos>(interval.GetFrom(), interval.GetTo());
};

static inline
bool s_CheckIfMatchesBioseq( const CSeq_loc &loc, 
    const CSeq_loc::ISubLocFilter *filter )
{
    if( NULL == filter ) {
        return true;
    }
    return (*filter)( loc.GetId() );
}

static inline
bool s_CheckIfMatchesBioseq( const CSeq_interval &loc, 
    const CSeq_loc::ISubLocFilter *filter )
{
    return true;
}

// Compare two ranges (reversed on minus strand)
static inline
int s_CompareRanges(const COpenRange<TSeqPos> &x_rng,
                    const COpenRange<TSeqPos> &y_rng,
                    bool minus_strand)
{
    if ( minus_strand ) {
        // This is backwards for compatibliity with C
        // Minus strand features should come in revered order

        // largest right extreme last
        if ( x_rng.GetToOpen() != y_rng.GetToOpen() ) {
            return x_rng.GetToOpen() < y_rng.GetToOpen()? -1: 1;
        }
        // longest last
        if ( x_rng.GetFrom() != y_rng.GetFrom() ) {
            return x_rng.GetFrom() > y_rng.GetFrom()? -1: 1;
        }
    }
    else {
        // smallest left extreme first
        if ( x_rng.GetFrom() != y_rng.GetFrom() ) {
            return x_rng.GetFrom() < y_rng.GetFrom()? -1: 1;
        }
        // longest first
        if ( x_rng.GetToOpen() != y_rng.GetToOpen() ) {
            return x_rng.GetToOpen() > y_rng.GetToOpen()? -1: 1;
        }
    }
    return 0;
}


// Compare two containers with intervals (CSeq_loc or CSeq_interval)
template< class Container1, class Container2 >
int s_CompareIntervals(const Container1& container1,
                       const Container2& container2,
                       bool minus_strand,
                       const CSeq_loc::ISubLocFilter *filter )
{
    typename Container1::const_iterator iter1 = container1.begin();
    typename Container1::const_iterator iter1end = container1.end();
    typename Container2::const_iterator iter2 = container2.begin();
    typename Container2::const_iterator iter2end = container2.end();
    for( ; iter1 != iter1end && iter2 != iter2end ; ++iter1, ++iter2 ) {

        // If specified, skip far location pieces
        if( NULL != filter ) {
            while( iter1 != iter1end && ! s_CheckIfMatchesBioseq(**iter1, filter) ) {
                ++iter1;
            }
            while( iter2 != iter2end && ! s_CheckIfMatchesBioseq(**iter2, filter) ) {
                ++iter2;
            }
            if( iter1 == iter1end || iter2 == iter2end ) {
                break;
            }
        }

        if ( int diff = s_CompareRanges(s_LocToRange(**iter1),
                                        s_LocToRange(**iter2),
                                        minus_strand) ) {
            return diff;
        }
    }

    // finally, shorter sequence first

    if( iter1 == iter1end ) {
        if( iter2 == iter2end ) {
            return 0;
        } else {
            return -1;
        }
    } else {
        return 1;
    }
}


END_LOCAL_NAMESPACE;


int CSeq_loc::CompareSubLoc(const CSeq_loc& loc2, ENa_strand strand, 
    const CSeq_loc::ISubLocFilter *filter) const
{
    bool minus_strand = IsReverse(strand);
    if ( IsMix() ) {
        if ( loc2.IsMix() ) {
            return s_CompareIntervals(GetMix().Get(),
                                      loc2.GetMix().Get(),
                                      minus_strand,
                                      filter );
        }
        else if ( loc2.IsPacked_int() ) {
            return s_CompareIntervals(GetMix().Get(),
                                      loc2.GetPacked_int().Get(),
                                      minus_strand,
                                      filter );
        }
        else {
            // complex loc1 is last on plus strand and first on minus strand
            return minus_strand? -1: 1;
        }
    }
    else if ( IsPacked_int() ) {
        if ( loc2.IsMix() ) {
            return -s_CompareIntervals(loc2.GetMix().Get(),
                                       GetPacked_int().Get(),
                                       minus_strand,
                                       filter );
        }
        else if ( loc2.IsPacked_int() ) {
            return s_CompareIntervals(GetPacked_int().Get(),
                                      loc2.GetPacked_int().Get(),
                                      minus_strand,
                                      filter );
        }
        else {
            // complex loc1 is last on plus strand and first on minus strand
            return minus_strand? -1: 1;
        }
    }
    else {
        if ( loc2.IsMix() || loc2.IsPacked_int() ) {
            // complex loc2 is last on plus strand and first on minus strand
            return minus_strand? 1: -1;
        }
        else {
            // two simple locations
            return 0;
        }
    }
}


bool CSeq_loc::IsSetStrand(EIsSetStrand flag) const
{
    switch ( Which() ) {
    case e_Int:
        return GetInt().IsSetStrand();
    case e_Pnt:
        return GetPnt().IsSetStrand();
    case e_Packed_int:
        return GetPacked_int().IsSetStrand(flag);
    case e_Packed_pnt:
        return GetPacked_pnt().IsSetStrand();
    case e_Mix:
        return GetMix().IsSetStrand(flag);
    case e_Bond:
        return GetBond().IsSetStrand(flag);

    case e_Equiv:
    case e_Feat:
    default:
        return false;
    }
}


ENa_strand CSeq_loc::GetStrand(void) const
{
    switch ( Which() ) {
    case e_not_set:
    case e_Null:
    case e_Empty:
        return eNa_strand_unknown;
    case e_Whole:
        return eNa_strand_both;
    case e_Int:
        return GetInt().IsSetStrand() ? GetInt().GetStrand() : eNa_strand_unknown;
    case e_Pnt:
        return GetPnt().IsSetStrand() ? GetPnt().GetStrand() : eNa_strand_unknown;
    case e_Packed_int:
        return GetPacked_int().GetStrand();
    case e_Packed_pnt:
        return GetPacked_pnt().IsSetStrand() ?
            GetPacked_pnt().GetStrand() : eNa_strand_unknown;
    case e_Mix:
        return GetMix().GetStrand();
    case e_Bond:
        return GetBond().GetStrand();

    case e_Equiv:
    case e_Feat:
    default:
        NCBI_THROW(CException, eUnknown,
            "CSeq_loc::GetStrand -- unsupported location type" +
            CSeq_loc::SelectionName(Which()));
    }
}


TSeqPos CSeq_loc::GetStart(ESeqLocExtremes ext) const
{
    switch ( Which() ) {
    case e_not_set:
    case e_Null:
    case e_Empty:
        {
            return kInvalidSeqPos;
        }
    case e_Whole:
        {
            return TRange::GetWhole().GetFrom();
        }
    case e_Int:
        {
            return GetInt().GetStart(ext);
        }
    case e_Pnt:
        {
            return GetPnt().GetPoint();
        }
    case e_Packed_int:
        {
            return GetPacked_int().GetStart(ext);
        }
    case e_Packed_pnt:
        {
            return GetPacked_pnt().GetStart(ext);
        }
    case e_Mix:
        {
            return GetMix().GetStart(ext);
        }
    case e_Bond:
        {
            return GetBond().GetStart(ext);
        }

    case e_Equiv:
    case e_Feat:
    default:
        {
            NCBI_THROW(CException, eUnknown,
                       "CSeq_loc::GetStart -- "
                       "unsupported location type");
        }
    }
}


TSeqPos CSeq_loc::GetStop(ESeqLocExtremes ext) const
{
    switch ( Which() ) {
    case e_not_set:
    case e_Null:
    case e_Empty:
        {
            return kInvalidSeqPos;
        }
    case e_Whole:
        {
            return TRange::GetWhole().GetTo();
        }
    case e_Int:
        {
            return GetInt().GetStop(ext);
        }
    case e_Pnt:
        {
            return GetPnt().GetPoint();
        }
    case e_Packed_int:
        {
            return GetPacked_int().GetStop(ext);
        }
    case e_Packed_pnt:
        {
            return GetPacked_pnt().GetStop(ext);
        }
    case e_Mix:
        {
            return GetMix().GetStop(ext);
        }
    case e_Bond:
        {
            return GetBond().GetStop(ext);
        }

    case e_Equiv:
    case e_Feat:
    default:
        {
            NCBI_THROW(CException, eUnknown,
                       "CSeq_loc::GetStart -- "
                       "unsupported location type");
        }
    }
}


TSeqPos CSeq_loc::GetCircularLength(TSeqPos seq_len) const
{
    if (seq_len == kInvalidSeqPos) {
        return GetTotalRange().GetLength();
    }
    TSeqPos start = GetStart(eExtreme_Biological);
    TSeqPos stop  = GetStop (eExtreme_Biological);
    bool    minus = IsReverseStrand();

    if (start < stop) {
        return minus ? (seq_len - stop + start + 1) : (stop - start + 1);
    } else {
        return minus ? (start - stop + 1) : (seq_len - start + stop + 1);
    }
}


CSeq_loc::const_iterator CSeq_loc::begin(void) const
{
    return CSeq_loc_CI(*this);
}


CSeq_loc::const_iterator CSeq_loc::end(void) const
{
    static CSeq_loc_CI s_Seq_loc_CI_end;
    return s_Seq_loc_CI_end;
}


// CSeq_loc_CI implementation

class CSeq_loc_CI_Impl : public CObject
{
public:
    CSeq_loc_CI_Impl(void);
    CSeq_loc_CI_Impl(const CSeq_loc& loc,
                     CSeq_loc_CI::EEmptyFlag empty_flag,
                     CSeq_loc_CI::ESeqLocOrder order);
    virtual ~CSeq_loc_CI_Impl(void) {}

    typedef SSeq_loc_CI_RangeInfo::TRange     TRange;
    typedef SSeq_loc_CI_RangeInfo::TRangeList TRanges;

    const TRanges& GetRanges(void) const { return m_Ranges; }

    bool IsEnd(size_t idx) const { return idx >= m_Ranges.size(); }

private:
    // Never copy this object
    CSeq_loc_CI_Impl(const CSeq_loc_CI_Impl&);
    CSeq_loc_CI_Impl& operator=(const CSeq_loc_CI_Impl&);

    // Process the location, fill the list
    void x_ProcessLocation(const CSeq_loc& loc);

    // Prevent seq-loc destruction
    CConstRef<CSeq_loc>      m_Location;
    // List of intervals
    TRanges                  m_Ranges;
    // Empty locations processing option
    CSeq_loc_CI::EEmptyFlag  m_EmptyFlag;
};


CSeq_loc_CI_Impl::CSeq_loc_CI_Impl(void)
{
}


CSeq_loc_CI_Impl::CSeq_loc_CI_Impl(const CSeq_loc&           loc,
                                   CSeq_loc_CI::EEmptyFlag   empty_flag,
                                   CSeq_loc_CI::ESeqLocOrder order)
    : m_Location(&loc),
      m_EmptyFlag(empty_flag)
{
    x_ProcessLocation(loc);
    if ( order == CSeq_loc_CI::eOrder_Positional  &&  loc.IsReverseStrand() ) {
        reverse(m_Ranges.begin(), m_Ranges.end());
    }
}


void CSeq_loc_CI_Impl::x_ProcessLocation(const CSeq_loc& loc)
{
    switch ( loc.Which() ) {
    case CSeq_loc::e_not_set:
    case CSeq_loc::e_Null:
    case CSeq_loc::e_Empty:
        {
            if (m_EmptyFlag == CSeq_loc_CI::eEmpty_Allow) {
                SSeq_loc_CI_RangeInfo info;
                if (loc.Which() == CSeq_loc::e_Empty) {
                    info.m_Id = &loc.GetEmpty();
                }
                else {
                    info.m_Id.Reset(new CSeq_id);
                }
                info.m_Range = TRange::GetEmpty();
                info.m_Loc = &loc;
                m_Ranges.push_back(info);
            }
            return;
        }
    case CSeq_loc::e_Whole:
        {
            SSeq_loc_CI_RangeInfo info;
            info.m_Id = &loc.GetWhole();
            info.m_Range = TRange::GetWhole();
            info.m_Loc = &loc;
            m_Ranges.push_back(info);
            return;
        }
    case CSeq_loc::e_Int:
        {
            const CSeq_interval& seq_int = loc.GetInt();
            SSeq_loc_CI_RangeInfo info;
            info.m_Id = &seq_int.GetId();
            info.m_Range.Set(seq_int.GetFrom(), seq_int.GetTo());
            if ( seq_int.IsSetStrand() ) {
                info.SetStrand(seq_int.GetStrand());
            }
            info.m_Loc = &loc;
            if (seq_int.IsSetFuzz_from()) {
                info.m_Fuzz.first = &seq_int.GetFuzz_from();
            }
            if (seq_int.IsSetFuzz_to()) {
                info.m_Fuzz.second = &seq_int.GetFuzz_to();
            }
            m_Ranges.push_back(info);
            return;
        }
    case CSeq_loc::e_Pnt:
        {
            const CSeq_point& pnt = loc.GetPnt();
            SSeq_loc_CI_RangeInfo info;
            info.m_Id = &pnt.GetId();
            info.m_Range.Set(pnt.GetPoint(), pnt.GetPoint());
            if ( pnt.IsSetStrand() ) {
                info.SetStrand(pnt.GetStrand());
            }
            info.m_Loc = &loc;
            if (pnt.IsSetFuzz()) {
                info.m_Fuzz.first = info.m_Fuzz.second = &pnt.GetFuzz();
            }
            m_Ranges.push_back(info);
            return;
        }
    case CSeq_loc::e_Packed_int:
        {
            const CPacked_seqint::Tdata& data = loc.GetPacked_int().Get();
            m_Ranges.reserve(m_Ranges.size() + data.size());
            ITERATE ( CPacked_seqint::Tdata, ii, data ) {
                const CSeq_interval& seq_int = **ii;
                SSeq_loc_CI_RangeInfo info;
                info.m_Id = &seq_int.GetId();
                info.m_Range.Set(seq_int.GetFrom(), seq_int.GetTo());
                if ( seq_int.IsSetStrand() ) {
                    info.SetStrand(seq_int.GetStrand());
                }
                info.m_Loc = &loc;
                if (seq_int.IsSetFuzz_from()) {
                    info.m_Fuzz.first = &seq_int.GetFuzz_from();
                }
                if (seq_int.IsSetFuzz_to()) {
                    info.m_Fuzz.second = &seq_int.GetFuzz_to();
                }
                m_Ranges.push_back(info);
            }
            return;
        }
    case CSeq_loc::e_Packed_pnt:
        {
            const CPacked_seqpnt& pack_pnt = loc.GetPacked_pnt();
            m_Ranges.reserve(m_Ranges.size() + pack_pnt.GetPoints().size());
            SSeq_loc_CI_RangeInfo info;
            info.m_Id = &pack_pnt.GetId();
            if ( pack_pnt.IsSetStrand() ) {
                info.SetStrand(pack_pnt.GetStrand());
            }
            if (pack_pnt.IsSetFuzz()) {
                info.m_Fuzz.first = info.m_Fuzz.second = &pack_pnt.GetFuzz();
            }
            info.m_Loc = &loc;
            ITERATE ( CPacked_seqpnt::TPoints, pi, pack_pnt.GetPoints() ) {
                info.m_Range.Set(*pi, *pi);
                m_Ranges.push_back(info);
            }
            return;
        }
    case CSeq_loc::e_Mix:
        {
            const CSeq_loc_mix::Tdata& data = loc.GetMix().Get();
            m_Ranges.reserve(m_Ranges.size() + data.size());
            ITERATE(CSeq_loc_mix::Tdata, li, data) {
                x_ProcessLocation(**li);
            }
            return;
        }
    case CSeq_loc::e_Equiv:
        {
            const CSeq_loc_equiv::Tdata& data = loc.GetEquiv().Get();
            m_Ranges.reserve(m_Ranges.size() + data.size());
            ITERATE(CSeq_loc_equiv::Tdata, li, data) {
                x_ProcessLocation(**li);
            }
            return;
        }
    case CSeq_loc::e_Bond:
        {
            const CSeq_bond& bond = loc.GetBond();
            const CSeq_point& a = bond.GetA();
            SSeq_loc_CI_RangeInfo infoA;
            infoA.m_Id = &a.GetId();
            infoA.m_Range.Set(a.GetPoint(), a.GetPoint());
            if ( a.IsSetStrand() ) {
                infoA.SetStrand(a.GetStrand());
            }
            infoA.m_Loc = &loc;
            if (a.IsSetFuzz()) {
                infoA.m_Fuzz.first = infoA.m_Fuzz.second = &a.GetFuzz();
            }
            m_Ranges.push_back(infoA);
            if ( bond.IsSetB() ) {
                const CSeq_point& b = bond.GetB();
                SSeq_loc_CI_RangeInfo infoB;
                infoB.m_Id = &b.GetId();
                infoB.m_Range.Set(b.GetPoint(), b.GetPoint());
                if ( b.IsSetStrand() ) {
                    infoB.SetStrand(b.GetStrand());
                }
                infoB.m_Loc = &loc;
                if (b.IsSetFuzz()) {
                    infoB.m_Fuzz.first = infoB.m_Fuzz.second = &b.GetFuzz();
                }
                m_Ranges.push_back(infoB);
            }
            return;
        }
    case CSeq_loc::e_Feat:
    default:
        {
            NCBI_THROW(CException, eUnknown,
                       "CSeq_loc_CI -- unsupported location type");
        }
    }
}


CSeq_loc_CI::CSeq_loc_CI(void)
    : m_Impl(new CSeq_loc_CI_Impl),
      m_Index(0)
{
}


CSeq_loc_CI::CSeq_loc_CI(const CSeq_loc& loc,
                         EEmptyFlag empty_flag,
                         ESeqLocOrder order)
    : m_Impl(new CSeq_loc_CI_Impl(loc, empty_flag, order)),
      m_Index(0)
{
}


CSeq_loc_CI::~CSeq_loc_CI()
{
}


CSeq_loc_CI::CSeq_loc_CI(const CSeq_loc_CI& iter)
    : m_Impl(iter.m_Impl),
      m_Index(iter.m_Index)
{
}


CSeq_loc_CI& CSeq_loc_CI::operator= (const CSeq_loc_CI& iter)
{
    m_Impl = iter.m_Impl;
    m_Index = iter.m_Index;
    return *this;
}


bool CSeq_loc_CI::operator== (const CSeq_loc_CI& iter) const
{
    // Check if both are at end.
    if (m_Impl->IsEnd(m_Index)  &&  iter.m_Impl->IsEnd(iter.m_Index)) {
        return true;
    }
    return m_Impl == iter.m_Impl  &&  m_Index == iter.m_Index;
}


bool CSeq_loc_CI::operator!= (const CSeq_loc_CI& iter) const
{
    return !(*this == iter);
}


const CSeq_loc& CSeq_loc_CI::GetSeq_loc(void) const
{
    return GetEmbeddingSeq_loc();
}


const CSeq_loc& CSeq_loc_CI::GetEmbeddingSeq_loc(void) const
{
    x_CheckNotValid("GetEmbeddingSeq_loc()");
    CConstRef<CSeq_loc> loc = x_GetRangeInfo().m_Loc;
    if ( !loc ) {
        NCBI_THROW(CException, eUnknown,
            "CSeq_loc_CI::GetSeq_loc() -- NULL seq-loc");
    }
    return *loc;
}


CConstRef<CSeq_loc> CSeq_loc_CI::GetRangeAsSeq_loc(void) const
{
    x_CheckNotValid("GetRangeAsSeq_loc()");
    const CSeq_loc& parent = GetEmbeddingSeq_loc();
    switch ( parent.Which() ) {
    // Single-range, empty or whole seq-loc can be used as-is.
    case CSeq_loc::e_not_set:
    case CSeq_loc::e_Null:
    case CSeq_loc::e_Empty:
    case CSeq_loc::e_Whole:
    case CSeq_loc::e_Int:
    case CSeq_loc::e_Pnt:
        return ConstRef(&parent);
    default:
        break;
    }
    // Create a new seq-loc for the current range.
    CRef<CSeq_loc> loc(new CSeq_loc);
    const SSeq_loc_CI_RangeInfo& rg_info = x_GetRangeInfo();
    if ( rg_info.m_Range.IsWhole() ) {
        // Whole location
        loc->SetWhole(const_cast<CSeq_id&>(*rg_info.m_Id));
    }
    else if ( rg_info.m_Range.Empty() ) {
        // Empty location
        loc->SetEmpty(const_cast<CSeq_id&>(*rg_info.m_Id));
    }
    else {
        loc->SetInt().SetFrom(rg_info.m_Range.GetFrom());
        loc->SetInt().SetTo(rg_info.m_Range.GetTo());
        loc->SetInt().SetId(const_cast<CSeq_id&>(*rg_info.m_Id));
        if ( rg_info.m_IsSetStrand ) {
            loc->SetInt().SetStrand(rg_info.m_Strand);
        }
        if ( rg_info.m_Fuzz.first ) {
            loc->SetInt().SetFuzz_from(
                const_cast<CInt_fuzz&>(*rg_info.m_Fuzz.first));
        }
        if ( rg_info.m_Fuzz.second ) {
            loc->SetInt().SetFuzz_to(
                const_cast<CInt_fuzz&>(*rg_info.m_Fuzz.second));
        }
    }
    return ConstRef(loc.Release());
}


bool CSeq_loc_CI::x_IsValid(void) const
{
    return m_Index < m_Impl->GetRanges().size();
}


void CSeq_loc_CI::x_ThrowNotValid(const char* where) const
{
    string msg;
    msg += "CSeq_loc_CI::";
    msg += where;
    msg += " -- iterator is not valid";
    NCBI_THROW(CException, eUnknown,
        msg);
}


const SSeq_loc_CI_RangeInfo& CSeq_loc_CI::x_GetRangeInfo(void) const
{
    // The index validity must be checked by the caller.
    return m_Impl->GetRanges()[m_Index];
}


// Append a string representation of a CSeq_id to label
inline
void s_GetLabel(const CSeq_id& id, string* label)
{
    CNcbiOstrstream os;
    id.WriteAsFasta(os);
    *label += CNcbiOstrstreamToString(os);
}


// Append to label info for a CSeq_point
inline
const CSeq_id* s_GetLabel
(const CSeq_point& pnt,
 const CSeq_id*    last_id,
 string*           label)
{
    // If CSeq_id does not match last_id, then append id to label
    if ( !last_id  ||  !last_id->Match(pnt.GetId()) ) {
        s_GetLabel(pnt.GetId(), label);
        *label += ":";
    }

    // Add strand info to label
    if (pnt.IsSetStrand()) {
        *label += GetTypeInfo_enum_ENa_strand()
            ->FindName(pnt.GetStrand(), true);
    }

    if (pnt.IsSetFuzz()) {
        // Append Fuzz info to label
        pnt.GetFuzz().GetLabel(label, pnt.GetPoint());
    } else {
        // Append 1 based point to label
        *label += NStr::IntToString(pnt.GetPoint()+1);
    }

    // update last_id
    last_id = &pnt.GetId();
    return last_id;
}


// Append to label info for CSeq_interval
inline
const CSeq_id* s_GetLabel
(const CSeq_interval& itval,
 const CSeq_id*       last_id,
 string*              label)
{
    if (!last_id || !last_id->Match(itval.GetId())) {
        s_GetLabel(itval.GetId(), label);
        *label += ":";
    }
    last_id = &itval.GetId();
    if (itval.IsSetStrand()) {
        *label += GetTypeInfo_enum_ENa_strand()
            ->FindName(itval.GetStrand(), true);
    }
    if (itval.IsSetStrand() &&
        (itval.GetStrand() == eNa_strand_minus ||
         itval.GetStrand() == eNa_strand_both_rev)) {
        if (itval.IsSetFuzz_to()) {
            itval.GetFuzz_to().GetLabel(label, itval.GetTo(), false);
        } else {
            *label += NStr::IntToString(itval.GetTo()+1);
        }
        *label += "-";
        if (itval.IsSetFuzz_from()) {
            itval.GetFuzz_from().GetLabel(label, itval.GetFrom());
        } else {
            *label += NStr::IntToString(itval.GetFrom()+1);
        }
    } else {
        if (itval.IsSetFuzz_from()) {
            itval.GetFuzz_from().GetLabel
                (label, itval.GetFrom(), false);
        } else {
            *label += NStr::IntToString(itval.GetFrom()+1);
        }
        *label += "-";
        if (itval.IsSetFuzz_to()) {
            itval.GetFuzz_to().GetLabel(label, itval.GetTo());
        } else {
            *label += NStr::IntToString(itval.GetTo()+1);
        }
    }
    return last_id;
}


// Forward declaration
const CSeq_id* s_GetLabel
(const CSeq_loc& loc,
 const CSeq_id*  last_id,
 string*         label,
 bool            first = false);


// Appends to label info for each CSeq_loc in a list of CSeq_locs
inline
const CSeq_id* s_GetLabel
(const list<CRef<CSeq_loc> >&  loc_list,
 const CSeq_id*                last_id,
 string*                       label)
{
    bool first = true;
    ITERATE (list<CRef<CSeq_loc> >, it, loc_list) {

        // Append to label for each CSeq_loc in list
        last_id = s_GetLabel(**it, last_id, label, first);
        first = false;
    }

    return last_id;
}


// Builds a label based upon a CSeq_loc and all embedded CSeq_locs
const CSeq_id* s_GetLabel
(const CSeq_loc& loc,
 const CSeq_id*  last_id,
 string*         label,
 bool            first)
{
    // Ensure label is not null
    if (!label) {
        return last_id;
    }

    // Put a comma separator if necessary
    if (!first) {
        *label += ", ";
    }

    // Loop through embedded CSeq_locs and create a label, depending on
    // type of CSeq_loc
    switch (loc.Which()) {
    case CSeq_loc::e_Null:
        *label += "~";
        break;
    case CSeq_loc::e_Empty:
        *label += "{";
        s_GetLabel(loc.GetEmpty(), label);
        last_id = &loc.GetEmpty();
        *label += "}";
        break;
    case CSeq_loc::e_Whole:
        s_GetLabel(loc.GetWhole(), label);
        last_id = &loc.GetWhole();
        break;
    case CSeq_loc::e_Int:
        last_id = s_GetLabel(loc.GetInt(), last_id, label);
        break;
    case CSeq_loc::e_Packed_int:
    {
        *label += "(";
        bool frst = true;
        ITERATE(CPacked_seqint::Tdata, it, loc.GetPacked_int().Get()) {
            if (!frst) {
                *label += ", ";
            }
            frst = false;
            last_id = s_GetLabel(**it, last_id, label);
        }
        *label += ")";
        break;
    }
    case CSeq_loc::e_Pnt:
        last_id = s_GetLabel(loc.GetPnt(), last_id, label);
        break;
    case CSeq_loc::e_Packed_pnt:
        *label += "(" + loc.GetPacked_pnt().GetId().AsFastaString() + ":";
        {{
             string str;
             ITERATE (CPacked_seqpnt::TPoints, iter,
                      loc.GetPacked_pnt().GetPoints()) {
                 if ( !str.empty() ) {
                     str += ", ";
                 }
                 str += NStr::IntToString(*iter);
             }
             *label += str;
         }}
        *label += ")";
        last_id = &loc.GetPacked_pnt().GetId();
        break;
    case CSeq_loc::e_Mix:
        *label += "[";
        last_id = s_GetLabel(loc.GetMix().Get(), last_id, label);
        *label += "]";
        break;
    case CSeq_loc::e_Equiv:
        *label += "[";
        last_id = s_GetLabel(loc.GetEquiv().Get(), last_id, label);
        *label += "]";
        break;
    case CSeq_loc::e_Bond:
        last_id = s_GetLabel(loc.GetBond().GetA(), last_id, label);
        *label += "=";
        if (loc.GetBond().IsSetB()) {
            last_id = s_GetLabel(loc.GetBond().GetB(), last_id, label);
        } else {
            *label += "?";
        }
        break;
    case CSeq_loc::e_Feat:
        *label += "(feat)";
        break;
    default:
        *label += "(?\?)";
        break;
    }
    return last_id;
}


bool CSeq_loc::IsPartialStart(ESeqLocExtremes ext) const
{
    switch (Which ()) {
        case e_Null :
            return true;

        case e_Int :
            return GetInt().IsPartialStart(ext);

        case e_Packed_int :
            return GetPacked_int().IsPartialStart(ext);

        case e_Pnt :
            return GetPnt().IsPartialStart(ext);

        case e_Packed_pnt :
            return GetPacked_pnt().IsPartialStart(ext);

        case e_Mix :
            return GetMix().IsPartialStart(ext);

        default :
            break;
    }

    return false;
}


bool CSeq_loc::IsPartialStop(ESeqLocExtremes ext) const
{
    switch (Which ()) {
        case e_Null :
            return true;

        case e_Int :
            return GetInt().IsPartialStop(ext);

        case e_Packed_int :
            return GetPacked_int().IsPartialStop(ext);

        case e_Pnt :
            return GetPnt().IsPartialStop(ext);

        case e_Packed_pnt :
            return GetPacked_pnt().IsPartialStop(ext);

        case e_Mix :
            return GetMix().IsPartialStop(ext);

        default :
            break;
    }

    return false;
}


void CSeq_loc::SetPartialStart(bool val, ESeqLocExtremes ext)
{
    if (val == IsPartialStart(ext)) {
        return;
    }

    switch (Which()) {
        case e_Int:
            SetInt().SetPartialStart(val, ext);
            break;

        case e_Packed_int :
            SetPacked_int().SetPartialStart(val, ext);
            break;

        case e_Pnt:
            SetPnt().SetPartialStart(val, ext);
            break;

        case e_Packed_pnt:
            SetPacked_pnt().SetPartialStart(val, ext);
            break;

        case e_Mix :
            SetMix().SetPartialStart(val, ext);
            break;

        default :
            break;
    }
}


void CSeq_loc::SetPartialStop(bool val, ESeqLocExtremes ext)
{
    if (val == IsPartialStop(ext)) {
        return;
    }

    switch (Which()) {
        case e_Int:
            SetInt().SetPartialStop(val, ext);
            break;

        case e_Packed_int :
            SetPacked_int().SetPartialStop(val, ext);
            break;

        case e_Pnt:
            SetPnt().SetPartialStop(val, ext);
            break;

        case e_Packed_pnt:
            SetPacked_pnt().SetPartialStop(val, ext);
            break;

        case e_Mix:
            SetMix().SetPartialStop(val, ext);
            break;

        default :
            break;
    }
}


bool CSeq_loc::IsTruncatedStart(ESeqLocExtremes ext) const
{
    switch (Which ()) {
        case e_Int :
            return GetInt().IsTruncatedStart(ext);

        case e_Packed_int :
            return GetPacked_int().IsTruncatedStart(ext);

        case e_Pnt :
            return GetPnt().IsTruncatedStart(ext);

        case e_Packed_pnt :
            return GetPacked_pnt().IsTruncatedStart(ext);

        case e_Mix :
            return GetMix().IsTruncatedStart(ext);

        default :
            break;
    }

    return false;
}


bool CSeq_loc::IsTruncatedStop(ESeqLocExtremes ext) const
{
    switch (Which ()) {
        case e_Int :
            return GetInt().IsTruncatedStop(ext);

        case e_Packed_int :
            return GetPacked_int().IsTruncatedStop(ext);

        case e_Pnt :
            return GetPnt().IsTruncatedStop(ext);

        case e_Packed_pnt :
            return GetPacked_pnt().IsTruncatedStop(ext);

        case e_Mix :
            return GetMix().IsTruncatedStop(ext);

        default :
            break;
    }

    return false;
}


void CSeq_loc::SetTruncatedStart(bool val, ESeqLocExtremes ext)
{
    if (val == IsTruncatedStart(ext)) {
        return;
    }

    switch (Which()) {
        case e_Int:
            SetInt().SetTruncatedStart(val, ext);
            break;

        case e_Packed_int :
            SetPacked_int().SetTruncatedStart(val, ext);
            break;

        case e_Pnt:
            SetPnt().SetTruncatedStart(val, ext);
            break;

        case e_Packed_pnt:
            SetPacked_pnt().SetTruncatedStart(val, ext);
            break;

        case e_Mix :
            SetMix().SetTruncatedStart(val, ext);
            break;

        default :
            break;
    }
}


void CSeq_loc::SetTruncatedStop(bool val, ESeqLocExtremes ext)
{
    if (val == IsTruncatedStop(ext)) {
        return;
    }

    switch (Which()) {
        case e_Int:
            SetInt().SetTruncatedStop(val, ext);
            break;

        case e_Packed_int :
            SetPacked_int().SetTruncatedStop(val, ext);
            break;

        case e_Pnt:
            SetPnt().SetTruncatedStop(val, ext);
            break;

        case e_Packed_pnt:
            SetPacked_pnt().SetTruncatedStop(val, ext);
            break;

        case e_Mix:
            SetMix().SetTruncatedStop(val, ext);
            break;

        default :
            break;
    }
}


// Appends a label suitable for display (e.g., error messages)
// label must point to an existing string object
// Method just returns if label is null
void CSeq_loc::GetLabel(string* label) const
{
    s_GetLabel(*this, 0, label, true);
}


// assign the 'id' field of each sub-interval to the supplied id
void CSeq_loc::SetId(CSeq_id& id)
{
    InvalidateCache();
    switch (Which()) {
    case e_Null:
        break;

    case e_Int:
        SetInt().SetId(id);
        break;

    case e_Pnt:
        SetPnt().SetId(id);
        break;

    case e_Packed_int:
        NON_CONST_ITERATE (CPacked_seqint::Tdata, iter, SetPacked_int().Set()) {
            (*iter)->SetId(id);
        }
        break;

    case e_Packed_pnt:
        SetPacked_pnt().SetId(id);
        break;

    case e_Mix:
        NON_CONST_ITERATE (CSeq_loc_mix::Tdata, iter, SetMix().Set()) {
            (*iter)->SetId(id);
        }
        break;

    case e_Whole:
        SetWhole(id);
        break;

    case e_Empty:
        SetEmpty(id);
        break;

    case e_Equiv:
        NON_CONST_ITERATE (CSeq_loc_equiv::Tdata, iter, SetEquiv().Set()) {
            (*iter)->SetId(id);
        }
        break;

    case e_Bond:
        if (GetBond().IsSetA()) {
            SetBond().SetA().SetId(id);
        }
        if (GetBond().IsSetB()) {
            SetBond().SetB().SetId(id);
        }
        break;

    case e_Feat:
        LOG_POST_X(1, Error
                      << "unhandled loc type in CSeq_loc::SetId(): e_Feat");
        break;

    default:
        LOG_POST_X(2, Error << "unhandled loc type in CSeq_loc::SetId(): "
                      << Which());
        break;
    }
}


bool CSeq_loc::x_CheckId(const CSeq_id*& id, bool may_throw) const
{
    switch ( Which() ) {
    case e_not_set:
    case e_Null:
        {
            // no Seq-id
            return true;
        }
    case e_Empty:
        {
            return x_UpdateId(id, &GetEmpty(), may_throw);
        }
    case e_Whole:
        {
            return x_UpdateId(id, &GetWhole(), may_throw);
        }
    case e_Int:
        {
            const CSeq_interval& loc = GetInt();
            return x_UpdateId(id, &loc.GetId(), may_throw);
        }
    case e_Pnt:
        {
            const CSeq_point& pnt = GetPnt();
            return x_UpdateId(id, &pnt.GetId(), may_throw);
        }
    case e_Packed_int:
        {
            // Check ID of each interval
            const CPacked_seqint& ints = GetPacked_int();
            ITERATE ( CPacked_seqint::Tdata, ii, ints.Get() ) {
                const CSeq_interval& loc = **ii;
                if ( !x_UpdateId(id, &loc.GetId(), may_throw) ) {
                    return false;
                }
            }
            return true;
        }
    case e_Packed_pnt:
        {
            const CPacked_seqpnt& pnts = GetPacked_pnt();
            return x_UpdateId(id, &pnts.GetId(), may_throw);
            break;
        }
    case e_Mix:
        {
            // Check ID of each sub-location.
            const CSeq_loc_mix& mix = GetMix();
            ITERATE( CSeq_loc_mix::Tdata, li, mix.Get() ) {
                if ( !(*li)->CheckId(id, may_throw) ) {
                    return false;
                }
            }
            return true;
        }
    case e_Bond:
        {
            const CSeq_bond& bond = GetBond();
            if ( bond.CanGetA() ) {
                if ( !x_UpdateId(id, &bond.GetA().GetId(), may_throw) ) {
                    return false;
                }
            }
            if ( bond.CanGetB() ) {
                return x_UpdateId(id, &bond.GetB().GetId(), may_throw);
            }
            return true;
        }
    case e_Equiv:
        {
            // Doesn't make much sense to test equiv, but ...
            ITERATE(CSeq_loc_equiv::Tdata, li, GetEquiv().Get()) {
                if ( !(*li)->CheckId(id, may_throw) ) {
                    return false;
                }
            }
            return true;
        }
    case e_Feat:
    default:
        {
            if (may_throw) {
                NCBI_THROW(CException, eUnknown,
                           "CSeq_loc::CheckId -- "
                           "unsupported location type");
            } else {
                return false;
            }
        }
    }
}


void CSeq_loc::ChangeToMix(void)
{
    switch ( Which() ) {
    case e_not_set:
        {
            SetMix();
            break;
        }
    case e_Mix:
        {
            break;
        }
    case e_Packed_int:
        {
            // unpack
            CRef<CSeq_loc> self(new CSeq_loc);
            self->Assign(*this, eShallow);

            CSeq_loc_mix& mix = SetMix();
            NON_CONST_ITERATE (CPacked_seqint::Tdata, it, self->SetPacked_int().Set()) {
                CRef<CSeq_loc> ival(new CSeq_loc);
                ival->SetInt(**it);
                mix.Set().push_back(ival);
            }
            break;
        }
    default:
        {
            CRef<CSeq_loc> self(new CSeq_loc);
            self->Assign(*this, eShallow);
            CSeq_loc_mix& mix = SetMix();
            mix.AddSeqLoc(*self);
        }
    }
}


void CSeq_loc::ChangeToPackedInt(void)
{
    switch ( Which() ) {
    case e_not_set:
    case e_Null:
        {
            SetPacked_int();
            return;
        }
    case e_Packed_int:
        {
            return;
        }
    case e_Int:
        {
            CConstRef<CSeq_interval> self(&GetInt());
            SetPacked_int().AddInterval(*self);
            return;
        }
    case e_Pnt:
        {
            // Make an interval of length 1
            CRef<CSeq_interval> new_int(new CSeq_interval);
            new_int->SetId().Assign(GetPnt().GetId());
            new_int->SetFrom(GetPnt().GetPoint());
            new_int->SetTo(GetPnt().GetPoint());
            if (GetPnt().IsSetStrand()) {
                new_int->SetStrand(GetPnt().GetStrand());
            }
            if (GetPnt().IsSetFuzz()) {
                const CInt_fuzz& fuzz = GetPnt().GetFuzz();
                if (!fuzz.IsLim()  ||  fuzz.GetLim() != CInt_fuzz::eLim_gt) {
                    new_int->SetFuzz_from().Assign(fuzz);
                }
                if (!fuzz.IsLim()  ||  fuzz.GetLim() != CInt_fuzz::eLim_lt) {
                    new_int->SetFuzz_to().Assign(fuzz);
                }
            }
            SetPacked_int().AddInterval(*new_int);
            return;
        }
    case e_Mix:
        {
            // Recursively convert each sub-location to packed-int, then merge.
            // Work with copies of sub-locs so that if an exception is thrown
            // the original location remains unchanged.
            vector<CRef<CSeq_loc> > sub_locs;
            sub_locs.reserve(GetMix().Get().size());
            ITERATE (CSeq_loc_mix::Tdata, orig_sub_loc, GetMix().Get()) {
                CRef<CSeq_loc> new_sub_loc(new CSeq_loc);
                new_sub_loc->Assign(**orig_sub_loc);
                new_sub_loc->ChangeToPackedInt();
                sub_locs.push_back(new_sub_loc);
            }
            SetPacked_int();  // in case there are zero intervals
            ITERATE (vector<CRef<CSeq_loc> >, sub_loc, sub_locs) {
                copy((*sub_loc)->GetPacked_int().Get().begin(),
                     (*sub_loc)->GetPacked_int().Get().end(),
                     back_inserter(SetPacked_int().Set()));
            }
            return;
        }
    default:
        {
            NCBI_THROW(CException, eUnknown,
                "Can not convert location to packed-int");
        }
    }
}


void CSeq_loc::x_ChangeToMix(const CSeq_loc& other)
{
    ChangeToMix();
    SetMix().AddSeqLoc(other);
}


void CSeq_loc::x_ChangeToPackedInt(const CSeq_interval& other)
{
    _ASSERT(IsInt());

    ChangeToPackedInt();
    SetPacked_int().AddInterval(other);
}


void CSeq_loc::x_ChangeToPackedInt(const CSeq_loc& other)
{
    _ASSERT(IsInt());
    _ASSERT(other.IsInt()  ||  other.IsPacked_int());

    ChangeToPackedInt();

    if ( other.IsInt() ) {
        SetPacked_int().AddInterval(other.GetInt());
    } else {  // other is packed int
        SetPacked_int().AddIntervals(other.GetPacked_int());
    }
}


void CSeq_loc::x_ChangeToPackedPnt(const CSeq_loc& other)
{
    _ASSERT(IsPnt());
    _ASSERT(other.IsPnt()  ||  other.IsPacked_pnt());

    CRef<CSeq_point> pnt(&SetPnt());
    CPacked_seqpnt& ppnt = SetPacked_pnt();
    if ( pnt->IsSetStrand() ) {
        ppnt.SetStrand(pnt->GetStrand());
    }
    if ( pnt->IsSetId() ) {
        ppnt.SetId(pnt->SetId());
    }
    if ( pnt->IsSetFuzz() ) {
        ppnt.SetFuzz(pnt->SetFuzz());
    }
    ppnt.AddPoint(pnt->GetPoint());

    if ( other.IsPnt() ) {
        ppnt.AddPoint(other.GetPnt().GetPoint());
    } else { // other is packed point
        ppnt.AddPoints(other.GetPacked_pnt().GetPoints());
    }
}


template<typename T1, typename T2>
bool s_CanAdd(const T1& obj1, const T2& obj2)
{
    // test strands
    {{
        ENa_strand s1 = obj1.CanGetStrand() ? obj1.GetStrand() : eNa_strand_unknown;
        ENa_strand s2 = obj2.CanGetStrand() ? obj2.GetStrand() : eNa_strand_unknown;
        if ( s1 != s2 ) {
            return false;
        }
    }}

    // test ids
    {{
        const CSeq_id* id1 = obj1.CanGetId() ? &obj1.GetId() : 0;
        const CSeq_id* id2 = obj2.CanGetId() ? &obj2.GetId() : 0;
        // both null - ok to add (this will probably never happen)
        if (id1 == 0  &&  id2 == 0) return true;
        // ids are different (one may be null)
        if (id1 == 0  ||  id2 == 0  ||  !id1->Match(*id2) ) {
            return false;
        }
    }}

    // test fuzz
    {{
        const CInt_fuzz* f1 = obj1.CanGetFuzz() ? &obj1.GetFuzz() : 0;
        const CInt_fuzz* f2 = obj2.CanGetFuzz() ? &obj2.GetFuzz() : 0;
        // both null - ok to add
        if (f1 == 0  &&  f2 == 0) return true;
        // fuzzes are different (one may be null)
        if (f1 == 0  ||  f2 == 0  ||  !f1->Equals(*f2)) {
            return false;
        }
    }}

    return true;
}


bool s_CanAdd(const CSeq_loc& loc1, const CSeq_loc& loc2)
{
    switch ( loc1.Which() ) {
    case CSeq_loc::e_Pnt:
        {
            switch ( loc2.Which() ) {
            case CSeq_loc::e_Pnt:
                return s_CanAdd(loc1.GetPnt(), loc2.GetPnt());
            case CSeq_loc::e_Packed_pnt:
                return s_CanAdd(loc1.GetPnt(), loc2.GetPacked_pnt());
            default:
                break;
            }
            break;
        }
    case CSeq_loc::e_Packed_pnt:
        {
            switch ( loc2.Which() ) {
            case CSeq_loc::e_Pnt:
                return s_CanAdd(loc1.GetPacked_pnt(), loc2.GetPnt());
            case CSeq_loc::e_Packed_pnt:
                return s_CanAdd(loc1.GetPacked_pnt(), loc2.GetPacked_pnt());
            default:
                break;
            }
            break;
        }
    default:
        {
            return false;
        }
    }

    return false;
}


void CSeq_loc::Add(const CSeq_loc& other)
{
    InvalidateCache();
    switch ( Which() ) {
    case e_not_set:
        {
            Assign(other);
            break;
        }
    case e_Null:
        {
            // ??? skip if other is null?
            x_ChangeToMix(other);
            break;
        }
    case e_Empty:
        {
            // ??? skip if other is empty and ids match?
            x_ChangeToMix(other);
            break;
        }

    case e_Whole:
        {
            x_ChangeToMix(other);
            break;
        }
    case e_Int:
        {
            if ( other.IsInt()  ||  other.IsPacked_int() ) {
                x_ChangeToPackedInt(other);
            } else {
                x_ChangeToMix(other);
            }
        }
        break;
    case e_Pnt:
        {
            if ( s_CanAdd(*this, other) ) {
                x_ChangeToPackedPnt(other);
            } else {
                x_ChangeToMix(other);
            }
            break;
        }
    case e_Packed_int:
        {
            if ( other.IsInt() ) {
                SetPacked_int().AddInterval(other.GetInt());
            } else if ( other.IsPacked_int() ) {
                SetPacked_int().AddIntervals(other.GetPacked_int());
            } else {
                x_ChangeToMix(other);
            }
            break;
        }
    case e_Packed_pnt:
        {
            if ( s_CanAdd(*this, other) ) {
                if ( other.IsPnt() ) {
                    SetPacked_pnt().AddPoint(other.GetPnt().GetPoint());
                } else if ( other.IsPacked_pnt() ) {
                    SetPacked_pnt().AddPoints(other.GetPacked_pnt().GetPoints());
                }
            } else {
                x_ChangeToMix(other);
            }
            break;
        }
    case e_Mix:
        {
            SetMix().AddSeqLoc(other);
            break;
        }
    case e_Equiv:
        {
            SetEquiv().Add(other);
            break;
        }
    case e_Bond:
        {
            x_ChangeToMix(other);
            break;
        }
    case e_Feat:
    default:
        {
            NCBI_THROW(CException, eUnknown,
                       "CSeq_loc::Add -- "
                       "unsupported location type");
        }
    }
}


void CSeq_loc::FlipStrand(void)
{
    switch ( Which() ) {
    case e_Int:
        SetInt().FlipStrand();
        break;
    case e_Pnt:
        SetPnt().FlipStrand();
        break;
    case e_Packed_int:
        SetPacked_int().FlipStrand();
        break;
    case e_Packed_pnt:
        SetPacked_pnt().FlipStrand();
        break;
    case e_Mix:
        SetMix().FlipStrand();
        break;

    default:
        break;
    }
}


// Types used in operations with seq-locs

class CRangeWithFuzz : public CSeq_loc::TRange
{
public:
    typedef CSeq_loc::TRange TParent;
    typedef CConstRef<CInt_fuzz>  TFuzz;

    CRangeWithFuzz(const TParent& rg)
        : TParent(rg), m_Strand(eNa_strand_unknown)
    {
    }
    CRangeWithFuzz(const CSeq_loc_CI& it)
        : TParent(it.GetRange()),
          m_Fuzz_from(it.GetFuzzFrom()),
          m_Fuzz_to(it.GetFuzzTo()),
          m_Strand(it.GetStrand())
    {
    }

    void ResetFuzzFrom(void) { m_Fuzz_from.Reset(); }
    void ResetFuzzTo(void) { m_Fuzz_to.Reset(); }
    TFuzz IsSetFuzzFrom(void) const { return m_Fuzz_from; }
    TFuzz IsSetFuzzTo(void) const { return m_Fuzz_to; }
    const CInt_fuzz& GetFuzzFrom(void) const { return *m_Fuzz_from; }
    const CInt_fuzz& GetFuzzTo(void) const { return *m_Fuzz_to; }

    // Add fuzzes assuming that both ranges had the same 'from'
    void AddFuzzFrom(const CRangeWithFuzz& rg)
    {
        x_AddFuzz(m_Fuzz_from, rg.m_Fuzz_from, rg.m_Strand);
    }

    // Add fuzzes assuming that both ranges had the same 'to'
    void AddFuzzTo(const CRangeWithFuzz& rg)
    {
        x_AddFuzz(m_Fuzz_to, rg.m_Fuzz_to, rg.m_Strand);
    }

    void CopyFrom(const CRangeWithFuzz& rg)
    {
        SetFrom(rg.GetFrom());
        m_Fuzz_from = rg.m_Fuzz_from;
    }

    void CopyTo(const CRangeWithFuzz& rg)
    {
        SetTo(rg.GetTo());
        m_Fuzz_to = rg.m_Fuzz_to;
    }

    CRangeWithFuzz& operator +=(const CRangeWithFuzz& rg)
    {
        TParent::position_type old_from = GetFrom();
        TParent::position_type old_to = GetTo();
        TParent::operator+=(rg);
        if (old_from != GetFrom()) {
            m_Fuzz_from.Reset(rg.m_Fuzz_from);
        }
        else if (old_from == rg.GetFrom()) {
            // Reset fuzz if it's not the same for both ranges
            AddFuzzFrom(rg);
        }
        if (old_to != GetTo()) {
            m_Fuzz_to.Reset(rg.m_Fuzz_to);
        }
        else if (old_to == rg.GetTo()) {
            AddFuzzTo(rg);
        }
        return *this;
    }

    CRangeWithFuzz& operator +=(const TParent& rg)
    {
        TParent::position_type old_from = GetFrom();
        TParent::position_type old_to = GetTo();
        TParent::operator+=(rg);
        // Reset fuzz if the corresponding extreme changes
        if (old_from != GetFrom()) {
            ResetFuzzFrom();
        }
        if (old_to != GetTo()) {
            ResetFuzzTo();
        }
        return *this;
    }

private:
    CRef<CInt_fuzz> x_SetFuzz(TFuzz&           fuzz,
                              const CInt_fuzz* copy_from)
    {
        // Since TFuzz is a const-ref, setting fuzz requires creating
        // a new object
        CRef<CInt_fuzz> new_fuzz(new CInt_fuzz);
        // The new value is optional
        if ( copy_from ) {
            new_fuzz->Assign(*copy_from);
        }
        fuzz.Reset(new_fuzz);
        return new_fuzz;
    }

    void x_AddFuzz(TFuzz&       fuzz,
                   const TFuzz& other,
                   ENa_strand   other_strand)
    {
        if ( !fuzz ) {
            // Use fuzz from the other range if available
            if ( other ) {
                x_SetFuzz(fuzz, other.GetPointerOrNull());
            }
            return;
        }
        if ( !other ) {
            // The other range has no fuzz, keep the current one
            return;
        }
        if (fuzz->Which() != other->Which()) {
            // Fuzzes have different types, reset to lim-unk.
            CRef<CInt_fuzz> new_fuzz = x_SetFuzz(fuzz, NULL);
            new_fuzz->SetLim(CInt_fuzz::eLim_unk);
            return;
        }

        const CInt_fuzz& fz = *fuzz;
        const CInt_fuzz& ofz = *other;
        // Both fuzzes are set and have the same type, try to merge them
        switch ( fz.Which() ) {
        case CInt_fuzz::e_Lim:
            {
                CInt_fuzz::ELim this_lim = fz.GetLim();
                CInt_fuzz::ELim other_lim = ofz.GetLim();
                bool this_rev = IsReverse(m_Strand);
                bool other_rev = IsReverse(other_strand);
                bool other_lt = other_lim == CInt_fuzz::eLim_lt  ||
                    (!other_rev  &&  other_lim == CInt_fuzz::eLim_tl)  ||
                    (other_rev  &&  other_lim == CInt_fuzz::eLim_tr);
                bool other_gt = other_lim == CInt_fuzz::eLim_gt  ||
                    (!other_rev  &&  other_lim == CInt_fuzz::eLim_tr)  ||
                    (other_rev  &&  other_lim == CInt_fuzz::eLim_tl);
                switch ( fz.GetLim() ) {
                case CInt_fuzz::eLim_lt:
                    if ( other_lt ) {
                        return; // the same
                    }
                    break;
                case CInt_fuzz::eLim_gt:
                    if ( other_gt ) {
                        return; // the same
                    }
                    break;
                case CInt_fuzz::eLim_tl:
                    if ((!this_rev  &&  other_lt)  ||
                        (this_rev  &&  other_gt)) {
                        return; // the same
                    }
                    break;
                case CInt_fuzz::eLim_tr:
                    if ((!this_rev  &&  other_gt)  ||
                        (this_rev  &&  other_lt)) {
                        return; // the same
                    }
                    break;
                default:
                    if (other_lim == this_lim) {
                        return;
                    }
                    break;
                }
                // Different limits - reset to lim-unk.
                CRef<CInt_fuzz> new_fuzz = x_SetFuzz(fuzz, NULL);
                new_fuzz->SetLim(CInt_fuzz::eLim_unk);
                break;
            }
        case CInt_fuzz::e_Alt:
            {
                // Use union
                CRef<CInt_fuzz> new_fuzz = x_SetFuzz(fuzz, NULL);
                new_fuzz->SetAlt().insert(
                    new_fuzz->SetAlt().end(),
                    fz.GetAlt().begin(),
                    fz.GetAlt().end());
                new_fuzz->SetAlt().insert(
                    new_fuzz->SetAlt().end(),
                    ofz.GetAlt().begin(),
                    ofz.GetAlt().end());
                break;
            }
        case CInt_fuzz::e_Range:
            {
                // Use union
                CInt_fuzz::C_Range::TMin min1 = fz.GetRange().GetMin();
                CInt_fuzz::C_Range::TMin min2 = ofz.GetRange().GetMin();
                CInt_fuzz::C_Range::TMax max1 = fz.GetRange().GetMax();
                CInt_fuzz::C_Range::TMax max2 = ofz.GetRange().GetMax();
                if (min1 > min2  ||  max1 < max2) {
                    CRef<CInt_fuzz> new_fuzz = x_SetFuzz(fuzz, NULL);
                    new_fuzz->SetRange().SetMin(min1 < min2 ? min1 : min2);
                    new_fuzz->SetRange().SetMax(max1 > max2 ? max1 : max2);
                }
                break;
            }
        case CInt_fuzz::e_P_m:
            {
                // Use max value
                CInt_fuzz::TP_m pm = ofz.GetP_m();
                if (fz.GetP_m() < pm) {
                    CRef<CInt_fuzz> new_fuzz = x_SetFuzz(fuzz, NULL);
                    new_fuzz->SetP_m(pm);
                }
                break;
            }
        case CInt_fuzz::e_Pct:
            {
                // Use max value
                CInt_fuzz::TPct pct = ofz.GetPct();
                if (fz.GetPct() < pct) {
                    CRef<CInt_fuzz> new_fuzz = x_SetFuzz(fuzz, NULL);
                    new_fuzz->SetPct(pct);
                }
                break;
            }
        default:
            // Failed to merge fuzzes
            fuzz.Reset();
            break;
        }
    }

    TFuzz m_Fuzz_from;
    TFuzz m_Fuzz_to;
    ENa_strand m_Strand;
};


typedef CRangeWithFuzz                      TRangeWithFuzz;
typedef vector<TRangeWithFuzz>              TRanges;
typedef map<CSeq_id_Handle, TRanges>        TIdToRangeMap;
typedef CRangeCollection<TSeqPos>           TRangeColl;
typedef map<CSeq_id_Handle, TRangeColl>     TIdToRangeColl;


class CRange_Less
{
public:
    CRange_Less(void) {}

    bool operator() (const TRangeWithFuzz& rg1, const TRangeWithFuzz& rg2) const
    {
        if ( rg1.IsWhole() ) {
            return !rg2.IsWhole();
        }
        if ( rg1.Empty() ) {
            return !rg2.Empty()  &&  !rg2.IsWhole();
        }
        return !rg2.IsWhole()  &&  !rg2.Empty()  &&  rg1 < rg2;
    }
};


class CRange_ReverseLess
{
public:
    CRange_ReverseLess(void) {}

    bool operator() (const TRangeWithFuzz& rg1, const TRangeWithFuzz& rg2) const
    {
        if ( rg1.IsWhole() ) {
            return !rg2.IsWhole();
        }
        if ( rg1.Empty() ) {
            return !rg2.Empty()  &&  !rg2.IsWhole();
        }
        if ( rg2.IsWhole()  ||  rg2.Empty() ) {
            return false;
        }
        if (rg1.GetTo() != rg2.GetTo()) {
            return rg1.GetTo() > rg2.GetTo();
        }
        return rg1.GetFrom() < rg2.GetFrom();
    }
};


static inline
bool x_MatchStrand(ENa_strand str1, ENa_strand str2, CSeq_loc::TOpFlags flags)
{
    return (flags & CSeq_loc::fStrand_Ignore) != 0 ||
    IsReverse(str1) == IsReverse(str2);
}


static
bool x_MergeRanges(TRangeWithFuzz& rg1, ENa_strand str1,
                   const TRangeWithFuzz& rg2, ENa_strand str2,
                   CSeq_loc::TOpFlags flags)
{
    if ( !x_MatchStrand(str1, str2, flags) ) {
        return false;
    }
    // Check contained
    if ( (flags & CSeq_loc::fMerge_Contained) != 0 ) {
        if (rg1.GetFrom() <= rg2.GetFrom()  &&  rg1.GetTo() >= rg2.GetTo()) {
            // rg2 already contained in rg1
            if (rg1.GetFrom() == rg2.GetFrom()) {
                rg1.AddFuzzFrom(rg2);
            }
            if (rg1.GetTo() == rg2.GetTo()) {
                rg1.AddFuzzTo(rg2);
            }
            return true;
        }
        if (rg1.GetFrom() >= rg2.GetFrom()  &&  rg1.GetTo() <= rg2.GetTo()) {
            // rg1 contained in rg2
            bool same_from = rg1.GetFrom() == rg2.GetFrom();
            bool same_to = rg1.GetTo() == rg2.GetTo();
            rg1 = rg2;
            if (same_from) {
                rg1.AddFuzzFrom(rg2);
            }
            if (same_to) {
                rg1.AddFuzzTo(rg2);
            }
            return true;
        }
    }
    // Check overlapping
    if ( (flags & CSeq_loc::fMerge_OverlappingOnly) != 0  &&
        rg1.IntersectingWith(rg2) ) {
        rg1 += rg2;
        return true;
    }
    // Check abutting
    if ((flags & CSeq_loc::fMerge_AbuttingOnly) != 0) {
        if ( !IsReverse(str1) ) {
            if ( rg1.GetToOpen() == rg2.GetFrom() ) {
                rg1.CopyTo(rg2);
                return true;
            }
        }
        else {
            if (rg1.GetFrom() == rg2.GetToOpen()) {
                rg1.CopyFrom(rg2);
                return true;
            }
        }
    }
    return false;
}


static
void x_PushRange(CSeq_loc& dst,
                 const CSeq_id_Handle& idh,
                 const TRangeWithFuzz& rg,
                 ENa_strand strand)
{
    if (dst.Which() != CSeq_loc::e_not_set) {
        if ( !dst.IsMix() ) {
            dst.ChangeToMix();
        }
    }
    if ( !idh ) {
        // NULL
        if (dst.IsMix()) {
            dst.SetMix().Set().push_back(Ref(new CSeq_loc(CSeq_loc::e_Null)));
        }
        else {
            dst.SetNull();
        }
        return;
    }
    CRef<CSeq_id> id(new CSeq_id());
    id->Assign(*idh.GetSeqId());
    if ( rg.IsWhole() ) {
        if (dst.IsMix()) {
            CRef<CSeq_loc> whole(new CSeq_loc);
            whole->SetWhole(*id);
            dst.SetMix().Set().push_back(whole);
        }
        else {
            dst.SetWhole(*id);
        }
    }
    else if ( rg.Empty() ) {
        if (dst.IsMix()) {
            CRef<CSeq_loc> empty(new CSeq_loc);
            empty->SetEmpty(*id);
            dst.SetMix().Set().push_back(empty);
        }
        else {
            dst.SetEmpty(*id);
        }
    }
    else if ( rg.GetLength() == 1 &&
        ( !rg.IsSetFuzzFrom() || !rg.IsSetFuzzTo() ||
        rg.GetFuzzFrom().Equals(rg.GetFuzzTo()) ) )
    {
        // Preserve points
        CRef<CSeq_point> pnt(new CSeq_point);
        pnt->SetId(*id);
        pnt->SetPoint(rg.GetFrom());
        if (strand != eNa_strand_unknown) {
            pnt->SetStrand(strand);
        }
        if ( rg.IsSetFuzzFrom() ) {
            pnt->SetFuzz().Assign(rg.GetFuzzFrom());
        } else if( rg.IsSetFuzzTo() ) {
            pnt->SetFuzz().Assign(rg.GetFuzzTo());
        }
        if (dst.IsMix()) {
            CRef<CSeq_loc> pnt_loc(new CSeq_loc);
            pnt_loc->SetPnt(*pnt);
            dst.SetMix().Set().push_back(pnt_loc);
        }
        else {
            dst.SetPnt(*pnt);
        }
    }
    else {
        if (dst.IsMix()) {
            CRef<CSeq_loc> int_loc(new CSeq_loc);
            CSeq_interval& ival = int_loc->SetInt();
            ival.SetFrom(rg.GetFrom());
            ival.SetTo(rg.GetTo());
            ival.SetId().Assign(*id);
            if (strand != eNa_strand_unknown) {
                ival.SetStrand(strand);
            }
            if ( rg.IsSetFuzzFrom() ) {
                ival.SetFuzz_from().Assign(rg.GetFuzzFrom());
            }
            if ( rg.IsSetFuzzTo() ) {
                ival.SetFuzz_to().Assign(rg.GetFuzzTo());
            }
            dst.SetMix().Set().push_back(int_loc);
        }
        else {
            CRef<CSeq_interval> interval(new CSeq_interval(*id,
                rg.GetFrom(),
                rg.GetTo(),
                strand));
            if ( rg.IsSetFuzzFrom() ) {
                interval->SetFuzz_from().Assign(rg.GetFuzzFrom());
            }
            if ( rg.IsSetFuzzTo() ) {
                interval->SetFuzz_to().Assign(rg.GetFuzzTo());
            }
            dst.SetInt(*interval);
        }
    }
}


static
void x_SingleRange(CSeq_loc& dst,
                   const CSeq_loc& src,
                   ISynonymMapper& syn_mapper)
{
    // Create a single range
    TRangeWithFuzz total_rg(TRangeWithFuzz::GetEmpty());
    CSeq_id_Handle first_id;
    ENa_strand first_strand = eNa_strand_unknown;
    for (CSeq_loc_CI it(src, CSeq_loc_CI::eEmpty_Allow); it; ++it) {
        CSeq_id_Handle next_id = syn_mapper.GetBestSynonym(it.GetSeq_id());
        if ( !next_id ) {
            // Ignore NULLs
            continue;
        }
        if ( first_id ) {
            // Seq-id may be missing for NULL seq-loc
            if (next_id  &&  first_id != next_id) {
                NCBI_THROW(CException, eUnknown,
                    "Can not merge multi-id seq-loc");
            }
        }
        else {
            first_id = next_id;
            first_strand = it.GetStrand();
        }
        total_rg += TRangeWithFuzz(it);
    }
    if ( first_id ) {
        CRef<CSeq_id> id(new CSeq_id);
        id->Assign(*first_id.GetSeqId());
        CRef<CSeq_interval> interval(new CSeq_interval(*id,
                                                       total_rg.GetFrom(),
                                                       total_rg.GetTo(),
                                                       first_strand));
        if ( total_rg.IsSetFuzzFrom() ) {
            interval->SetFuzz_from().Assign(total_rg.GetFuzzFrom());
        }
        if ( total_rg.IsSetFuzzTo() ) {
            interval->SetFuzz_to().Assign(total_rg.GetFuzzTo());
        }
        dst.SetInt(*interval);
    }
    else {
        // Null seq-loc
        dst.SetNull();
    }
}


static
void x_RangesToSeq_loc(CSeq_loc& dst,
                       TIdToRangeMap& id_map,
                       ENa_strand default_strand,
                       CSeq_loc::TOpFlags flags)
{
    // Iterate ids for each strand
    NON_CONST_ITERATE(TIdToRangeMap, id_it, id_map) {
        if ( !id_it->first ) {
            // All NULLs merged
            x_PushRange(dst,
                        id_it->first,
                        TRangeWithFuzz(TRangeWithFuzz::GetEmpty()),
                        eNa_strand_unknown);
            continue;
        }
        CRef<CSeq_id> id(new CSeq_id);
        id->Assign(*id_it->first.GetSeqId());
        TRanges& ranges = id_it->second;
        if ( (flags & CSeq_loc::fSort) != 0 ) {
            if ( !IsReverse(default_strand) ) {
                sort(ranges.begin(), ranges.end(), CRange_Less());
            }
            else {
                sort(ranges.begin(), ranges.end(), CRange_ReverseLess());
            }
        }
        // Merge ranges according to the flags, add to destination
        TRangeWithFuzz last_rg(TRangeWithFuzz::GetEmpty());
        bool have_range = false;
        ITERATE(TRanges, rg, ranges) {
            if (x_MergeRanges(last_rg, default_strand,
                            *rg, default_strand,
                            flags)) {
                have_range = true;
                continue;
            }
            // No merging - push current range, reset last values
            if (have_range) {
                x_PushRange(dst, id_it->first, last_rg, default_strand);
            }
            last_rg = *rg;
            have_range = true;
        }
        if (have_range) {
            x_PushRange(dst, id_it->first, last_rg, default_strand);
        }
    }
}


static
void x_MergeNoSort(CSeq_loc& dst,
                   const CSeq_loc& src,
                   CSeq_loc::TOpFlags flags,
                   ISynonymMapper& syn_mapper)
{
    _ASSERT((flags & CSeq_loc::fSort) == 0);
    CSeq_id_Handle last_id;
    TRangeWithFuzz last_rg(TRangeWithFuzz::GetEmpty());
    ENa_strand last_strand = eNa_strand_unknown;
    bool have_range = false;
    for (CSeq_loc_CI it(src, CSeq_loc_CI::eEmpty_Allow); it; ++it) {
        CSeq_id_Handle idh = syn_mapper.GetBestSynonym(it.GetSeq_id());
        // ID and strand must match
        TRangeWithFuzz it_rg(it);
        if ( have_range  &&  last_id == idh ) {
            if (x_MergeRanges(last_rg,
                              last_strand,
                              it_rg,
                              it.GetStrand(),
                              flags)) {
                have_range = true;
                continue;
            }
        }
        // No merging - push current range, reset last values
        if ( have_range ) {
            x_PushRange(dst, last_id, last_rg, last_strand);
        }
        last_id = idh;
        last_rg = it_rg;
        last_strand = it.GetStrand();
        have_range = true;
    }
    if ( have_range ) {
        x_PushRange(dst, last_id, last_rg, last_strand);
    }
    if (dst.Which() == CSeq_loc::e_not_set) {
        dst.SetNull();
    }
}


static
void x_MergeAndSort(CSeq_loc& dst,
                    const CSeq_loc& src,
                    CSeq_loc::TOpFlags flags,
                    ISynonymMapper& syn_mapper)
{
    bool use_strand = (flags & CSeq_loc::fStrand_Ignore) == 0;
    // Id -> range map for both strands
    auto_ptr<TIdToRangeMap> pid_map_minus(use_strand ?
        new TIdToRangeMap : 0);
    TIdToRangeMap id_map_plus;
    TIdToRangeMap& id_map_minus = use_strand ?
        *pid_map_minus.get() : id_map_plus;

    // Prepare default strands
    ENa_strand default_plus = use_strand ?
        eNa_strand_plus : eNa_strand_unknown;
    ENa_strand default_minus = use_strand ?
        eNa_strand_minus : eNa_strand_unknown;

    // Split location by by id/strand/range
    for (CSeq_loc_CI it(src, CSeq_loc_CI::eEmpty_Allow); it; ++it) {
        CSeq_id_Handle idh = syn_mapper.GetBestSynonym(it.GetSeq_id());
        if ( IsReverse(it.GetStrand()) ) {
            id_map_minus[idh].push_back(TRangeWithFuzz(it));
        }
        else {
            id_map_plus[idh].push_back(TRangeWithFuzz(it));
        }
    }

    x_RangesToSeq_loc(dst, id_map_plus, default_plus, flags);
    if ( use_strand ) {
        x_RangesToSeq_loc(dst, id_map_minus, default_minus, flags);
    }
    if (dst.Which() == CSeq_loc::e_not_set) {
        dst.SetNull();
    }
}


static
void x_SingleRange(CSeq_loc& dst,
                   const CSeq_loc& src,
                   TIdToRangeColl& rg_coll_plus,
                   TIdToRangeColl& rg_coll_minus,
                   ISynonymMapper& syn_mapper,
                   ILengthGetter& len_getter)
{
    TRangeWithFuzz total_rg(TRangeWithFuzz::GetEmpty());
    CSeq_id_Handle first_id;
    ENa_strand first_strand = eNa_strand_unknown;
    for (CSeq_loc_CI it(src, CSeq_loc_CI::eEmpty_Allow); it; ++it) {
        CSeq_id_Handle next_id = syn_mapper.GetBestSynonym(it.GetSeq_id());
        if ( !next_id ) {
            // Ignore NULLs
            continue;
        }
        if ( first_id ) {
            // Seq-id may be missing for NULL seq-loc
            if (next_id  &&  first_id != next_id) {
                NCBI_THROW(CException, eUnknown,
                    "Can not merge multi-id seq-loc");
            }
        }
        else {
            first_id = next_id;
            first_strand = it.GetStrand();
        }
        TRangeWithFuzz it_range = TRangeWithFuzz(it);
        if (it_range.GetFrom() >= total_rg.GetFrom()  &&
            it_range.GetTo() <= total_rg.GetTo()) {
            // Nothing new can be added from this interval
            continue;
        }
        if ( it_range.IsWhole() ) {
            it_range.SetOpen(0, len_getter.GetLength(it.GetSeq_id()));
            it_range.ResetFuzzFrom();
            it_range.ResetFuzzTo();
        }
        TRangeColl it_rg_coll(it_range);
        TIdToRangeColl& rg_coll = IsReverse(it.GetStrand()) ?
            rg_coll_minus : rg_coll_plus;
        TIdToRangeColl::const_iterator id_it = rg_coll.find(next_id);
        if (id_it != rg_coll.end()) {
            it_rg_coll -= id_it->second;
        }
        TRangeWithFuzz curr_rg(it_rg_coll.GetLimits());
        if (curr_rg.GetFrom() == it_range.GetFrom()) {
            curr_rg.AddFuzzFrom(it_range);
        }
        else if (curr_rg.GetTo() == it_range.GetTo()) {
            curr_rg.AddFuzzTo(it_range);
        }
        total_rg += curr_rg;
    }

    if ( first_id ) {
        CRef<CSeq_id> id(new CSeq_id);
        id->Assign(*first_id.GetSeqId());
        CRef<CSeq_interval> interval(new CSeq_interval(*id,
                                                       total_rg.GetFrom(),
                                                       total_rg.GetTo(),
                                                       first_strand));
        if ( total_rg.IsSetFuzzFrom() ) {
            CRef<CInt_fuzz> fuzz(new CInt_fuzz);
            fuzz->Assign(total_rg.GetFuzzFrom());
            interval->SetFuzz_from(*fuzz);
        }
        if ( total_rg.IsSetFuzzTo() ) {
            CRef<CInt_fuzz> fuzz(new CInt_fuzz);
            fuzz->Assign(total_rg.GetFuzzTo());
            interval->SetFuzz_to(*fuzz);
        }
        dst.SetInt(*interval);
    }
    else {
        // Null seq-loc
        dst.SetNull();
    }
}


static
void x_SubNoSort(CSeq_loc& dst,
                 const CSeq_loc& src,
                 TIdToRangeColl& rg_coll_plus,
                 TIdToRangeColl& rg_coll_minus,
                 ISynonymMapper& syn_mapper,
                 ILengthGetter& len_getter,
                 CSeq_loc::TOpFlags flags)
{
    _ASSERT((flags & CSeq_loc::fSort) == 0);
    CSeq_id_Handle last_id;
    TRangeWithFuzz last_rg(TRangeWithFuzz::GetEmpty());
    ENa_strand last_strand = eNa_strand_unknown;
    bool have_range = false;
    for (CSeq_loc_CI it(src, CSeq_loc_CI::eEmpty_Allow); it; ++it) {
        CSeq_id_Handle idh = syn_mapper.GetBestSynonym(it.GetSeq_id());
        TRangeWithFuzz it_range = TRangeWithFuzz(it);
        if ( it_range.IsWhole() ) {
            it_range.SetOpen(0, len_getter.GetLength(it.GetSeq_id()));
        }
        TRangeColl it_rg_coll(it_range);
        TIdToRangeColl& rg_coll = IsReverse(it.GetStrand()) ?
            rg_coll_minus : rg_coll_plus;
        TIdToRangeColl::const_iterator id_it = rg_coll.find(idh);
        bool modified = false;
        if (id_it != rg_coll.end()) {
            // Check if there's anything to subtract
            ITERATE(TRangeColl, check_it, id_it->second) {
                if ( it_rg_coll.IntersectingWith(*check_it) ) {
                    it_rg_coll -= id_it->second;
                    modified = true;
                    break;
                }
            }
        }
        if ( modified ) {
            ITERATE(TRangeColl, rg_it, it_rg_coll) {
                TRangeWithFuzz curr_rg(*rg_it);
                if (curr_rg.GetFrom() == it_range.GetFrom()) {
                    curr_rg.AddFuzzFrom(it_range);
                }
                else if (curr_rg.GetTo() == it_range.GetTo()) {
                    curr_rg.AddFuzzTo(it_range);
                }
                if ( have_range  &&  last_id == idh ) {
                    if (x_MergeRanges(last_rg,
                                    last_strand,
                                    curr_rg,
                                    it.GetStrand(),
                                    flags)) {
                        have_range = true;
                        continue;
                    }
                }
                // No merging - push current range, reset last values
                if ( have_range ) {
                    x_PushRange(dst, last_id, last_rg, last_strand);
                }
                last_id = idh;
                last_rg = curr_rg;
                last_strand = it.GetStrand();
                have_range = true;
            }
        }
        else {
            if ( have_range ) {
                bool merged = false;
                if (last_id == idh) {
                    merged = x_MergeRanges(last_rg,
                        last_strand,
                        it_range,
                        it.GetStrand(),
                        flags);
                }
                if ( !merged ) {
                    x_PushRange(dst, last_id, last_rg, last_strand);
                }
            }
            last_id = idh;
            last_rg = it_range;
            last_strand = it.GetStrand();
            have_range = true;
        }
    }
    if ( have_range ) {
        x_PushRange(dst, last_id, last_rg, last_strand);
    }
    if (dst.Which() == CSeq_loc::e_not_set) {
        dst.SetNull();
    }
}


static
void x_SubAndSort(CSeq_loc& dst,
                  const CSeq_loc& src,
                  TIdToRangeColl& rg_coll_plus,
                  TIdToRangeColl& rg_coll_minus,
                  ISynonymMapper& syn_mapper,
                  ILengthGetter& len_getter,
                  CSeq_loc::TOpFlags flags)
{
    bool use_strand = (flags & CSeq_loc::fStrand_Ignore) == 0;

    // Id -> range map for both strands
    auto_ptr<TIdToRangeMap> p_id_map_minus(use_strand ?
        new TIdToRangeMap : 0);
    TIdToRangeMap id_map_plus;
    TIdToRangeMap& id_map_minus = use_strand ?
        *p_id_map_minus.get() : id_map_plus;

    // Prepare default strands
    ENa_strand default_plus = use_strand ?
        eNa_strand_plus : eNa_strand_unknown;
    ENa_strand default_minus = use_strand ?
        eNa_strand_minus : eNa_strand_unknown;

    for (CSeq_loc_CI it(src, CSeq_loc_CI::eEmpty_Allow); it; ++it) {
        CSeq_id_Handle idh = syn_mapper.GetBestSynonym(it.GetSeq_id());
        TRangeWithFuzz it_range = TRangeWithFuzz(it);
        if ( it_range.IsWhole() ) {
            it_range.SetOpen(0, len_getter.GetLength(it.GetSeq_id()));
            it_range.ResetFuzzFrom();
            it_range.ResetFuzzTo();
        }
        TRangeColl it_rg_coll(it_range);
        TRanges& rg_map = IsReverse(it.GetStrand()) ?
            id_map_minus[idh] : id_map_plus[idh];
        TIdToRangeColl& rg_coll = IsReverse(it.GetStrand()) ?
            rg_coll_minus : rg_coll_plus;
        TIdToRangeColl::const_iterator id_it = rg_coll.find(idh);
        bool modified = false;
        if (id_it != rg_coll.end()) {
            // Check if there's anything to subtract
            ITERATE(TRangeColl, check_it, id_it->second) {
                if ( it_rg_coll.IntersectingWith(*check_it) ) {
                    it_rg_coll -= id_it->second;
                    modified = true;
                    break;
                }
            }
        }
        if ( modified ) {
            ITERATE(TRangeColl, rg_it, it_rg_coll) {
                TRangeWithFuzz curr_rg(*rg_it);
                if (curr_rg.GetFrom() == it_range.GetFrom()) {
                    curr_rg.AddFuzzFrom(it_range);
                }
                else if (curr_rg.GetTo() == it_range.GetTo()) {
                    curr_rg.AddFuzzTo(it_range);
                }
                rg_map.push_back(curr_rg);
            }
        }
        else {
            rg_map.push_back(it_range);
        }
    }

    x_RangesToSeq_loc(dst, id_map_plus, default_plus, flags);
    if ( use_strand ) {
        x_RangesToSeq_loc(dst, id_map_minus, default_minus, flags);
    }
    if (dst.Which() == CSeq_loc::e_not_set) {
        dst.SetNull();
    }
}


class CDummySynonymMapper : public ISynonymMapper
{
public:
    CDummySynonymMapper(void) {}
    virtual ~CDummySynonymMapper(void) {}

    virtual CSeq_id_Handle GetBestSynonym(const CSeq_id& id)
        {
            return CSeq_id_Handle::GetHandle(id);
        }
};


class CDummyLengthGetter : public ILengthGetter
{
public:
    CDummyLengthGetter(void) {}
    virtual ~CDummyLengthGetter(void) {}

    virtual TSeqPos GetLength(const CSeq_id& id)
        {
            return CSeq_loc::TRange::GetWholeToOpen();
        }
};


CRef<CSeq_loc> CSeq_loc::Merge(TOpFlags        flags,
                               ISynonymMapper* syn_mapper) const
{
    auto_ptr<CDummySynonymMapper> p_mapper;
    if ( !syn_mapper ) {
        p_mapper.reset(new CDummySynonymMapper);
        syn_mapper = p_mapper.get();
    }

    CRef<CSeq_loc> ret(new CSeq_loc);
    if ( (flags & CSeq_loc::fMerge_SingleRange) != 0 ) {
        x_SingleRange(*ret, *this, *syn_mapper);
    }
    else if ( (flags & CSeq_loc::fSort) == 0 ) {
        x_MergeNoSort(*ret, *this, flags, *syn_mapper);
    }
    else {
        x_MergeAndSort(*ret, *this, flags, *syn_mapper);
    }
    return ret;
}


CRef<CSeq_loc> CSeq_loc::Add(const CSeq_loc& other,
                             TOpFlags        flags,
                             ISynonymMapper* syn_mapper) const
{
    auto_ptr<CDummySynonymMapper> p_mapper;
    if ( !syn_mapper ) {
        p_mapper.reset(new CDummySynonymMapper);
        syn_mapper = p_mapper.get();
    }

    CRef<CSeq_loc> ret(new CSeq_loc);
    CSeq_loc tmp;
    tmp.SetMix().AddSeqLoc(const_cast<CSeq_loc&>(*this));
    tmp.SetMix().AddSeqLoc(const_cast<CSeq_loc&>(other));
    if ( (flags & CSeq_loc::fMerge_SingleRange) != 0 ) {
        x_SingleRange(*ret, tmp, *syn_mapper);
    }
    else if ( (flags & CSeq_loc::fSort) == 0 ) {
        x_MergeNoSort(*ret, tmp, flags, *syn_mapper);
    }
    else {
        x_MergeAndSort(*ret, tmp, flags, *syn_mapper);
    }
    return ret;
}


CRef<CSeq_loc> CSeq_loc::Subtract(const CSeq_loc& other,
                                  TOpFlags        flags,
                                  ISynonymMapper* syn_mapper,
                                  ILengthGetter*  len_getter) const
{
    auto_ptr<CDummySynonymMapper> p_mapper;
    if ( !syn_mapper ) {
        p_mapper.reset(new CDummySynonymMapper);
        syn_mapper = p_mapper.get();
    }
    auto_ptr<CDummyLengthGetter> p_getter;
    if ( !len_getter ) {
        p_getter.reset(new CDummyLengthGetter);
        len_getter = p_getter.get();
    }

    CRef<CSeq_loc> ret(new CSeq_loc);

    bool use_strand = (flags & CSeq_loc::fStrand_Ignore) == 0;

    // Range collection for each strand
    auto_ptr<TIdToRangeColl> p_rg_coll_minus(use_strand ?
        new TIdToRangeColl : 0);
    TIdToRangeColl rg_coll_plus;
    TIdToRangeColl& rg_coll_minus = use_strand ?
        *p_rg_coll_minus.get() : rg_coll_plus;

    // Create range collection(s) for loc2
    for (CSeq_loc_CI it(other); it; ++it) {
        if ( it.IsEmpty() ) {
            continue;
        }
        CSeq_id_Handle idh = syn_mapper->GetBestSynonym(it.GetSeq_id());
        TRangeColl& rmap = IsReverse(it.GetStrand()) ?
            rg_coll_minus[idh] : rg_coll_plus[idh];
        rmap += TRangeWithFuzz(it);
    }

    if ( (flags & CSeq_loc::fMerge_SingleRange) != 0 ) {
        x_SingleRange(*ret,
                      *this,
                      rg_coll_plus,
                      rg_coll_minus,
                      *syn_mapper,
                      *len_getter);
    }
    else if ( (flags & CSeq_loc::fSort) == 0 ) {
        x_SubNoSort(*ret,
                    *this,
                    rg_coll_plus,
                    rg_coll_minus,
                    *syn_mapper,
                    *len_getter,
                    flags);
    }
    else {
        x_SubAndSort(*ret,
                     *this,
                     rg_coll_plus,
                     rg_coll_minus,
                     *syn_mapper,
                     *len_getter,
                     flags);
    }

    return ret;
}


CRef<CSeq_loc> CSeq_loc::Intersect(const CSeq_loc& other,
                                   TOpFlags        flags,
                                   ISynonymMapper* syn_mapper) const
{
    auto_ptr<CDummyLengthGetter> len_getter(new CDummyLengthGetter);
    CRef<CSeq_loc> tmp = Subtract(other,
        // This flag should be used only in the second subtraction
        flags & ~fMerge_SingleRange,
        syn_mapper, len_getter.get());
    return Subtract(*tmp, flags, syn_mapper, len_getter.get());
}


void CSeq_loc::SetStrand(ENa_strand strand)
{
    switch ( Which() ) {
    case e_Int:
        SetInt().SetStrand(strand);
        break;
    case e_Pnt:
        SetPnt().SetStrand(strand);
        break;
    case e_Packed_int:
        SetPacked_int().SetStrand(strand);
        break;
    case e_Packed_pnt:
        SetPacked_pnt().SetStrand(strand);
        break;
    case e_Mix:
        SetMix().SetStrand(strand);
        break;

    default:
        break;
    }
}


void CSeq_loc::ResetStrand(void)
{
    switch ( Which() ) {
    case e_Int:
        SetInt().ResetStrand();
        break;
    case e_Pnt:
        SetPnt().ResetStrand();
        break;
    case e_Packed_int:
        SetPacked_int().ResetStrand();
        break;
    case e_Packed_pnt:
        SetPacked_pnt().ResetStrand();
        break;
    case e_Mix:
        SetMix().ResetStrand();
        break;

    default:
        break;
    }
}


END_objects_SCOPE // namespace ncbi::objects::
END_NCBI_SCOPE

#undef NCBI_USE_ERRCODE_X
