/*
	MIT License http://www.opensource.org/licenses/mit-license.php
	Author Tobias Koppers @sokra
*/
var should = require("should");
var path = require("path");
var resolve = require("../");

var fixtures = path.join(__dirname, "fixtures");

function testResolve(name, context, moduleName, result) {
	describe(name, function() {
		it("should resolve sync correctly", function() {
			var filename = resolve.sync(context, moduleName);
			should.exist(filename);
			filename.should.equal(result);
		});
		it("should resolve async correctly", function(done) {
			resolve(context, moduleName, function(err, filename) {
				if(err) return done(err);
				should.exist(filename);
				filename.should.equal(result);
				done();
			});
		});
	});
}

function testResolveLoader(name, context, moduleName, result) {
	describe(name, function() {
		it("should resolve sync correctly", function() {
			var filename = resolve.loader.sync(context, moduleName);
			should.exist(filename);
			filename.should.equal(result);
		});
		it("should resolve async correctly", function(done) {
			resolve.loader(context, moduleName, function(err, filename) {
				if(err) return done(err);
				should.exist(filename);
				filename.should.equal(result);
				done();
			});
		});
	});
}

function testResolveContext(name, context, moduleName, result) {
	describe(name, function() {
		it("should resolve async correctly", function(done) {
			resolve.context(context, moduleName, function(err, filename) {
				if(err) done(err);
				should.exist(filename);
				filename.should.equal(result);
				done();
			});
		});
		it("should resolve sync correctly", function() {
			var filename = resolve.context.sync(context, moduleName);
			should.exist(filename);
			filename.should.equal(result);
		});
	});
}
describe("resolve", function() {
	testResolve("absolute path",
		fixtures, path.join(fixtures, "main1.js"), path.join(fixtures, "main1.js"));

	testResolve("file with .js",
		fixtures, "./main1.js", path.join(fixtures, "main1.js"));
	testResolve("file without extension",
		fixtures, "./main1", path.join(fixtures, "main1.js"));
	testResolve("another file with .js",
		fixtures, "./a.js", path.join(fixtures, "a.js"));
	testResolve("another file without extension",
		fixtures, "./a", path.join(fixtures, "a.js"));
	testResolve("file in module with .js",
		fixtures, "m1/a.js", path.join(fixtures, "node_modules", "m1", "a.js"));
	testResolve("file in module without extension",
		fixtures, "m1/a", path.join(fixtures, "node_modules", "m1", "a.js"));
	testResolve("another file in module without extension",
		fixtures, "complexm/step1", path.join(fixtures, "node_modules", "complexm", "step1.js"));
	testResolve("from submodule to file in sibling module",
		path.join(fixtures, "node_modules", "complexm"), "m2/b.js", path.join(fixtures, "node_modules", "m2", "b.js"));
	testResolve("from submodule to file in sibling of parent module",
		path.join(fixtures, "node_modules", "complexm", "web_modules", "m1"), "m2/b.js", path.join(fixtures, "node_modules", "m2", "b.js"));
	testResolve("from nested directory to overwritten file in module",
		path.join(fixtures, "multiple_modules"), "m1/a.js", path.join(fixtures, "multiple_modules", "node_modules", "m1", "a.js"));
	testResolve("from nested directory to not overwritten file in module",
		path.join(fixtures, "multiple_modules"), "m1/b.js", path.join(fixtures, "node_modules", "m1", "b.js"));

	testResolve("file with query",
		fixtures, "./main1.js?query", path.join(fixtures, "main1.js") + "?query");
	testResolve("file in module with query",
		fixtures, "m1/a?query", path.join(fixtures, "node_modules", "m1", "a.js") + "?query");

	testResolveContext("context for fixtures",
		fixtures, "./", fixtures);
	testResolveContext("context for fixtures/lib",
		fixtures, "./lib", path.join(fixtures, "lib"));
	testResolveContext("context for fixtures with ..",
		fixtures, "./lib/../../fixtures/./lib/..", fixtures);

	testResolveContext("context for fixtures with query",
		fixtures, "./?query", fixtures + "?query");

	testResolve("differ between directory and file, resolve file",
		fixtures, "./dirOrFile", path.join(fixtures, "dirOrFile.js"));
	testResolve("differ between directory and file, resolve directory",
		fixtures, "./dirOrFile/", path.join(fixtures, "dirOrFile", "index.js"));

	testResolveLoader("loader with template without extension",
		fixtures, "m2/b", path.join(fixtures, "node_modules", "m2-loader", "b.js"));
	testResolveLoader("loader with template as file",
		fixtures, "l", path.join(fixtures, "node_modules", "l-loader.js"));

	testResolve("find node_modules outside of node_modules",
		path.join(fixtures, "browser-module", "node_modules"), "m1/a", path.join(fixtures, "node_modules", "m1", "a.js"));
});
