///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/viewport/Viewport.h>
#include <core/viewport/ViewportManager.h>
#include <core/scene/animation/AnimManager.h>
#include <core/gui/properties/BooleanPropertyUI.h>
#include <core/gui/properties/SubObjectParameterUI.h>
#include <core/gui/mainwnd/MainFrame.h>

#include "ClusterAtomsModifier.h"

#include <atomviz/utils/NearestNeighborList.h>
#include <atomviz/utils/OnTheFlyNeighborList.h>

namespace AtomViz {

IMPLEMENT_SERIALIZABLE_PLUGIN_CLASS(ClusterAtomsModifier, AtomsObjectAnalyzerBase)
DEFINE_REFERENCE_FIELD(ClusterAtomsModifier, DataChannel, "AtomClustersChannel", _atomClustersChannel)
DEFINE_PROPERTY_FIELD(ClusterAtomsModifier, "OnlySelectedAtoms", _onlySelectedAtoms)
SET_PROPERTY_FIELD_LABEL(ClusterAtomsModifier, _onlySelectedAtoms, "Use only selected atoms")

/******************************************************************************
* Constructs the modifier object.
******************************************************************************/
ClusterAtomsModifier::ClusterAtomsModifier(bool isLoading)
	: AtomsObjectAnalyzerBase(isLoading), _onlySelectedAtoms(false)
{
	INIT_PROPERTY_FIELD(ClusterAtomsModifier, _atomClustersChannel);
	INIT_PROPERTY_FIELD(ClusterAtomsModifier, _onlySelectedAtoms);
	if(!isLoading) {
		// Create data channel for the computation results.
		_atomClustersChannel = new DataChannel(DataChannel::ClusterChannel);
	}
}

/******************************************************************************
* Applies the previously calculated analysis results to the atoms object.
******************************************************************************/
EvaluationStatus ClusterAtomsModifier::applyResult(TimeTicks time, TimeInterval& validityInterval)
{
	// Check if it is still valid.
	if(input()->atomsCount() != atomClusters()->size())
		throw Exception(tr("Number of atoms of input object has changed. Analysis results became invalid."));

	// Create a copy of the cluster channel that stores the calculated values.
	CloneHelper cloneHelper;
	DataChannel::SmartPtr channelClone = cloneHelper.cloneObject(atomClusters(), true);

	// Put the new channel into the output object.
	output()->insertDataChannel(channelClone);

	return EvaluationStatus(EvaluationStatus::EVALUATION_SUCCESS, QString(), tr("Number of clusters: %1\n").arg(numClusters));
}

/******************************************************************************
* This is the actual analysis method.
******************************************************************************/
EvaluationStatus ClusterAtomsModifier::doAnalysis(TimeTicks time, bool suppressDialogs)
{
	// Perform analysis.
	if(calculate(input(), suppressDialogs))
		return EvaluationStatus();
	else
		return EvaluationStatus(EvaluationStatus::EVALUATION_ERROR, tr("Calculation has been canceled by the user."));
}

/******************************************************************************
* Performs the analysis.
* Throws an exception on error.
* Returns false when the operation has been canceled by the user.
******************************************************************************/
bool ClusterAtomsModifier::calculate(AtomsObject* atomsObject, bool suppressDialogs)
{
	ProgressIndicator progress(tr("Finding atom clusters"), atomsObject->atomsCount(), suppressDialogs);

	// Prepare the neighbor list.
	OnTheFlyNeighborList neighborList(nearestNeighborList()->nearestNeighborCutoff());
	if(!neighborList.prepare(atomsObject, suppressDialogs)) {
		_atomClustersChannel->setSize(0);
		return false;
	}

	// Reset everything
	numClusters = 0;

	// Prepare the output channel.
	_atomClustersChannel->setSize(atomsObject->atomsCount());
	fill(_atomClustersChannel->dataInt(), _atomClustersChannel->dataInt() + atomsObject->atomsCount(), 0);

	// This bit array stores for each atom whether it has already been
	// assigned to a cluster or not.
	QBitArray flagArray(atomsObject->atomsCount());
	int numAtomsProcessed = 0;

	// If the algorithm should only operate on selected atoms then
	// we mark unselected atoms as already visited.
	if(onlySelectedAtoms()) {
		DataChannel* selectionChannel = inputStandardChannel(DataChannel::SelectionChannel);
		if(selectionChannel) {
			const int* s = selectionChannel->constDataInt();
			for(int atomIndex = 0; atomIndex < atomsObject->atomsCount(); atomIndex++)
				if(*s++ == false) {
					flagArray.setBit(atomIndex);
					numAtomsProcessed++;
				}
		}
		else {
			flagArray.fill(true);
			numAtomsProcessed = atomsObject->atomsCount();
		}
	}

	// Iterate over each atom.
	for(int atomIndex = 0; atomIndex < atomsObject->atomsCount(); atomIndex++) {
		// Skip atoms that have already been assigned to a cluster.
		if(flagArray.at(atomIndex)) continue;

		// Since this atom is not part of a cluster we can create a new cluster for it.
		int cluster = numClusters;
		numClusters++;

		flagArray.setBit(atomIndex);
		_atomClustersChannel->setInt(atomIndex, cluster);

		// Now recursively iterate over all neighbors of the initial atom and put them
		// into the cluster too.
		QStack<int> toProcess;
		toProcess.push(atomIndex);

		do {
			numAtomsProcessed++;

			// Update progress bar.
			if((numAtomsProcessed % 4096) == 0) {
				progress.setValue(numAtomsProcessed);
				if(progress.isCanceled()) {
					// Throw away results obtained so far if the user cancels the calculation.
					_atomClustersChannel->setSize(0);
					return false;
				}
			}

			int currentAtom = toProcess.pop();
			for(OnTheFlyNeighborList::iterator neighborIter(neighborList, currentAtom); !neighborIter.atEnd(); neighborIter.next()) {
				int neighborIndex = neighborIter.current();
				if(!flagArray.at(neighborIndex)) {
					flagArray.setBit(neighborIndex);
					_atomClustersChannel->setInt(neighborIndex, cluster);
					toProcess.push(neighborIndex);
				}
			}
		}
		while(toProcess.empty() == false);
	}

	// Make sure all atoms have been visited.
	OVITO_ASSERT(numAtomsProcessed == atomsObject->atomsCount());

	return true;
}

/******************************************************************************
* Saves the class' contents to the given stream.
******************************************************************************/
void ClusterAtomsModifier::saveToStream(ObjectSaveStream& stream)
{
	AtomsObjectAnalyzerBase::saveToStream(stream);

	stream.beginChunk(0x6623AF);
	stream << numClusters;
	stream.endChunk();
}

/******************************************************************************
* Loads the class' contents from the given stream.
******************************************************************************/
void ClusterAtomsModifier::loadFromStream(ObjectLoadStream& stream)
{
	AtomsObjectAnalyzerBase::loadFromStream(stream);

	stream.expectChunk(0x6623AF);
	stream >> numClusters;
	stream.closeChunk();
}

/******************************************************************************
* Creates a copy of this object.
******************************************************************************/
RefTarget::SmartPtr ClusterAtomsModifier::clone(bool deepCopy, CloneHelper& cloneHelper)
{
	// Let the base class create an instance of this class.
	ClusterAtomsModifier::SmartPtr clone = static_object_cast<ClusterAtomsModifier>(AtomsObjectAnalyzerBase::clone(deepCopy, cloneHelper));
	clone->numClusters = this->numClusters;
	return clone;
}


IMPLEMENT_PLUGIN_CLASS(ClusterAtomsModifierEditor, AtomsObjectModifierEditorBase)

/******************************************************************************
* Sets up the UI widgets of the editor.
******************************************************************************/
void ClusterAtomsModifierEditor::createUI(const RolloutInsertionParameters& rolloutParams)
{
	// Create a rollout.
	QWidget* rollout = createRollout(tr("Atom clusters"), rolloutParams);

    // Create the rollout contents.
	QVBoxLayout* layout1 = new QVBoxLayout(rollout);
	layout1->setContentsMargins(4,4,4,4);
	layout1->setSpacing(0);

	BooleanPropertyUI* autoUpdateUI = new BooleanPropertyUI(this, PROPERTY_FIELD_DESCRIPTOR(AtomsObjectAnalyzerBase, _autoUpdateOnTimeChange));
	layout1->addWidget(autoUpdateUI->checkBox());

	// Only selected atoms parameter.
	BooleanPropertyUI* onlySelectedPUI = new BooleanPropertyUI(this, PROPERTY_FIELD_DESCRIPTOR(ClusterAtomsModifier, _onlySelectedAtoms));
	layout1->addWidget(onlySelectedPUI->checkBox());
	onlySelectedPUI->setWhatsThis(tr("If enabled the clustering algorithm operates only on selected atoms."));

	QPushButton* recalcButton = new QPushButton(tr("Calculate"), rollout);
	layout1->addSpacing(6);
	layout1->addWidget(recalcButton);
	connect(recalcButton, SIGNAL(clicked(bool)), this, SLOT(onRecalculate()));

	// Status label.
	layout1->addSpacing(10);
	layout1->addWidget(statusLabel());

	// Open a sub-editor for the NearestNeighborList sub-object.
	SubObjectParameterUI* subEditorUI = new SubObjectParameterUI(this, PROPERTY_FIELD_DESCRIPTOR(AtomsObjectAnalyzerBase, _nearestNeighborList), rolloutParams.before(rollout));
}

/******************************************************************************
* Is called when the user presses the Recalculate button.
******************************************************************************/
void ClusterAtomsModifierEditor::onRecalculate()
{
	if(!editObject()) return;
	ClusterAtomsModifier* modifier = static_object_cast<ClusterAtomsModifier>(editObject());
	try {
		modifier->performAnalysis(ANIM_MANAGER.time());
	}
	catch(Exception& ex) {
		ex.prependGeneralMessage(tr("Failed to perform clustering of atoms."));
		ex.showError();
	}
}

};	// End of namespace AtomViz
