///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <gsl/gsl_eigen.h>

#include "AtomicDeformationGradientModifier.h"

namespace CrystalAnalysis {

/// Computes the eigenvalues and optionally eigenvectors of a symmetric tensor.
void eigenvalues(const SymmetricTensor2& tensor, FloatType lambda[3], Matrix3* eigenvectors)
{
	// Use the GNU Scientific Library to compute eigensystem.

	double A[3][3];
	for(size_t i=0; i<3; i++)
		for(size_t j=0; j<3; j++)
			A[i][j] = tensor(i, j);
	gsl_matrix_view Aview = gsl_matrix_view_array((double*)A, 3, 3);
	
	double eval[3];
	gsl_vector_view evalview = gsl_vector_view_array(eval, 3);

	if(eigenvectors) {
		double evec[3][3];
		gsl_matrix_view evecview = gsl_matrix_view_array((double*)evec, 3, 3);

		gsl_eigen_symmv_workspace* ws = gsl_eigen_symmv_alloc(3);
		gsl_eigen_symmv(&Aview.matrix, &evalview.vector, &evecview.matrix, ws);
		gsl_eigen_symmv_free(ws);
		
		for(size_t i=0; i<3; i++) {
			eigenvectors->column(i).X = (FloatType)evec[i][0]; 
			eigenvectors->column(i).Y = (FloatType)evec[i][1]; 
			eigenvectors->column(i).Z = (FloatType)evec[i][2];
		}
	}
	else {
		gsl_eigen_symm_workspace* ws = gsl_eigen_symm_alloc(3);
		gsl_eigen_symm(&Aview.matrix, &evalview.vector, ws);
		gsl_eigen_symm_free(ws);		
	}
	
	lambda[0] = (FloatType)eval[0];
	lambda[1] = (FloatType)eval[1];
	lambda[2] = (FloatType)eval[2];	
}

/// Computes the eigenvalues of a symmetric tensor.
void eigenvalues(const SymmetricTensor4& tensor, FloatType lambdas[6])
{
	gsl_matrix* gslmat = gsl_matrix_alloc(6, 6);
	for(size_t i=0; i<6; i++)
		for(size_t j=0; j<6; j++)
			gsl_matrix_set(gslmat, i, j, tensor(i,j));
	gsl_eigen_symm_workspace* w = gsl_eigen_symm_alloc(6);
	gsl_vector* eigenvalues = gsl_vector_alloc(6);
	gsl_eigen_symm(gslmat, eigenvalues, w);
	for(size_t i=0; i<6; i++)
		lambdas[i] = gsl_vector_get(eigenvalues, i);
	gsl_eigen_symm_free(w);
	gsl_vector_free(eigenvalues);
	gsl_matrix_free(gslmat);
}

/// Finds the minimum eigenvalue of the symmetric tensor.
FloatType minEigenvalue(const SymmetricTensor4& tensor)
{
	using namespace std;

	FloatType lambda[6];
	eigenvalues(tensor, lambda);

	// Find minimum eigenvalue.
	FloatType dmin = FLOATTYPE_MAX;
	for(size_t i=0; i<6; i++)
		dmin = min(dmin, lambda[i]);
	return dmin;
}

};	// End of namespace CrystalAnalysis
