#!/usr/bin/python3
# SPDX-License-Identifier: BSD-3-Clause
#
# Copyright 2022 Raritan Inc. All rights reserved.

import copy, sys, os
sys.path.append(os.path.dirname(os.path.abspath(__file__)) + "/pdu-python-api")
from raritan.rpc import Agent, sensors
from raritan.rpc.peripheral import DeviceManager, GatewaySensorManager, ModbusCfg

ip = "10.0.42.2"
user = "admin"
pw = "raritan"

try:
    ip = sys.argv[1]
    user = sys.argv[2]
    pw = sys.argv[3]
except IndexError:
    pass # use defaults

agent = Agent("https", ip, user, pw, disable_certificate_verification=True)
mgr = DeviceManager("/model/peripheraldevicemanager", agent)
gwcfg = mgr.getGatewaySensorManager()

noThresholds = sensors.NumericSensor.Thresholds(
    upperCriticalActive = False,
    upperCritical = 0,
    upperWarningActive = False,
    upperWarning = 0,
    lowerWarningActive = False,
    lowerWarning = 0,
    lowerCriticalActive = False,
    lowerCritical = 0,
    assertionTimeout = 0,
    deassertionHysteresis = 0
)

ClassVoltage = GatewaySensorManager.NumericSensorClass(
    classId = "Voltage",
    metadata = sensors.NumericSensor.MetaData(
        type = sensors.Sensor.TypeSpec(
            sensors.Sensor.NUMERIC,
            sensors.Sensor.VOLTAGE,
            sensors.Sensor.VOLT
        ),
        decdigits = 1,
        accuracy = 0.0,
        resolution = 1.0,
        tolerance = 0,
        noiseThreshold = 1,
        range = sensors.NumericSensor.Range(0, 32767),
        thresholdCaps = sensors.NumericSensor.ThresholdCapabilities(True, True, True, True),
    ),
    defaultThresholds = noThresholds,
    preferCommonThresholds = False
)


ClassFrequency = GatewaySensorManager.NumericSensorClass(
    classId = "Frequency",
    metadata = sensors.NumericSensor.MetaData(
        type = sensors.Sensor.TypeSpec(
            sensors.Sensor.NUMERIC,
            sensors.Sensor.FREQUENCY,
            sensors.Sensor.HZ
        ),
        decdigits = 2,
        accuracy = 0.0,
        resolution = 0.01,
        tolerance = 0,
        noiseThreshold = 0.01,
        range = sensors.NumericSensor.Range(45, 65),
        thresholdCaps = sensors.NumericSensor.ThresholdCapabilities(True, True, True, True),
    ),
    defaultThresholds = noThresholds,
    preferCommonThresholds = False
)

ClassPower = GatewaySensorManager.NumericSensorClass(
    classId = "Power",
    metadata = sensors.NumericSensor.MetaData(
        type = sensors.Sensor.TypeSpec(
            sensors.Sensor.NUMERIC,
            sensors.Sensor.POWER,
            sensors.Sensor.WATT
        ),
        decdigits = 2,
        accuracy = 0.0,
        resolution = 0.01,
        tolerance = 0,
        noiseThreshold = 0.01,
        range = sensors.NumericSensor.Range(0, 1e10),
        thresholdCaps = sensors.NumericSensor.ThresholdCapabilities(True, True, True, True),
    ),
    defaultThresholds = noThresholds,
    preferCommonThresholds = False
)

ClassEnergy = GatewaySensorManager.NumericSensorClass(
    classId = "Energy",
    metadata = sensors.NumericSensor.MetaData(
        type = sensors.Sensor.TypeSpec(
            sensors.Sensor.NUMERIC,
            sensors.Sensor.ENERGY,
            sensors.Sensor.WATT_HOUR
        ),
        decdigits = 0,
        accuracy = 0.0,
        resolution = 1,
        tolerance = 0,
        noiseThreshold = 1,
        range = sensors.NumericSensor.Range(0, 1e10),
        thresholdCaps = sensors.NumericSensor.ThresholdCapabilities(True, True, True, True),
    ),
    defaultThresholds = noThresholds,
    preferCommonThresholds = False
)

RemoteDevice = GatewaySensorManager.RemoteModbusRTUDevice(
    deviceId = "PM710",
    disabled = False,
    name = "Schneider PM-710",
    timeoutMs = 100,
    retry = 0, # use default
    detectionIdentifiers = {
        GatewaySensorManager.MODBUS_VENDOR_NAME: "Schneider Electric",
        GatewaySensorManager.MODBUS_PRODUCT_CODE: "PM710",
        GatewaySensorManager.MODBUS_REVISION: "V03.150"
    },
    busInterface = "sensorhub1-rs485",
    busSettings = ModbusCfg.SerialSettings(
        baud = 19200,
        parity = ModbusCfg.SerialSettings.Parity.EVEN,
        dataBits = 8,
        stopBits = 1
    ),
    unitId = 1,
    interframeDelayDeciChars = 0, # use default
)

Long = GatewaySensorManager.ModbusValueEncoding32(
    encodingId = "Long",
    type = GatewaySensorManager.EncodingType.INT,
    invertState = False,
    interpretationRules = [],
    scalingFactor = 1.0,
    offset = 0.0,
    byteSwap = False,
    mask = 0,
    endianness = GatewaySensorManager.ModbusEndianness.MODBUS_LITTLE_ENDIAN
)

Float = copy.deepcopy(Long)
Float.encodingId = "Float"
Float.type = GatewaySensorManager.EncodingType.IEEE754

Integer = GatewaySensorManager.ModbusValueEncoding16(
    encodingId = "Integer",
    type = GatewaySensorManager.EncodingType.INT,
    invertState = False,
    interpretationRules = [],
    scalingFactor = 1.0,
    offset = 0.0,
    byteSwap = False,
    mask = 0
)

Integer100 = copy.deepcopy(Integer)
Integer100.encodingId = "Integer100"
Integer100.scalingFactor = 0.01

S4010 = GatewaySensorManager.ModbusSensor(
    sensorId = "",
    disabled = False,
    deviceId = "PM710",
    classId = "Voltage",
    encodingId = "Integer",
    defaultName = "Voltage, L-L, 3P Average",
    function = ModbusCfg.ModbusFunction.HOLDING_REGISTER,
    regAddr = 4010
)

S4013 = GatewaySensorManager.ModbusSensor(
    sensorId = "",
    disabled = False,
    deviceId = "PM710",
    classId = "Frequency",
    encodingId = "Integer100",
    defaultName = "Frequency",
    function = ModbusCfg.ModbusFunction.HOLDING_REGISTER,
    regAddr = 4013
)

S1022 = GatewaySensorManager.ModbusSensor(
    sensorId = "",
    disabled = False,
    deviceId = "PM710",
    classId = "Power",
    encodingId = "Float",
    defaultName = "Real Power, Total Demand Present",
    function = ModbusCfg.ModbusFunction.HOLDING_REGISTER,
    regAddr = 1022
)

S4000 = GatewaySensorManager.ModbusSensor(
    sensorId = "",
    disabled = False,
    deviceId = "PM710",
    classId = "Energy",
    encodingId = "Long",
    defaultName = "Real Energy, Total",
    function = ModbusCfg.ModbusFunction.HOLDING_REGISTER,
    regAddr = 4000
)

cfg = {
    "PM710" : GatewaySensorManager.ConfigurationPackage(False, "Schneider PM-710",
        [ ClassVoltage, ClassFrequency, ClassPower, ClassEnergy ],
        [ RemoteDevice ],
        [ Float, Long, Integer, Integer100 ],
        [ S4010, S4013, S1022, S4000 ]
    )
}

ret = gwcfg.setConfiguration(cfg)

match ret:
    case GatewaySensorManager.ERR_CONFIG_INCONSISTENT:
        print("FAILED: Consistency check has failed (e.g. non-unique identifiers)")

    case GatewaySensorManager.ERR_CONFIG_STORAGE_FAILED:
        print("FAILED: Configuration cannot be stored.")

    case _:
        print("Configuration stored")
