# Copyright 2023 Teemu Ikonen
# SPDX-License-Identifier: GPL-3.0-only

import re

import gi
from pynmea2.nmea import NMEASentence

from satellite.nmeasource import (  # noqa: E402
    ModemError,
    ModemLockedError,
    ModemNoNMEAError,
    NmeaSource,
    NmeaSourceNotFoundError,
)

gi.require_version('ModemManager', '1.0')
from gi.repository import Gio, ModemManager  # noqa: E402, I100


class ModemManagerGLibNmeaSource(NmeaSource):

    def __init__(self, update_callback, quirks=[], **kwargs):
        super().__init__(update_callback, **kwargs)
        self.bus = None
        self.manager = None
        self.modem = None
        self.mlocation = None
        self.old_refresh_rate = None
        self.old_sources_enabled = None
        self.old_signals_location = None
        self.location_updated = None
        self.quirks = set(quirks)

    def initialize(self):
        # If reinitializing, disconnect old update cb
        if self.mlocation is not None:
            self.mlocation.disconnect_by_func(self.update_callback)
        self.bus = Gio.bus_get_sync(Gio.BusType.SYSTEM, None)
        self.manager = ModemManager.Manager.new_sync(
            self.bus, Gio.DBusObjectManagerClientFlags.DO_NOT_AUTO_START, None)
        if self.manager.get_name_owner() is None:
            raise NmeaSourceNotFoundError("ModemManager is not running")
        objs = self.manager.get_objects()
        if objs:
            self.modem = objs[0].get_modem()
            self.mlocation = objs[0].get_modem_location()
        else:
            raise NmeaSourceNotFoundError("No Modems Found")
        self.manufacturer = self.modem.get_manufacturer()
        self.model = self.modem.get_model()
        self.revision = self.modem.get_revision()

        if 'detect' in self.quirks:
            self.quirks.remove('detect')
            if (self.model.startswith('QUECTEL')
                    and self.manufacturer == 'QUALCOMM INCORPORATED'):
                self.quirks.add('QuectelTalker')
            # Detect SDM845 GNSS unit and disable MSB assistance,
            # which causes stalling at startup due to some bug somewhere
            if (self.manufacturer == 'QUALCOMM INCORPORATED'
                    and self.model == '0'
                    and self.revision.find('SDM845') >= 0):
                self.quirks.add('NoMSB')

        try:
            state = self.modem.get_state()
            if int(state) > 0:
                if self.old_refresh_rate is None:
                    self.old_refresh_rate = self.mlocation.props.gps_refresh_rate
                if self.old_sources_enabled is None:
                    self.old_sources_enabled = self.mlocation.props.enabled
                if self.old_signals_location is None:
                    self.old_signals_location = self.mlocation.props.signals_location
                caps = self.mlocation.get_capabilities()
                if not caps & ModemManager.ModemLocationSource.GPS_NMEA:
                    raise NmeaSourceNotFoundError(
                        "Modem does not support NMEA")
                enable = ModemManager.ModemLocationSource.GPS_NMEA
                if (caps & ModemManager.ModemLocationSource.AGPS_MSB
                        and 'NoMSB' not in self.quirks):
                    enable |= ModemManager.ModemLocationSource.AGPS_MSB
                self.mlocation.setup_sync(enable, True, None)
            else:
                raise ModemError("Modem state is: %d" % state)
        except AttributeError as e:
            if state == ModemManager.ModemState.LOCKED:
                raise ModemLockedError from e
            else:
                raise e
        except gi.repository.GLib.GError as e:
            # Ignore error on AGPS enablement by this hack
            if 'agps-msb' not in str(e):
                raise e

        self.mlocation.set_gps_refresh_rate_sync(self.refresh_rate, None)
        self.mlocation.connect('notify::location', self.update_callback)

        self.initialized = True

    def _really_get(self):
        if not self.initialized:
            self.initialize()
        try:
            loc = self.mlocation.get_signaled_gps_nmea()
        except Exception as e:
            self.initialized = False
            raise e

        if loc is None:
            raise ModemNoNMEAError

        nmeas = loc.get_traces()
        if nmeas is None:
            self.initialized = False
            raise ModemNoNMEAError

        if 'QuectelTalker' in self.quirks:
            nmeas = self.quectel_talker_quirk(nmeas)

        return '\r\n'.join(nmeas)

    def close(self):
        if self.mlocation is None:
            return
        try:
            self.mlocation.disconnect_by_func(self.update_callback)
        except TypeError:
            pass  # Ignore error when nothing is connected
        if self.old_sources_enabled is not None:
            self.mlocation.setup_sync(
                ModemManager.ModemLocationSource(self.old_sources_enabled),
                self.old_signals_location, None)
        if self.old_refresh_rate is not None:
            self.mlocation.set_gps_refresh_rate_sync(self.old_refresh_rate, None)

    def quectel_talker_quirk(self, nmeas):
        pq_re = re.compile(r"""
            ^\s*\$?
            (?P<talker>PQ)
            (?P<sentence>\w{3})
            (?P<data>[^*]*)
            (?:[*](?P<checksum>[A-F0-9]{2}))$""", re.VERBOSE)
        out = []
        for nmea in (n for n in nmeas if n):
            mo = pq_re.match(nmea)
            if mo:
                # The last extra data field is Signal ID, these are
                # 1 = GPS, 2 = Glonass, 3 = Galileo, 4 = BeiDou, 5 = QZSS
                # Determine talker from Signal ID
                talker = 'QZ' if mo.group('data').endswith('5') else 'BD'
                # Fake talker and checksum
                fake = talker + "".join(mo.group(2, 3))
                out.append('$' + fake + "*%02X" % NMEASentence.checksum(fake))
            else:
                out.append(nmea)

        return out
