/*
 * Copyright (C) 2013 Canonical, Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Author: Pawel Stolowski <pawel.stolowski@canonical.com>
 */

#include "SmartScopesParserTest.h"
#include <QTest>
#include <QList>
#include <QMap>
#include <iostream>

#include "SmartScopesClient.h"
#include "SmartScopesParser.h"
#include "SmartScopesPreviewParser.h"

void SmartScopesParserTest::parseSearchResultLineTest()
{
    const QString input[] = {
        "{",  // invalid json
        "{\"type:1\"}", // invalid type
        "{\"scopes\": [[\"searchin-scopes.scope\", \"server\"], [\"info-openweathermap.scopes\", \"server\"], [\"reference-exact.scope\", \"server\"], [\"reference-rest.scope\", \"server\"], [\"news-googlenews.scope\", \"server\"], [\"info-reddit.scope\", \"server\"], [\"news-yahoostock.scope\", \"server\"], [\"music-grooveshark.scope\", \"server\"], [\"music-soundcloud.scope\", \"server\"], [\"info-foursquare.scope\", \"server\"], [\"music-songkick.scope\", \"server\"], [\"more_suggestions-u1ms.scope\", \"server\"], [\"more_suggestions-amazon.scope\", \"server\"], [\"more_suggestions-etsy.scope\", \"server\"], [\"more_suggestions-skimlinks.scope\", \"server\"], [\"more_suggestions-ebay.scope\", \"server\"]], \"server_sid\": \"abcdef\", \"type\": \"recommendations\"}",
        "{\"info\": {\"info-foursquare.scope\": [{\"category\": 0, \"comment\": \"55 Basinghall St London\", \"title\": \"Imprima (LON)\", \"uri\": \"https://foursquare.com/v/imprima-lon/4b60\", \"icon_hint\": \"/usr/share/icons/unity-icon-theme/places/svg/result-info.svg\", \"mimetype\": \"text/html\", \"dnd_uri\": \"https://foursquare.com/v/imprima-lon/4b606badf964a520e6e429e3\", \"result_type\": 0, \"metadata\": {\"id\": \"foursquare:1fc9f552b1f9161ab81fbc0616234f9ff1bdeeee\"}}, {\"category\": 0, \"comment\": \"123 Buckingham Palace Rd London\", \"title\": \"LON-BEL-123 Gym\", \"uri\": \"https://foursquare.com/v/lonbel123-gym/4d9ac297674ca143ceb6c243\", \"icon_hint\": \"https://foursquare.com/img/categories_v2/building/gym_88.png\", \"mimetype\": \"text/html\", \"dnd_uri\": \"https://foursquare.com/v/lonbel123-gym/4d9ac297674ca143ceb6c243\", \"result_type\": 0, \"metadata\": {\"id\": \"foursquare:d86115198ba3a379740206d3c6da18220956a85c\"}}, {\"category\": 0, \"comment\": \"The Qube\", \"title\": \"HOK LON First Friday\", \"uri\": \"https://foursquare.com/v/hok-lon-first-friday/4f510633e4b044219085e211\", \"icon_hint\": \"https://foursquare.com/img/categories_v2/nightlife/default_88.png\", \"mimetype\": \"text/html\", \"dnd_uri\": \"https://foursquare.com/v/hok-lon-first-friday/4f510633e4b044219085e211\", \"result_type\": 0, \"metadata\": {\"id\": \"foursquare:e43c6843b4d3f0ab395a8b92e9f25b3be72e699f\"}}]}, \"type\": \"results\"}"
    };

    SmartScopesParser parser;
    QList<QPair<QString, std::shared_ptr<SmartScopesResult>>> results;
    QString server_sid = "";

    for (unsigned i=0; i<sizeof(input)/sizeof(QString); i++)
    {
        parser.parseLine(input[i],
                         [&results](const QString& scopeId, std::shared_ptr<SmartScopesResult> res) {
                             results.append(QPair<QString, std::shared_ptr<SmartScopesResult>>(scopeId, res));
                         },
                         [&server_sid](const QString &sid) {
                             server_sid = sid;
                         }
            );
    }

    QCOMPARE(server_sid, QString("abcdef"));
    QCOMPARE(results.size(), 3);
    QCOMPARE(results[0].first, QString("info-foursquare.scope"));
    QCOMPARE(results[1].first, QString("info-foursquare.scope"));
    QCOMPARE(results[2].first, QString("info-foursquare.scope"));

    auto res = results[0].second;

    QCOMPARE(res->uri, QString("https://foursquare.com/v/imprima-lon/4b60"));
    QCOMPARE(res->title, QString("Imprima (LON)"));
    QCOMPARE(res->comment, QString("55 Basinghall St London"));
    QCOMPARE(res->icon_hint, QString("/usr/share/icons/unity-icon-theme/places/svg/result-info.svg"));
    QCOMPARE(res->dnd_uri, QString("https://foursquare.com/v/imprima-lon/4b606badf964a520e6e429e3"));

    res = results[1].second;

    QCOMPARE(res->uri, QString("https://foursquare.com/v/lonbel123-gym/4d9ac297674ca143ceb6c243"));
    QCOMPARE(res->title, QString("LON-BEL-123 Gym"));
    QCOMPARE(res->comment, QString("123 Buckingham Palace Rd London"));
    QCOMPARE(res->icon_hint, QString("https://foursquare.com/img/categories_v2/building/gym_88.png"));
    QCOMPARE(res->dnd_uri, QString("https://foursquare.com/v/lonbel123-gym/4d9ac297674ca143ceb6c243"));
}

void SmartScopesParserTest::parseGenericPreviewTest()
{
    const QString input = "{\"info\": [{\"display_name\": \"Published\", \"icon_hint\": \"\", \"id\": \"Published\", \"value\": \"2013-05-07\"}], \"renderer_name\": \"preview-generic\", \"description\": \"The second exciting adventure in The Door Within trilogy.\", \"title\": \"The Rise of the Wyrm Lord\", \"actions\": [{\"display_name\": \"View\", \"icon_hint\": \"/usr/share/icons/unity-icon-theme/places/svg/service-googlebooks.svg\", \"id\": \"view\", \"activation_uri\": \"http://foo.com/abc\"}], \"subtitle\": \"Wayne Thomas Batson\", \"image_hint\": \"http://bks3.foo.com/books?id=R6B\"}";

    SmartScopesPreviewParser parser;
    auto preview = parser.parse(input);
    QVERIFY(preview != nullptr);
    QVERIFY(UNITY_IS_GENERIC_PREVIEW(preview) != 0);

    QCOMPARE(unity_preview_get_title(preview), "The Rise of the Wyrm Lord");
    QCOMPARE(unity_preview_get_subtitle(preview), "Wayne Thomas Batson");

    unity_object_unref(preview);
}

void SmartScopesParserTest::parseMusicPreviewTest()
{
    const QString input = "{\"info\": [], \"subtitle\": \"Los Mustang\", \"description\": \"\", \"title\": \"Los Mustang - Los Jovenes\", \"_format\": \"simple-serialization-v0\", \"attribution\": \"Certain content that appears in this preview comes from Ubuntu One Music Store. This content is provided 'as is' and is subject to change or removal at any time.\", \"actions\": [{\"icon_hint\": \"file:///usr/share/icons/unity-icon-theme/places/svg/service-u1.svg\", \"activation_uri\": \"https://foobar.com/x\", \"display_name\": \"Download\", \"extra_text\": \"9,99 USD\"}], \"tracks\": [{\"length\": 196, \"uri\": \"http://api.foobar.com/1\", \"track_no\": 1, \"title\": \"Los jovenes The young ones\"}, {\"length\": 149, \"uri\": \"http://api.foobar.com/2\", \"track_no\": 2, \"title\": \"Diana\"}, {\"length\": 162, \"uri\": \"http://api.7digital.com/3\", \"track_no\": 3, \"title\": \"Living doll\"}], \"renderer_name\": \"preview-music\", \"image_hint\": \"http://cdn.foobar.com/350.jpg\"}";
    
    SmartScopesPreviewParser parser;
    auto preview = parser.parse(input);
    QVERIFY(preview != nullptr);
    QVERIFY(UNITY_IS_MUSIC_PREVIEW(preview) != 0);

    QCOMPARE(unity_preview_get_title(preview), "Los Mustang - Los Jovenes");
    QCOMPARE(unity_preview_get_subtitle(preview), "Los Mustang");

    unity_object_unref(preview);
}

QTEST_MAIN(SmartScopesParserTest)
